package csbase.logic.algorithms.parameters.validators;

import java.io.IOException;
import java.rmi.RemoteException;
import java.util.Arrays;
import java.util.EnumSet;

import tecgraf.javautils.core.io.FileUtils;
import csbase.exception.ServiceFailureException;
import csbase.logic.AccessSGAPathPermission;
import csbase.logic.ClientFile;
import csbase.logic.ClientProjectFile;
import csbase.logic.ClientSGAFile;
import csbase.logic.User;
import csbase.logic.algorithms.parameters.FileParameterMode;
import csbase.logic.algorithms.parameters.FileParameterPipeAcceptance;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.URLParameter;
import csbase.logic.algorithms.parameters.URLProtocol;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.logic.algorithms.validation.LocalizedMessage;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.ValidationError;
import csbase.logic.algorithms.validation.ValidationMode;
import csbase.logic.algorithms.validation.ValidationSuccess;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;
import csbase.remote.SGAServiceInterface;

/**
 * Validador de parmetros do tipo URL.
 *
 * @author Tecgraf
 */
public class URLParameterValidator extends SimpleParameterValidator<FileURLValue> {

  /**
   * Cria um validador de URL.
   *
   * @param optional Indica se o valor  opcional {@code true} ou obrigatrio
   *        {@code false}.
   */
  public URLParameterValidator(boolean optional) {
    super(optional);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final Validation validateValue(SimpleParameter<?> parameter,
    FileURLValue value, ValidationContext context) throws RemoteException {
    if (!parameter.isVisible()) {
      return new ValidationSuccess();
    }
    if (!parameter.isEnabled()) {
      return new ValidationSuccess();
    }
    URLParameter urlParameter = (URLParameter) parameter;
    if (urlParameter.hasLink()) {
      return new ValidationSuccess();
    }
    else {
      if (urlParameter.usesPipe() == FileParameterPipeAcceptance.ALWAYS
        && context.getMode() == ValidationMode.FULL) {
        LocalizedMessage message =
          new LocalizedMessage(URLParameterValidator.class, "must_use_pipe",
            new Object[] { parameter.getLabel() });
        return new ValidationError(message);
      }
    }

    Validation result = super.validateValue(parameter, value, context);
    if (!result.isWellSucceded()) {
      return result;
    }
    boolean mustExist = urlParameter.mustExist();
    FileParameterMode mode = urlParameter.getMode();
    return validateURL(value, urlParameter.getAllowedProtocols(), mustExist,
      mode, context);
  }

  /**
   * Valida uma url selecionada pelo usurio.
   *
   * @param url A url
   * @param allowedProtocols lista de protocolos vlidos.
   * @param mustExist Indica se a url deve existir.
   * @param mode O modo do parmetro (No aceita {@code null}).
   * @param context o contexto de validao.
   * @return o resultado da validao.
   * @throws RemoteException em caso de erro na comunicao com servidor.
   */
  public Validation validateURL(FileURLValue url,
    EnumSet<URLProtocol> allowedProtocols, boolean mustExist,
    FileParameterMode mode, ValidationContext context) throws RemoteException {
    if (url != null) {
      URLProtocol protocol = url.getProtocol();
      if (!allowedProtocols.contains(protocol)) {
        LocalizedMessage message =
          new LocalizedMessage(URLParameterValidator.class, "invalid_protocol",
            new Object[] { protocol });
        return new ValidationError(message);
      }
      switch (protocol) {
        case PROJECT:
          return validateProjectFile(url, context, mustExist, mode);
        case SGA:
          return validateSGAFile(url, context, mustExist, mode);
        case LOCAL:
          return validateLocalFile(url, context, mustExist, mode);
        default:
          return validateAbsoluteFileName(url);
      }
    }
    return new ValidationSuccess();
  }

  /**
   * Verifica se o arquivo tem um nome e um caminho absoluto vlidos.
   *
   * @param url a url.
   * @return o resultado da validao.
   */
  public static Validation validateAbsoluteFileName(FileURLValue url) {
    String path = url.getPath();
    if (!FileURLValue.isAbsolutePath(path)) {
      LocalizedMessage message =
        new LocalizedMessage(URLParameterValidator.class, "not_absolute_path",
          new Object[] { path });
      return new ValidationError(message);
    }
    String invalidChars = "[^\\w:.\\-_/]";
    String normalizedString = path.replaceAll(invalidChars, "_");
    if (!normalizedString.equals(path)) {
      LocalizedMessage message =
        new LocalizedMessage(URLParameterValidator.class,
          "invalid_absolute_file_name", new Object[] { path });
      return new ValidationError(message);
    }
    return new ValidationSuccess();
  }

  /**
   * Verifica se o arquivo tem um nome e um caminho vlidos.
   *
   * @param path o caminho do arquivo.
   * @return o resultado da validao.
   */
  public static Validation validateProjectFileName(String[] path) {
    for (String component : path) {
      String fixedFileName = FileUtils.fixFileName(component);
      if (!fixedFileName.equals(component)) {
        LocalizedMessage message =
          new LocalizedMessage(URLParameterValidator.class,
            "invalid_file_name", new Object[] { component });
        return new ValidationError(message);
      }
    }
    return new ValidationSuccess();
  }

  /**
   * Valida um arquivo de projeto selecionado pelo usurio.
   *
   * @param file arquivo (No aceita {@code null}).
   * @param context contexto que contm o identificador do projeto no qual o
   *        algoritmo vai ser executado. Esse identificador  necessrio para
   *        validar, por exemplo, se existe o arquivo escolhido.
   * @param mustExist indica se o arquivo deve existir.
   * @param mode modo do parmetro (No aceita {@code null}).
   *
   * @return resultado da validao.
   * @throws RemoteException em caso de erro na comunicao com servidor.
   */
  private Validation validateProjectFile(FileURLValue file,
    ValidationContext context, boolean mustExist, FileParameterMode mode)
      throws RemoteException {

    Object projectId = context.getProjectId();
    if (projectId == null) {
      LocalizedMessage message =
        new LocalizedMessage(URLParameterValidator.class, "no_project",
          new Object[] { file.getPath() });
      return new ValidationError(message);
    }
    String[] path = file.getPathAsArray();
    Validation fileNameResult = validateProjectFileName(path);
    if (!fileNameResult.isWellSucceded()) {
      return fileNameResult;
    }
    ClientProjectFile projectFile = getProjectFile(path, projectId);
    if (projectFile != null) {
      return validateFileMode(mode, projectFile);
    }
    else {
      if (mustExist) {
        LocalizedMessage message =
          new LocalizedMessage(URLParameterValidator.class, "file_not_found",
            new Object[] { file.getPath() });
        return new ValidationError(message);
      }
      else {
        return validateParent(file, projectId);
      }
    }
  }

  /**
   * Valida uma url local selecionada pelo usurio.
   *
   * @param url arquivo (No aceita {@code null}).
   * @param context contexto que contm o identificador do projeto no qual o
   *        algoritmo vai ser executado. Esse identificador  necessrio para
   *        validar, por exemplo, se existe o arquivo escolhido.
   * @param mustExist indica se o arquivo deve existir.
   * @param mode modo do parmetro (No aceita {@code null}).
   *
   * @return resultado da validao.
   * @throws RemoteException em caso de erro na comunicao com servidor.
   */
  private Validation validateLocalFile(FileURLValue url, ValidationContext context,
    boolean mustExist, FileParameterMode mode) throws RemoteException {

    Validation fileNameResult = validateAbsoluteFileName(url);
    if (!fileNameResult.isWellSucceded()) {
      return fileNameResult;
    }
    return new ValidationSuccess();
  }

  /**
   * Valida um arquivo de SGA selecionado pelo usurio.
   *
   * @param value valor da URL.
   * @param context contexto da validao.
   * @param mustExist indica se o arquivo deve existir.
   * @param mode modo do parmetro (No aceita {@code null}).
   * @return resultado da validao.
   *
   * @throws RemoteException em caso de erro na comunicao com o servidor.
   */
  private Validation validateSGAFile(FileURLValue value, ValidationContext context,
    boolean mustExist, FileParameterMode mode) throws RemoteException {

    Validation fileNameResult = validateAbsoluteFileName(value);
    if (!fileNameResult.isWellSucceded()) {
      return fileNameResult;
    }

    Object userId = context.getUserId();
    User user = User.getUser(userId);
    if (!AccessSGAPathPermission.canReadPath(user, value.getHost(), value
      .getPath())) {
      LocalizedMessage message =
        new LocalizedMessage(URLParameterValidator.class, "sga.cannot.read",
          new Object[] { value.getHost(), value.getPath() });
      return new ValidationError(message);
    }

    String sgaName = value.getHost();
    if (sgaName == null) {
      LocalizedMessage message =
        new LocalizedMessage(URLParameterValidator.class, "sga.not.defined");
      return new ValidationError(message);
    }
    ClientSGAFile sgaFile = getSGAFile(sgaName, value.getPath());
    try {
      if (mustExist && (sgaFile == null || !sgaFile.exists())) {
        if (mode == FileParameterMode.DIRECTORY) {
          LocalizedMessage message =
            new LocalizedMessage(URLParameterValidator.class,
              "sga.dir.not.found", new Object[] { value.getHost(),
              value.getPath() });
          return new ValidationError(message);
        }

        LocalizedMessage message =
          new LocalizedMessage(URLParameterValidator.class,
            "sga.file.not.found", new Object[] { value.getHost(),
            value.getPath() });
        return new ValidationError(message);
      }
    }
    catch (IOException e) {
      if (mode == FileParameterMode.DIRECTORY) {
        LocalizedMessage message =
          new LocalizedMessage(URLParameterValidator.class,
            "sga.dir.not.found", new Object[] { value.getHost(),
            value.getPath() });
        return new ValidationError(message);
      }

      LocalizedMessage message =
        new LocalizedMessage(URLParameterValidator.class, "sga.file.not.found",
          new Object[] { value.getHost(), value.getPath() });
      return new ValidationError(message);
    }

    if (sgaFile != null) {
      return validateFileMode(mode, sgaFile);
    }
    return new ValidationSuccess();
  }

  /**
   * Valida o arquivo escolhido em relao ao modo do parmetro. Se parmetro
   * aceitar somente diretrios, por exemplo, a validao falha ao encontrar um
   * arquivo comum.
   *
   * @param mode modo do parmetro.
   * @param file arquivo selecionado.
   *
   * @return resultado da validao.
   */
  private Validation validateFileMode(FileParameterMode mode, ClientFile file) {
    if (mode == FileParameterMode.DIRECTORY && !file.isDirectory()) {
      LocalizedMessage message =
        new LocalizedMessage(URLParameterValidator.class, "not_directory",
          new Object[] { file.getStringPath() });
      return new ValidationError(message);
    }
    else if (mode == FileParameterMode.REGULAR_FILE && file.isDirectory()) {
      LocalizedMessage message =
        new LocalizedMessage(URLParameterValidator.class, "not_file",
          new Object[] { file.getStringPath() });
      return new ValidationError(message);
    }
    else {
      return new ValidationSuccess();
    }
  }

  /**
   * Valida o diretrio onde o arquivo se encontra. O diretrio-pai do arquivo
   * precisa existir, mesmo que o arquivo sendo validado no exista.
   *
   * @param file o valor do parmetro sendo validado.
   * @param projectId o identificador do projeto que contm o arquivo.
   *
   * @return o resultado da validao.
   * @throws RemoteException em caso de erro na comunicao com servidor.
   */
  private Validation validateParent(FileURLValue file, Object projectId)
    throws RemoteException {
    String[] parentPath = file.getPathAsArray();
    while (parentPath.length > 1) {
      parentPath = Arrays.copyOf(parentPath, parentPath.length - 1);
      String parentName = parentPath[parentPath.length - 1];
      if (!parentName.equals(".")) {
        ClientProjectFile parentProjectFile =
          getProjectFile(parentPath, projectId);
        if (parentProjectFile == null) {
          LocalizedMessage message =
            new LocalizedMessage(URLParameterValidator.class, "file_not_found",
              new Object[] { FileUtils.joinPath(parentPath) });
          return new ValidationError(message);
        }
        if (!parentProjectFile.isDirectory()) {
          LocalizedMessage message =
            new LocalizedMessage(URLParameterValidator.class, "not_directory",
              new Object[] { parentProjectFile.getStringPath() });
          return new ValidationError(message);
        }
        else {
          return new ValidationSuccess();
        }
      }
    }
    return new ValidationSuccess();
  }

  /**
   * Obtm um arquivo de projeto.
   *
   * @param path o caminho do arquivo.
   * @param projectId o identificador do projeto onde o arquivo se encontra.
   * @return o arquivo de projeto ou nulo, caso no exista ou no seja possvel
   *         obt-lo.
   * @throws RemoteException em caso de erro na comunicao com servidor.
   */
  private ClientProjectFile getProjectFile(String[] path, Object projectId)
    throws RemoteException {
    try {
      ProjectServiceInterface projectService =
        ClientRemoteLocator.projectService;
      if (projectService.existsFile(projectId, path)) {
        return projectService.getChild(projectId, path);
      }
      else {
        return null;
      }
    }
    catch (ServiceFailureException e) {
      return null;
    }
  }

  /**
   * Obtm o arquivo SGA.
   *
   * @param sgaName nome do SGA.
   * @param path path do arquivo.
   * @return arquivo SGA.
   *
   * @throws RemoteException em caso de erro na comunicao com o servidor.
   */
  private ClientSGAFile getSGAFile(String sgaName, String path)
    throws RemoteException {
    SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
    return sgaService.getFile(sgaName, path);
  }

}