package csbase.logic.algorithms.parsers;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.ClientSGAFileParameter;
import csbase.logic.algorithms.parameters.Parameter;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;

/**
 * Fbrica de parmetros do tipo {@link ClientSGAFileParameter}.
 * 
 * @author Tecgraf
 */
public class ClientSGAFileParameterFactory extends
  SimpleParameterParser<ClientSGAFileParameter> {

  /**
   * Atributo que define se o usurio pode escolher mais de um arquivo do SGA.
   * NOTA: este atributo s  usado se o valor do atributo
   * {@link ClientSGAFileParameterFactory#MODE_ATTRIBUTE} for "entrada".
   */
  private static final String MULTIPLE_SELECTION_ATTRIBUTE = "selecao_multipla";

  /**
   * Atributo que define se o usurio escolhe arquivos ou diretrios.
   */
  private static final String CATEGORY_ATTRIBUTE = "categoria";

  /** Define que o usurio pode escolher apenas arquivos. */
  private static final String CATEGORY_ATTRIBUTE_VALUE_REGULAR_FILE = "arquivo";

  /** Define que o usurio pode escolher apenas diretrios. */
  private static final String CATEGORY_ATTRIBUTE_VALUE_DIRECTORY = "diretorio";

  /**
   * Atributo que define se o uso do parmetro {@link ClientSGAFileParameter} 
   * leitura ou escrita.
   */
  private static final String MODE_ATTRIBUTE = "modo";

  /** Define que o usurio vai escolher arquivo(s) como entrada. */
  private static final String MODE_ATTRIBUTE_VALUE_INPUT = "entrada";

  /** Define que o usurio vai escolher um arquivo como saida. */
  private static final String MODE_ATTRIBUTE_VALUE_OUTPUT = "saida";

  /** Define uma lista de tipos a serem filtrados. */
  private static final String TYPE_ATTRIBUTE = "tipo";

  /**
   * {@inheritDoc}
   */
  @Override
  public ClientSGAFileParameter createSimpleParameter(XmlParser parser,
    String name, String label, String description, boolean isOptional,
    boolean isVisible, String commandLinePattern, ParameterGroup group,
    SimpleAlgorithmConfigurator configurator) throws ParseException {

    ClientSGAFileParameter parameter = new ClientSGAFileParameter(name, label,
      description, null, isOptional, isVisible, commandLinePattern);

    parameter.setMultipleSelection(isMultipleSelection(parser));
    parameter.setShowFiles(showFiles(parser));
    parameter.setInputMode(isInputMode(parser));
    parameter.setTypes(types(parser));

    if (parameter.isMultipleSelection() && !parameter.isInputMode()) {
      throw new ParseException("O parmetro " + name
        + " no pode ser configurado simultaneamente para seleo mltipla e modo save.");
    }

    return parameter;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Class<? extends Parameter>> getParameterTypes() {
    return Collections.singletonList(ClientSGAFileParameter.class);
  }

  /**
   * Retorna true se o usurio puder escolher multiplos arquivos; false caso
   * contrrio.
   * 
   * Nota 1: false  o valor padro. <br/>
   * Nota 2: s  possvel escolher mltiplos arquivos em modo de 'entrada'.
   * 
   * @param parser - parser XML.
   * @return true se o usurio puder escolher multiplos arquivos; false caso
   *         contrrio.
   */
  private boolean isMultipleSelection(XmlParser parser) {
    try {
      return parser.extractAttributeValueAsBoolean(MULTIPLE_SELECTION_ATTRIBUTE,
        false);
    }
    catch (ParseException e) {
      return false;
    }
  }

  /**
   * Retorna true se o usurio escolhe apenas arquivos; false define o caso em
   * que o usurio escolhe apenas diretrios.
   * 
   * Nota: true  o valor padro.
   * 
   * @param parser - parser XML.
   * @return true se o usurio escolhe apenas arquivos; false define o caso em
   *         que o usurio escolhe apenas diretrios.
   */
  private boolean showFiles(XmlParser parser) {
    String category = parser.extractAttributeValue(CATEGORY_ATTRIBUTE,
      CATEGORY_ATTRIBUTE_VALUE_REGULAR_FILE);

    if (!category.equalsIgnoreCase(CATEGORY_ATTRIBUTE_VALUE_REGULAR_FILE)
      && !category.equalsIgnoreCase(CATEGORY_ATTRIBUTE_VALUE_DIRECTORY)) {
      category = CATEGORY_ATTRIBUTE_VALUE_REGULAR_FILE;
    }

    return category.equalsIgnoreCase(CATEGORY_ATTRIBUTE_VALUE_REGULAR_FILE);
  }

  /**
   * Retorna true se o usurio escolhe arquivo(s) de entrada; false define que o
   * usurio escolhe um arquivo de saida.
   * 
   * Nota: true  o valor default.
   * 
   * @param parser - parser XML.
   * @return true se o usurio escolhe arquivo(s) de entrada; false define que o
   *         usurio escolhe um arquivo de saida.
   */
  private boolean isInputMode(XmlParser parser) {
    String mode = parser.extractAttributeValue(MODE_ATTRIBUTE,
      MODE_ATTRIBUTE_VALUE_INPUT);

    if (!mode.equalsIgnoreCase(MODE_ATTRIBUTE_VALUE_INPUT) && !mode
      .equalsIgnoreCase(MODE_ATTRIBUTE_VALUE_OUTPUT)) {
      mode = MODE_ATTRIBUTE_VALUE_INPUT;
    }

    return mode.equalsIgnoreCase(MODE_ATTRIBUTE_VALUE_INPUT);
  }

  /**
   * Retorna a lista com os tipos de arquivos usados no filtro do chooser.
   * 
   * @param parser - parser XML.
   * @return lista com os tipos de arquivos usados no filtro do chooser.
   */
  private List<String> types(XmlParser parser) {
    String types = parser.extractAttributeValue(TYPE_ATTRIBUTE, "");
    String[] splited = types.split("(?: *),(?: *)?");

    List<String> result = new ArrayList<String>();
    for (String s : splited) {
      if (!s.isEmpty()) {
        result.add(s);
      }
    }
    return result;
  }
}
