package csbase.logic;

import java.util.regex.PatternSyntaxException;

/**
 * Super-classe das classes de permisso que possuem, alm do nome e da
 * descrio, uma lista de atributos textuais.
 */
public abstract class AttributesPermission extends Permission {
  /**
   * Atirbutos
   */
  String[] attributes;

  /**
   * Construtor.
   */
  public AttributesPermission() {
    super();
  }

  /**
   * Construtor. Define o nome, a descrio e a lista de atributos textuais da
   * permisso sendo criada.
   * 
   * @param name .
   * @param description .
   * @param attributes .
   */
  public AttributesPermission(String name, String description,
    String[] attributes) {
    super(name, description);
    this.attributes = attributes;
  }

  /**
   * Retorna a lista de atributos da permisso.
   * 
   * @return um array de String's com os atributos.
   */
  public String[] getAttributes() {
    return attributes;
  }

  /**
   * Troca a lista de atributos da permisso.
   * 
   * @param attributes um array de String's com os novos atributos.
   */
  public void setAttributes(String[] attributes) {
    this.attributes = attributes;
  }

  /**
   * Verifica se a permisso possui um determinado atributo.
   * 
   * @param attribute o atributo a ser procurado.
   * 
   * @return <code>true</code> caso o atributo seja encontrado,
   *         <code>false</code> caso contrrio.
   */
  public boolean hasAttribute(String attribute) {
    for (int i = 0; i < attributes.length; i++) {
      if (attributes[i].equals(attribute)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Procura um atributo idntico ao fornecido ou que, analisado como uma
   * expresso regular, case com o texto fornecido. As expresses regulares
   * fornecidas nos atributos devem atender  sintaxe definida em
   * {@link java.util.regex.Pattern}.
   * 
   * @param text o texto a ser procurado ou casado com cada atributo-regexp
   * 
   * @return true caso encontreo um atributo igual ao texto, ou que case com
   *         ele, ou false caso nenhum seja encontrado.
   */
  public boolean hasMatchAttribute(String text) {
    if (getMatchAttribute(text) != null) {
      return true;
    }
    return false;
  }

  /**
   * Procura um atributo idntico ao fornecido ou que, analisado como uma
   * expresso regular, case com o texto fornecido. As expresses regulares
   * fornecidas nos atributos devem atender  sintaxe definida em
   * {@link java.util.regex.Pattern}.
   * 
   * @param text o texto a ser procurado ou casado com cada atributo-regexp
   * 
   * @return o primeiro atributo igual ao texto, ou que case com ele, ou
   *         <code>null</code> caso nenhum seja encontrado.
   */
  public String getMatchAttribute(String text) {
    for (int i = 0; i < attributes.length; i++) {
      try {
        if (text.equals(attributes[i]) || text.matches(attributes[i])) {
          return attributes[i];
        }
      }
      catch (PatternSyntaxException e) {
        // Ignorado para poder haver atributos normais.
      }
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equalContents(Object obj) {
    if (!super.equalContents(obj)) {
      return false;
    }
    if (!(obj instanceof AttributesPermission)) {
      return false;
    }
    AttributesPermission permission = (AttributesPermission) obj;
    String[] pAttributes = permission.getAttributes();
    if (pAttributes.length != attributes.length) {
      return false;
    }
    for (int i = 0; i < pAttributes.length; i++) {
      if (!pAttributes[i].equals(attributes[i])) {
        return false;
      }
    }
    return true;
  }
}
