package csbase.remote;

import java.rmi.ConnectException;
import java.rmi.NoSuchObjectException;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import csbase.logic.ObserverData;

/**
 * Notificador de observadores remotos. Possui mtodos para gerenciamento da
 * incluso/remoo de observadores com locks de leitura e escrita onde
 * necessrio, e mtodo para notificao dos observadores que remove de forma
 * segura os observadores que no puderam ser notificados por falha de RMI.
 * <p>
 * <b>Para maior escalabilidade os mtodos de notificao tipicamente devem ser
 * executados em threads separadas do fluxo principal.</b>
 * 
 * @author Tecgraf
 */
public class RemoteObserversNotificationManager {
  /**
   * Conjunto de observadores.
   */
  private final Set<ObserverData> observersSet = new HashSet<ObserverData>();
  /**
   * Objeto externo responsvel pela notificao.
   */
  private final RemoteObserverNotifierInterface notifier;
  /**
   * Gerador dos locks.
   */
  private final ReentrantReadWriteLock rwl = new ReentrantReadWriteLock();
  /**
   * Lock de leitura.
   */
  private final Lock readLock = rwl.readLock();
  /**
   * Lock de escrita.
   */
  private final Lock writeLock = rwl.writeLock();
  /**
   * Formatador de datas para mensagens de log.
   */
  private final DateFormat format;

  /**
   * Construtor de um gerenciador de observadores remotos.
   * 
   * @param notifier Objeto externo responsvel pela notificao.
   * @param locale Locale para formatar a data.
   */
  public RemoteObserversNotificationManager(
    RemoteObserverNotifierInterface notifier, Locale locale) {
    this.notifier = notifier;
    format =
      DateFormat.getDateTimeInstance(DateFormat.MEDIUM, DateFormat.MEDIUM,
        locale);
  }

  /**
   * Adiciona um novo observador.
   * 
   * @param observer observador
   * @return <code>true</code> se o observador ainda no havia sido adicionado
   */
  public boolean addObserver(ObserverData observer) {
    boolean inserted;
    writeLock.lock();
    try {
      inserted = observersSet.add(observer);
    }
    finally {
      writeLock.unlock();
    }
    if (inserted) {
      if (notifier.isLoggingNotifications()) {
        StringBuffer msg = new StringBuffer();
        msg.append(this);
        msg.append(": adicionado observador: ");
        msg.append(observer.observer);
        msg.append(" - ");
        msg.append(observer.arg);
        notifier.logNotification(msg.toString());
      }
    }
    else {
      if (notifier.isLoggingNotifications()) {
        StringBuffer msg = new StringBuffer();
        msg.append(this);
        msg.append(": observador j existia: ");
        msg.append(observer.observer);
        msg.append(" - ");
        msg.append(observer.arg);
        notifier.logNotification(msg.toString());
      }
    }
    return inserted;
  }

  /**
   * Remove um observador.
   * 
   * @param observer observador
   * @return <code>true</code> se o observador foi removido
   */
  public boolean deleteObserver(ObserverData observer) {
    boolean removed;
    writeLock.lock();
    try {
      removed = observersSet.remove(observer);
    }
    finally {
      writeLock.unlock();
    }
    if (removed) {
      if (notifier.isLoggingNotifications()) {
        StringBuffer msg = new StringBuffer();
        msg.append(this);
        msg.append(": removido observador: ");
        msg.append(observer.observer);
        msg.append(" - ");
        msg.append(observer.arg);
        notifier.logNotification(msg.toString());
      }
    }
    else {
      if (notifier.isLoggingNotifications()) {
        StringBuffer msg = new StringBuffer();
        msg.append(this);
        msg.append(": observador no existia: ");
        msg.append(observer.observer);
        msg.append(" - ");
        msg.append(observer.arg);
        notifier.logNotification(msg.toString());
      }
    }
    return removed;
  }

  /**
   * Remove todos os observadores.
   */
  public void deleteObservers() {
    writeLock.lock();
    try {
      observersSet.clear();
    }
    finally {
      writeLock.unlock();
    }
    if (notifier.isLoggingNotifications()) {
      StringBuffer msg = new StringBuffer();
      msg.append(this);
      msg.append(": todos os observadores foram removidos");
      notifier.logNotification(msg.toString());
    }
  }

  /**
   * Obtm o nmero de observadores.
   * 
   * @return nmero de observadores
   */
  public int numObservers() {
    readLock.lock();
    try {
      return observersSet.size();
    }
    finally {
      readLock.unlock();
    }
  }

  /**
   * Verifica se nenhum observador foi cadastrado.
   * 
   * @return <code>true</code> se nenhum observador foi cadastrado
   */
  public boolean isEmpty() {
    return numObservers() == 0;
  }

  /**
   * Envia uma notificao para todos os observadores, removendo aqueles que no
   * puderem ser notificados por erro de RMI.
   * <p>
   * <b>Para maior escalabilidade este mtodo deve ser executado em uma thread
   * separada do fluxo principal</b>.
   * 
   * @param event Evento a ser enviado para os observadores.
   */
  public void notifyObservers(RemoteEvent event) {
    readLock.lock();
    List<ObserverData> failedObservers = new ArrayList<ObserverData>();
    try {
      for (ObserverData obsData : observersSet) {
        try {
          if (notifier.has2update(obsData, event)) {
            if (notifier.isLoggingNotifications()) {
              StringBuffer msg = new StringBuffer();
              msg.append(this);
              msg.append(": notificando: ");
              msg.append(obsData.observer.toString());
              msg.append(" - ");
              msg.append(event.toString());
              msg.append(" de ");
              msg.append(format.format(event.getCreationDate()));
              notifier.logNotification(msg.toString());
            }
            obsData.observer.update(event);
            if (notifier.isLoggingNotifications()) {
              StringBuffer msg = new StringBuffer();
              msg.append(this);
              msg.append(": notificao a ");
              msg.append(obsData.observer.toString());
              msg.append(" enviada com sucesso");
              notifier.logNotification(msg.toString());
            }
          }
        }
        catch (ConnectException e) {
          notifier.handleFatalError(obsData, event, e);
          failedObservers.add(obsData);
        }
        catch (NoSuchObjectException e) {
          notifier.handleFatalError(obsData, event, e);
          failedObservers.add(obsData);
        }
        catch (Exception e) {
          notifier.handleException(obsData, event, e);
        }
      }
    }
    finally {
      readLock.unlock();
    }
    for (ObserverData observerData : failedObservers) {
      deleteObserver(observerData);
    }
  }

  /**
   * Envia uma lista de notificaes para todos os observadores, removendo
   * aqueles que no puderem ser notificados por erro de RMI.
   * <p>
   * <b>Para maior escalabilidade este mtodo deve ser executado em uma thread
   * separada do fluxo principal</b>.
   * 
   * @param notifier Objeto externo responsvel por determinar se o observador
   *        deve ser notificado e por exibir mensagens de erro.
   * @param events Os eventos a serem enviados para os observadores.
   */
  public void notifyObservers(RemoteObserverNotifierInterface notifier,
    RemoteEvent[] events) {
    readLock.lock();
    List<ObserverData> failedObservers = new ArrayList<ObserverData>();
    try {
      for (ObserverData obsData : observersSet) {
        try {
          if (notifier.has2update(obsData, events)) {
            if (notifier.isLoggingNotifications()) {
              StringBuffer msg = new StringBuffer();
              msg.append(this);
              msg.append(": notificando: ");
              msg.append(obsData.observer.toString());
              msg.append(" - ");
              for (int i = 0; i < events.length; i++) {
                if (i > 0) {
                  msg.append(" - ");
                }
                msg.append(events[i].toString());
                msg.append(" de ");
                msg.append(format.format(events[i].getCreationDate()));
              }
              notifier.logNotification(msg.toString());
            }
            obsData.observer.update(events);
            if (notifier.isLoggingNotifications()) {
              StringBuffer msg = new StringBuffer();
              msg.append(this);
              msg.append(": notificaes a ");
              msg.append(obsData.observer.toString());
              msg.append(" enviadas com sucesso");
              notifier.logNotification(msg.toString());
            }
          }
        }
        catch (ConnectException e) {
          notifier.handleFatalError(obsData, events, e);
          failedObservers.add(obsData);
        }
        catch (NoSuchObjectException e) {
          notifier.handleFatalError(obsData, events, e);
          failedObservers.add(obsData);
        }
        catch (Exception e) {
          notifier.handleException(obsData, events, e);
        }
      }
    }
    finally {
      readLock.unlock();
    }
    for (ObserverData observerData : failedObservers) {
      deleteObserver(observerData);
    }
  }

  /**
   * Adiciona um novo observador.
   * 
   * @param observer observador
   * @return <code>true</code> se o observador ainda no havia sido adicionado
   */
  public boolean addObserver(RemoteObserver observer) {
    return addObserver(new ObserverData(observer, null));
  }

  /**
   * Remove um observador.
   * 
   * @param observer observador
   * @return <code>true</code> se o observador foi removido
   */
  public boolean deleteObserver(RemoteObserver observer) {
    return deleteObserver(new ObserverData(observer, null));
  }
}
