package csbase.logic;

/**
 * Permisso para indicar se um usurio, em um determinado sistema, pode
 * executar um ou mais algoritmos instalados no sistema atual.
 * 
 * Essa permisso possui dois tipos de atributos:
 * 
 * - {@code sistema=<id_do_sistema>} : representa a identificao (string nica) do
 * sistema em que o usurio ter permisso para executa os algoritmos
 * especificados na permisso. Para representar o sistema atual, voc deve criar
 * um atributo "sistema=." Um exemplo de uso desse atributo com o valor
 * diferente de "." seria no caso de uso do OpenDreams para execuo de comandos
 * por fora do sistema atual. Nesse caso, o mesmo valor usado pelo Opendreams
 * para representar o sistema deve ser passado para o administrador do sistema
 * atual, para ser inserido na permisso.
 * 
 * - {@code nome=<nome do algoritmo>} : representa o nome do algoritmo que o usuario
 * pode executar ou uma expresso regular. Por exemplo, para permitir executar
 * todos os algoritmos, basta criar um atributo {@code "nome=.*"}.
 * 
 */
public class AlgorithmExecutionPermission extends AttributesPermission {
  /** Atributo que define o nome do algoritmo */
  public static final String ALGORITHM_NAME = "algoritmo=";

  /**
   * Atributo que define o identificador (string) nico do sistema em que essa
   * permisso vai ser vlida.
   */
  public static final String SYSTEM_ID = "sistema=";

  /**
   * Identificador do sistema corrente, caso nenhum sistema tenha sido
   * especificado. Esse sistema indica que a chamada remota foi originada no
   * sistema atual.
   */
  private final static String CURRENT_SYSTEM_ID = ".";

  /**
   * Constri uma permisso para execuo de algoritmos.
   */
  public AlgorithmExecutionPermission() {
    super();
  }

  /**
   * Constri uma permisso para execuo de algoritmos.
   * 
   * @param name nome da permisso
   * @param description descrio
   * @param attributes atributos da permisso
   */
  public AlgorithmExecutionPermission(String name, String description,
    String[] attributes) {
    super(name, description, attributes);
  }

  /**
   * Verifica se o usurio, no sistema especificado, tem permisso para executar
   * o algoritmo que possui um determinado nome.
   * 
   * @param user usurio que solicita um algoritmo
   * @param systemId identificador do sistema, em que foi originada a chamada
   *        remota
   * @param algoName nome do algoritmo a ser executado
   * 
   * @return retorna true se o usurio, no sistema especificado, tem permisso
   *         para executar o algoritmo com o nome especificado, caso contrrio,
   *         retorna false
   * @throws Exception erro ocorrido durante a busca pela permisso (ex: de rmi)
   * 
   */
  public static boolean checkSystemAndAlgorithmExecPermission(User user,
    String systemId, String algoName) throws Exception {
    boolean hasPermission = false;
    // Cria o atributo que indica o sistema que originou a chamada
    String systemAttribute = AlgorithmExecutionPermission.SYSTEM_ID
      + ((systemId == null) ? CURRENT_SYSTEM_ID : systemId);

    // Cria o atributo que indica o nome do algoritmo a ser executado
    String algoAttribute = AlgorithmExecutionPermission.ALGORITHM_NAME
      + algoName;

    String[] attributes = new String[] { systemAttribute, algoAttribute };
    AlgorithmExecutionPermission algoExecPermission;
    algoExecPermission = (AlgorithmExecutionPermission) user
      .getMatchAttributesPermission(AlgorithmExecutionPermission.class,
        attributes);
    if (algoExecPermission != null) {
      hasPermission = true;
    }
    return hasPermission;
  }
}