package csbase.logic;

import csbase.exception.BugException;
import org.apache.commons.codec.binary.Base64;

import javax.crypto.Cipher;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.spec.RSAKeyGenParameterSpec;

/**
 * Classe utilitria para criptografar e descriptografar a senha do login.
 * Possui mtodo para gerar um par de chaves pblica e privada. O servidor
 * precisa fornecer para o cliente uma chave pblica que ser utilizada para
 * criptografar a senha do login. O servidor dever utilizar a chave privada
 * para descriptografar a senha recebida pelo usurio.
 */
public final class LoginPasswordCipher {

  private static final String ALGORITHM = "RSA/ECB/PKCS1Padding";

  private LoginPasswordCipher() {
  }

  /**
   * Cria um par de chaves pblica e privada para ser utilizada na criptografia
   * da senha do login.
   * 
   * @return o par de chaves pblica e privada.
   * @throws BugException se ocorrer algum erro ao gerar o par de chaves.
   */
  public static KeyPair generateKeyPair() throws BugException {
    try {
      KeyPairGenerator rsaKeyPairGenerator =
        KeyPairGenerator.getInstance("RSA");
      RSAKeyGenParameterSpec spec =
        new RSAKeyGenParameterSpec(1024, RSAKeyGenParameterSpec.F4);
      rsaKeyPairGenerator.initialize(spec);
      return rsaKeyPairGenerator.generateKeyPair();
    }
    catch (Exception exception) {
      throw new BugException(
        "Erro ao gerar o par de chaves pblica e privada.", exception);
    }
  }

  /**
   * Mtodo utilitrio para criptografar a senha do login.
   *
   * @param plainPassword a senha em texto simples
   * @param publicKey a chave pblica
   * @return a senha criptografada
   * @throws BugException caso ocorra um erro ao criptografar a senha do login.
   */
  public static EncryptedPassword encrypt(String plainPassword,
    PublicKey publicKey) throws BugException {
    try {
      Cipher rsaCipher = Cipher.getInstance(LoginPasswordCipher.ALGORITHM);
      rsaCipher.init(Cipher.ENCRYPT_MODE, publicKey);

      byte[] passwordBytes = plainPassword.getBytes("UTF-8");
      byte[] encryptedPasswordBytes = rsaCipher.doFinal(passwordBytes);
      // TODO trocar pelo java.util.Base64 do Java 8
      // byte[] passwordEncoded =
      // Base64.getEncoder().encode(encryptedPasswordBytes);
      byte[] passwordEncoded = Base64.encodeBase64(encryptedPasswordBytes);
      String password = new String(passwordEncoded, "UTF-8");
      return new EncryptedPassword(password);
    }
    catch (Exception exception) {
      throw new BugException("Erro ao criptografar a senha do login.",
        exception);
    }
  }

  /**
   * * Mtodo utilitrio para descriptografar a senha do login.
   *
   * @param encryptedPassword a senha criptografada.
   * @param privateKey a chave privada
   * @return a senha em texto simples
   * @throws BugException caso ocorra um erro ao descriptografar a senha do
   *         login.
   */
  public static String decrypt(EncryptedPassword encryptedPassword,
    PrivateKey privateKey) throws BugException {
    try {
      Cipher rsaCipher = Cipher.getInstance(LoginPasswordCipher.ALGORITHM);
      rsaCipher.init(Cipher.DECRYPT_MODE, privateKey);
      // TODO trocar pelo java.util.Base64 do Java 8
      // byte[] decoded =
      // Base64.getDecoder().decode(encryptedPassword.getBytes("UTF-8"));
      byte[] decoded =
        Base64.decodeBase64(encryptedPassword.getPassword().getBytes("UTF-8"));
      byte[] decryptedBytes = rsaCipher.doFinal(decoded);
      String decryptedPassword = new String(decryptedBytes, "UTF-8");
      return decryptedPassword;
    }
    catch (Exception exception) {
      throw new BugException("Erro ao descriptografar a senha do login.",
        exception);
    }
  }
}
