/*
 * Role.java
 * 
 * $Author: cviana $ $Revision: 179359 $ - $Date: 2008-02-11 18:35:52 -0200
 * (Mon, 11 Feb 2008) $
 */
package csbase.logic;

import java.io.Serializable;
import java.rmi.RemoteException;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.List;
import java.util.Observable;
import java.util.Observer;
import java.util.Vector;

import csbase.remote.ClientRemoteLocator;
import csbase.util.restart.RestartListener;
import csbase.util.restart.RestartManager;

/**
 * A classe <code>Role</code> representa um perfil de permisses.
 */
public class Role implements Serializable, IdInterface {
  /** Identificao do perfil, gerado pelo LocalRoleProvider */
  private final Object id;

  /** Informaes do perfil: nome, descrio e lista de permisses */
  private final RoleInfo info;

  /** Cache local de perfis */
  private static Hashtable<Object, Role> roles = new Hashtable<Object, Role>(); //cache local

  /** Informa se a cache est completa */
  private static boolean hasAllRoles = false;

  /** Objeto que permite a observao local desse perfil */
  private static Observable observable = new Observable() {
    @Override
    public void notifyObservers(Object arg) {
      setChanged();
      super.notifyObservers(arg);
    }
  };

  /**
   * Cria um perfil. S ser executado no servidor. Objetos so serializados
   * para o cliente.
   * 
   * @param id o identificador do perfil
   * @param info as informaes do perfil
   */
  public Role(Object id, RoleInfo info) {
    this.id = id;
    this.info = info;
  }

  /**
   * Solicita ao provedor um perfil que possui uma determinada identificao.
   * 
   * @param id a identificao de um perfil
   * 
   * @return o perfil que possui a identificao requisitada ou null caso esse
   *         perfil no exista
   * 
   * @throws RemoteException falha de rmi
   */
  public static Role getRole(Object id) throws RemoteException {
    if (id == null) {
      return null;
    }
    Role role = roles.get(id);
    if ((role == null) && !hasAllRoles) {
      role = ClientRemoteLocator.administrationService.getRole(id);
      if (role != null) {
        roles.put(id, role);
      }
    }
    return role;
  }

  /**
   * Obtm o perfil que possui o nome especificado.
   * 
   * @param name nome do perfil procurado
   * 
   * @return perfil que possui o nome procurado ou null caso esse perfil no
   *         exista
   *
   * @throws RemoteException falha de rmi
   */
  public static Role getRoleByName(String name) throws Exception {
    List<Role> allRoles = getAllRoles();
    for (Role role : allRoles) {
      if (role.getName().equalsIgnoreCase(name)) {
        return role;
      }
    }
    return null;
  }

  /**
   * Solicita ao provedor todos os perfis cadastrados no sistema.
   * 
   * @return um vetor com os perfis
   * 
   * @throws RemoteException falha de rmi
   */
  public static List<Role> getAllRoles() throws RemoteException {
    if (hasAllRoles) {
      return rolesToVector();
    }
    List<Role> allRoles =
      ClientRemoteLocator.administrationService.getAllRoles();
    for (Role role : allRoles) {
      if (!roles.containsKey(role.getId())) {
        roles.put(role.getId(), role);
      }
    }
    hasAllRoles = true;
    return allRoles;
  }

  /**
   * Verifica se j existe um outro perfil com o mesmo nome.
   * 
   * @param id o identificador de um perfil
   * @param name o nome sendo procurado
   * 
   * @return true se existir uma outro perfil com o nome procurado ou false no
   *         caso de no existir
   *
   * @throws Exception em caso de erro.
   */
  public static boolean existsAnotherRole(Object id, String name)
    throws Exception {
    Role anotherRole = getRoleByName(name);
    if ((anotherRole != null) && !anotherRole.getId().equals(id)) {
      return true;
    }
    return false;
  }

  /**
   * Solicita ao provedor todos os perfis cadastrados no sistema. Os perfis
   * podem estar ordenados por nome, se assim for solicitado.
   * 
   * @param sortByName <code>true</code> se os perfis devero estar ordenadas
   *        alfabeticamente pelo nome e <code>false</code> se a ordenao no 
   *        necessria
   * 
   * @return um vetor com os perfis
   *
   * @throws Exception em caso de erro.
   */
  public static List<Role> getAllRoles(boolean sortByName) throws Exception {
    List<Role> allRoles = getAllRoles();
    if (sortByName) {
      Collections.sort(allRoles, getNameComparator());
    }
    return allRoles;
  }

  /**
   * Devolve um vetor com todos perfis da cache.
   * 
   * @return um vetor com todos os perfis da cache
   */
  private static Vector<Role> rolesToVector() {
    Vector<Role> allRoles = new Vector<Role>();
    for (Enumeration<Role> e = roles.elements(); e.hasMoreElements();) {
      allRoles.add(e.nextElement());
    }
    return allRoles;
  }

  /**
   * Solicita ao provedor para criar um novo perfil no sistema.
   * 
   * @param info as informaes do perfil a ser criado.
   * 
   * @return o perfil criado ou null caso j exista um perfil com a mesma
   *         identificao
   *
   * @throws Exception em caso de erro.
   */
  public static Role createRole(RoleInfo info) throws Exception {
    if ((info.name == null) || info.name.trim().equals("")) {
      throw new Exception("Role.createRole: Nome do perfil no pode ser vazio");
    }
    if (getRoleByName(info.name) != null) {
      return null;
    }
    Role role = ClientRemoteLocator.administrationService.createRole(info);
    if (role != null) {
      roles.put(role.getId(), role);
    }
    return role;
  }

  /**
   * Solicita ao provedor para modificar um perfil no sistema.
   * 
   * @param id a identificao do perfil a ser modificado
   * @param info as informaes a serem usuadas na modificao do perfil.
   * 
   * @return o perfil modificado ou null se a modificao no pode ser efetuada:
   *         modificao de nome ou se no existe perfil com o identificador
   *         especificado.
   *
   * @throws Exception em caso de erro.
   */
  public static Role modifyRole(Object id, RoleInfo info) throws Exception {
    Role role = getRole(id);
    if (role == null) {
      return null;
    }
    if (!role.getName().equals(info.name) && existsAnotherRole(id, info.name)) {
      return null;
    }
    role = ClientRemoteLocator.administrationService.modifyRole(id, info);
    if (role == null) {
      roles.put(id, role);
    }
    return role;
  }

  /**
   * Solicita ao provedor para remover um perfil do sistema.
   * 
   * @param id o identificador do perfil a ser removido do sistema.
   *
   * @throws Exception em caso de erro.
   */
  public static void deleteRole(Object id) throws Exception {
    ClientRemoteLocator.administrationService.deleteRole(id);
    roles.remove(id);
  }

  /**
   * Esse mtodo  chamado quando um servio de administrao sofre alguma
   * alterao relativa a perfis. Notifica seus observadores locais.
   * 
   * @param event o evento que ocorreu no provedor de perfis
   */
  public static void update(AdministrationEvent event) {
    Role role = (Role) event.item;
    Object id = role.getId();
    switch (event.type) {
      case AdministrationEvent.CREATE:
      case AdministrationEvent.MODIFY:
        roles.put(id, role);
        break;
      case AdministrationEvent.DELETE:
        roles.remove(id);
    }
    observable.notifyObservers(event);
  }

  /**
   * Adiciona um observador local da classe. Sempre que a classe  avisada pelo
   * servio de administrao de que alguma mudana ocorreu, os observadores
   * locais tambm so notificados.
   * 
   * @param o um observador local
   */
  public static void addObserver(Observer o) {
    observable.addObserver(o);
  }

  /**
   * Remove um observador local da classe.
   * 
   * @param o o observador a ser removido
   */
  public static void deleteObserver(Observer o) {
    observable.deleteObserver(o);
  }

  /**
   * Obtm o identificador do perfil.
   * 
   * @return o identificador do perfil.
   */
  public Object getId() {
    return id;
  }

  /**
   * Obtm nome do perfil.
   * 
   * @return o nome do perfil
   */
  public String getName() {
    return info.name;
  }

  /**
   * Obtm a descrio do perfil.
   * 
   * @return a descrio do perfil
   */
  public String getDescription() {
    return info.description;
  }

  /**
   * Obtm as permisses do perfil.
   * 
   * @return o array de permisses do perfil.
   */
  public Object[] getPermissionIds() {
    return info.permissionIds;
  }

  /**
   * Altera as permisses do perfil.
   * 
   * @param permissionIds array de permisses do perfil.
   */
  public void setPermissionIds(Object[] permissionIds) {
    info.permissionIds = permissionIds;
  }

  /**
   * Obtm uma cpia do RoleInfo deste perfil.
   * 
   * @return a cpia do RoleInfo
   */
  public RoleInfo getRoleInfo() {
    return (RoleInfo) info.clone();
  }

  /**
   * Obtm um comparator de <code>Role</code> pelo critrio de ordem alfabtica
   * do nome do perfil.
   * 
   * @return um comparador por nome
   */
  public static Comparator<Role> getNameComparator() {
    Comparator<Role> roleComparator = new Comparator<Role>() {
      public int compare(Role r1, Role r2) {
        return r1.getName().compareTo(r2.getName());
      }
    };
    return roleComparator;
  }

  /**
   * Obtm um comparator de <code>Role</code> pelo critrio de ordem alfabtica
   * da descrio do perfil.
   * 
   * @return um comparador por descrio
   */
  public static Comparator<Role> getDescrComparator() {
    Comparator<Role> roleComparator = new Comparator<Role>() {
      public int compare(Role r1, Role r2) {
        return r1.getDescription().compareTo(r2.getDescription());
      }
    };
    return roleComparator;
  }

  /**
   * Obtm um getter para o nome de <code>Role</code>.
   * 
   * @return um getter para o nome
   */
  public static Getter getNameGetter() {
    Getter nameGetter = new Getter() {
      public Object get(Object o) {
        Role role = (Role) o;
        return role.getName();
      }
    };
    return nameGetter;
  }

  /**
   * Obtm um getter para a descrio de <code>Role</code>.
   * 
   * @return um getter para a descrio
   */
  public static Getter getDescrGetter() {
    Getter descrGetter = new Getter() {
      public Object get(Object o) {
        Role role = (Role) o;
        return role.getDescription();
      }
    };
    return descrGetter;
  }

  /**
   * Verifica se um perfil  igual a um outro. Dois perfis de usurios so
   * considerados o mesmo se eles possuem o mesmo identificador.
   * 
   * @param obj o outro objeto com o qual esse perfil est sendo comparado.
   * 
   * @return true ou false, se o perfil for ou no igual a um outro.
   */
  @Override
  public boolean equals(Object obj) {
    if (!(obj instanceof Role)) {
      return false;
    }
    Role role = (Role) obj;
    return role.getId().equals(id);
  }

  /**
   * Calcula o cdigo hash do objeto.
   * 
   * @return Inteiro que representa o cdigo hash do objeto.
   */
  @Override
  public int hashCode() {
    return id.hashCode();
  }

  public boolean equalContents(Object obj) {
    if (!(obj instanceof Role)) {
      return false;
    }
    Role role = (Role) obj;
    return role.getId().equals(id) && role.getRoleInfo().equals(info);
  }

  /**
   * Verifica se algum perfil possui uma determinada permisso.
   * 
   * @param permissionId Identificador da permisso.
   * 
   * @return true caso o perfil possua esta permisso, ou false, caso contrrio.
   * 
   * @throws RemoteException falha de rmi
   */
  public static boolean hasAnyRoleWithPermission(Object permissionId)
    throws RemoteException {
    List<Role> rs = getAllRoles();
    for (int i = 0; i < rs.size(); i++) {
      Object[] perms = rs.get(i).getPermissionIds();
      if (perms != null) {
        for (int j = 0; j < perms.length; j++) {
          if (permissionId.equals(perms[j])) {
            return true;
          }
        }
      }
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return id + " - " + info.toString();
  }

  static {
    RestartManager.getInstance().addListener(new RestartListener() {
      public void restart() {
        Role.hasAllRoles = false;
        Role.roles = new Hashtable<Object, Role>();
        Role.observable.deleteObservers();
      }
    });
  }
}
