/*
 * HttpServiceInterface.java
 * $Author: cviana $
 * $Revision: 179359 $  - $Date: 2017-03-13 16:58:16 -0300 (Mon, 13 Mar 2017) $
 */
package csbase.remote;

import java.rmi.RemoteException;
import java.util.Map;

import csbase.exception.HttpServiceException;

/**
 * Interface remota para acesso ao Servio de download de  arquivos.
 *
 * @author Taciana Melcop
 */
public interface HttpServiceInterface extends ServiceInterface {
  /** Nome do servio para o {@code ServiceManager} */
  public String SERVICE_NAME = "HttpService";

  //tipos de downloads

  /** tipo desconhecido */
  public int UNKNOWN_DOWNLOAD_TYPE = 0;

  /** tipo arquivo usando o caminho unix completo */
  public int FILE_DOWNLOAD_TYPE = 1;

  /** tipo URL */
  public int URL_DOWNLOAD_TYPE = 2;

  /** tipo texto ascii */
  public int TEXT_DOWNLOAD_TYPE = 3;

  /** tipo relatrio pdf */
  public int PDF_REPORT_DOWNLOAD_TYPE = 4;

  /** tipo texto csv */
  public int CSV_DOWNLOAD_TYPE = 5;

  /**
   * Obtem a url atraves da qual o arquivo deve ser acessado para download.
   * 
   * @param userId Identificador do usurio no servidor http remoto.
   * @param password Senha do usurio para autenticar o usurio que est fazendo
   *        a chamada do mtodo.
   * @param projectId Identificador do projeto.
   * @param filePath Caminho do arquivo no projeto.
   * @return Texto que representa a URL para acesso ao arquivo.
   * @throws RemoteException em caso de erro de comunicao.
   */
  public String getDownloadURL(Object userId, String password,
    Object projectId, String[] filePath) throws RemoteException;

  /**
   * Obtem a url atraves da qual o arquivo deve ser acessado para upload.
   * 
   * @param userId Identificador do usurio no servidor http remoto.
   * @param password Senha do usurio para autenticar o usurio que est fazendo
   *        a chamada do mtodo.
   * @param projectId Identificador do projeto.
   * @param filePath Caminho do arquivo no projeto.
   * @param fileType Tipo do arquivo a ser obtido.
   * @return Texto que representa a URL para acesso ao arquivo.
   * @throws RemoteException em caso de erro de comunicao.
   */
  public String getUploadURL(Object userId, String password, Object projectId,
    String[] filePath, String fileType) throws RemoteException;

  /**
   * Obtem o tipo do arquivo a ser exibido
   *
   * @param accessCode de acesso do arquivo
   * @param filePath caminho do arquivo
   *
   * @return .
   *
   * @throws RemoteException em caso de erro de comunicao.
   */
  public String getFileType(String accessCode, String filePath)
  throws RemoteException;

  /**
   * Obtem o caminho do arquivo a ser exibido
   *
   * @param accessCode de acesso do arquivo
   *
   * @return caminho canonico do arquivo
   *
   * @throws RemoteException em caso de erro de comunicao.
   */
  public String getFilePath(String accessCode)
  throws RemoteException;

  /**
   * Obtem o caminho do arquivo a ser exibido
   *
   * @param accessCode de acesso do arquivo
   * @param filePath caminho do arquivo
   *
   * @return caminho canonico do arquivo
   *
   * @throws RemoteException em caso de erro de comunicao.
   */
  public String getFilePath(String accessCode, String filePath)
  throws RemoteException;

  /**
   * Obtem o caminho do arquivo de entrada do upload.
   *
   * @param accessCode de acesso do arquivo
   *
   * @return caminho canonico do arquivo
   *
   * @throws RemoteException em caso de erro de comunicao.
   */
  public String getPresentationPath(String accessCode)
  throws RemoteException;

  /**
   * Obtem o caminho do arquivo de resultado do upload.
   *
   * @param accessCode de acesso do arquivo
   *
   * @return caminho canonico do arquivo
   *
   * @throws RemoteException em caso de erro de comunicao.
   */
  public String getResultPath(String accessCode)
  throws RemoteException;

  /**
   * Remove o upload da lista de requisicoes e informa que o arquivo nao esta'
   * mais em construcao.
   *
   * @param accessCode de acesso do arquivo
   * @param fileName O nome original do arquivo que sofreu upload.
   *
   * @return boolean se a operacao foi realizada com sucesso.
   *
   * @throws HttpServiceException .
   * @throws RemoteException em caso de erro de comunicao.
   */
  public boolean finishUpload(String accessCode, String fileName)
  throws HttpServiceException, RemoteException;

  /**
   * Verifica se o arquivo e' diretorio.
   *
   * @param accessCode de acesso do arquivo
   *
   * @return boolean que indica se o arquivo e' diretorio
   *
   * @throws RemoteException em caso de erro de comunicao.
   */
  public boolean isDirectory(String accessCode)
  throws RemoteException;

  /**
   * Cria um arquivo
   *
   * @param accessCode de acesso do arquivo
   * @param fileName Nome do arquivo
   *
   * @return caso o tipo tenha sido setado com sucesso
   *
   * @throws RemoteException em caso de erro de comunicao.
   */
  public boolean createFile(String accessCode, String fileName)
  throws RemoteException;

  /**
   * Pega o texto que est armazenado na RequestInfo referente ao accessCode
   * passado. Se o accessCode for null ou invlido o mtodo retornar null.
   *
   * @param accessCode de acesso do arquivo
   *
   * @return Texto a ser jogado no browser
   *
   * @throws RemoteException em caso de erro de comunicao.
   */
  public String getText(String accessCode)
  throws RemoteException;

  /**
   * Pega o tipo de mtodo utilizado no download referente ao accessCode
   * passado. Se o accessCode for null ou invlido o mtodo retornar
   * {@code UNKNOWN_DOWNLOAD_TYPE}.
   *
   * @param accessCode de acesso do arquivo
   *
   * @return tipo de donwload
   *
   * @throws RemoteException em caso de erro de comunicao.
   */
  public int getDownloadType(String accessCode)
  throws RemoteException;

  /**
   * Pega a URL que ser chamada pelo browser quando o tipo de dowload for
   * TEXTO.
   *
   * @param userId identificao do usurio.
   * @param password senha do usurio.
   * @param text texto contendo o contedo do CSV.
   *
   * @return a URL.
   *
   * @throws RemoteException em caso de erro de comunicao.
   */
  public String getDownloadTextURL(Object userId, String password, String text)
  throws RemoteException;

  /**
   * Pega a URL que ser chamada pelo browser quando o tipo de dowload for
   * TEXTO, permitindo a escolha do nome do arquivo.
   *
   * @param userId identificao do usurio.
   * @param password senha do usurio.
   * @param text texto contendo o contedo do CSV.
   * @param fileName o nome do arquivo.
   *
   * @return a URL.
   *
   * @throws RemoteException em caso de erro de comunicao.
   */
  public String getDownloadTextURL(Object userId, String password, String text,
      String fileName) throws RemoteException;
  
  /**
   * Pega a URL que ser chamada pelo browser quando o tipo de dowload for
   * CSV.
   *
   * @param userId identificao do usurio.
   * @param password senha do usurio.
   * @param text texto contendo o contedo do CSV.
   *
   * @return URL de download do arquivo CSV.
   *
   * @throws RemoteException em caso de erro de comunicao.
   */
  public String getDownloadCSVURL(Object userId, String password, String text)
  throws RemoteException;

  /**
   * Pega a URL que ser chamada pelo browser quando o tipo de download for
   * CSV.
   *
   * @param userId identificao do usurio.
   * @param password senha do usurio.
   * @param text texto contendo o contedo do CSV.
   * @param fileName o nome do arquivo.
   *
   * @return URL de download do arquivo CSV.
   *
   * @throws RemoteException em caso de erro de comunicao.
   */
  public String getDownloadCSVURL(Object userId, String password, String text, String fileName)
  throws RemoteException;

  /**
   * Obtm os parmetros da RequestInfo para usos gerais
   *
   * @param accessCode de acesso do arquivo
   *
   * @return os parmetros.
   *
   * @throws RemoteException em caso de erro de comunicao.
   */
  public Map getParameters(String accessCode)
  throws RemoteException;

  /**
   * Obtm o nome da classe do driver JDBC para criao de uma conexo com um
   * banco de dados a partir do servlets(upload e download), caso seja
   * necessrio.
   *
   * @param accessCode de acesso do arquivo
   *
   * @return o nome da classe do driver JDBC.
   *
   * @throws RemoteException em caso de erro de comunicao.
   */
  public String getJdbcDriverClassName(String accessCode)
  throws RemoteException;

  /**
   * Obtm URL JDBC para criao de uma conexo com um  banco de dados a partir
   * do servlets(upload e download), caso seja necessrio.
   *
   * @param accessCode de acesso do arquivo
   *
   * @return a URL JDBC.
   *
   * @throws RemoteException em caso de erro de comunicao.
   */
  public String getJdbcURL(String accessCode)
  throws RemoteException;

  /**
   * Obtm o usurio JDBC para criao de uma conexo com um  banco de dados a
   * partir do servlets(upload e download), caso seja necessrio.
   *
   * @param accessCode de acesso do arquivo
   *
   * @return o usurio JDBC.
   *
   * @throws RemoteException em caso de erro de comunicao.
   */
  public String getJdbcUser(String accessCode)
  throws RemoteException;

  /**
   * Obtm a senha JDBC para criao de uma conexo com um  banco de dados a
   * partir do servlets(upload e download), caso seja necessrio.
   *
   * @param accessCode de acesso do arquivo
   *
   * @return a senha JDBC.
   *
   * @throws RemoteException em caso de erro de comunicao.
   */
  public String getJdbcPassword(String accessCode)
  throws RemoteException;
  
  /**
   * Obtm o nmero de requisies de download retidos em memria, aguardando
   * limpeza peridica.
   * 
   * @return o nmero de requisies.
   *
   * @throws RemoteException em caso de erro de comunicao.
   */
  public int getRequestListSize()
  throws RemoteException;
}
