package csbase.logic;

import java.io.Serializable;
import java.rmi.RemoteException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.List;
import java.util.Observable;
import java.util.Observer;
import java.util.Vector;

import csbase.exception.administration.AdministrationDeleteException;
import csbase.remote.ClientRemoteLocator;
import csbase.util.restart.RestartListener;
import csbase.util.restart.RestartManager;
import tecgraf.javautils.core.lng.LNG;

/**
 * Define a super-classe das permisses do sistema CSBase. Toda permisso tem um
 * nome. Sub-classes sero criadas para cada diferente tipo de permisso
 * concreta a ser utilizada pelo sistema. Para permitir construir a interface
 * grfica de gerncia de permisses, sero criadas sub-classes a partir das
 * sub-classes abaixo: SimplePermission, UserPasswordPermission e
 * AttributesPermission.
 */
public abstract class Permission implements Serializable, Comparable<Object>,
  IdInterface {
  /** Identificao de uma Permission. */
  private Object id;

  /** Nome desta permisso. */
  protected String name;

  /** Descrio desta permisso. */
  protected String description;

  /** Cache local de permisses */
  private static Hashtable<Object, Permission> permissions =
    new Hashtable<Object, Permission>(); //cache local

  /** Informa se a cache est completa */
  private static boolean hasAllPermissions = false;

  /** Objeto que permite a observao local da classe Permission. */
  private static Observable observable = new Observable() {
    @Override
    public void notifyObservers(Object arg) {
      setChanged();
      super.notifyObservers(arg);
    }
  };

  /**
   * Construtor.
   */
  protected Permission() {
  }

  /**
   * Construtor. Define o nome e a descrio da permisso sendo criada.
   *
   * @param name Nome desta permisso.
   * @param description Descrio desta permisso.
   */
  protected Permission(String name, String description) {
    this.name = (name != null ? name : "");
    this.description = (description != null ? description : "");
  }

  /**
   * Obtm o identificador da Permission.
   *
   * @return o identificador da Permission.
   */
  @Override
  public Object getId() {
    return id;
  }

  /**
   * Retorna o nome da permisso. Esse nome  fornecido pelo usurio quando 
   * criada uma instncia de alguma classe concreta de permisso.
   *
   * @return .
   */
  public String getName() {
    return name;
  }

  /**
   * Altera o nome da permisso. Esse nome  fornecido pelo usurio quando 
   * criada uma instncia de alguma classe concreta de permisso.
   *
   * @param name .
   */
  public void setName(String name) {
    this.name = name;
  }

  /**
   * Atribui o identificador da permisso. Esse identificador  gerado apenas no
   * servio.
   *
   * @param id .
   */
  public void setId(Object id) {
    this.id = id;
  }

  /**
   * Retorna um texto descritivo da permisso. Esse texto  fornecido pelo
   * usurio quando  criada uma instncia de alguma classe concreta de
   * permisso.
   *
   * @return .
   */
  public String getDescription() {
    return description;
  }

  /**
   * Altera o texto descritivo da permisso. Esse texto  fornecido pelo usurio
   * quando  criada uma instncia de alguma classe concreta de permisso.
   *
   * @param description .
   */
  public void setDescription(String description) {
    this.description = description;
  }

  /**
   * Solicita a lista de todas as classes de permisses cadastradas no sistema.
   *
   * @return um vetor com todas as permisses
   *
   * @throws Exception em caso de erro de comunicao.
   */
  public static List<String> getPermissionClasses() throws Exception {
    return ClientRemoteLocator.administrationService.getPermissionClasses();
  }

  /**
   * Valida a permisso como entrada na tela e retorna true se estiver ok
   * 
   * @return true se estiver ok
   */
  public boolean validate() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int compareTo(Object obj) {
    return this.name.compareTo(((Permission) obj).name);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (!(obj instanceof Permission)) {
      return false;
    }
    Permission p = (Permission) obj;
    return p.getId().equals(id);
  }

  /**
   * Calcula o cdigo hash do objeto.
   *
   * @return Inteiro que representa o cdigo hash do objeto.
   */
  @Override
  public int hashCode() {
    return id.hashCode();
  }

  /**
   * Verifica se o contedo de uma permisso  igual ao contedo desta.
   *
   * @param obj Permisso cujo contedo ser comparado ao contedo desta.
   * @return True, se o contedo das permisses for igual, false caso contrtio.
   */
  public boolean equalContents(Object obj) {
    if (!(obj instanceof Permission)) {
      return false;
    }
    Permission p = (Permission) obj;
    return p.getId().equals(id) && p.getName().equals(name) && p
      .getDescription().equals(description);
  }

  /**
   * Solicita a lista de todas as permisses cadastradas no sistema. As
   * pemisses so colocadas no cache.
   *
   * @return um vetor com todas as permisses
   *
   * @throws Exception em caso de erro de comunicao.
   */
  public static List<Permission> getAllPermissions() throws Exception {
    if (hasAllPermissions) {
      return permissionsToVector();
    }
    List<Permission> allPermissions = ClientRemoteLocator.administrationService
      .getAllPermissions();
    for (Permission permission : allPermissions) {
      if (!permissions.containsKey(permission.getId())) {
        permissions.put(permission.getId(), permission);
      }
    }
    hasAllPermissions = true;
    return allPermissions;
  }

  /**
   * Obtem a permisso que possui uma determinada identificao. A permisso 
   * guardada no cache, caso ela no esteja l.
   *
   * @param id a identificao de uma permisso
   *
   * @return a permisso que possui a identificao requisitada ou null caso
   *         essa permisso no exista.
   *
   * @throws RemoteException falha de rmi
   */
  public static Permission getPermission(Object id) throws RemoteException {
    if (id == null) {
      return null;
    }
    Permission permission = permissions.get(id);
    if ((permission == null) && !hasAllPermissions) {
      permission = ClientRemoteLocator.administrationService.getPermission(id);
      if (permission != null) {
        permissions.put(id, permission);
      }
    }
    return permission;
  }

  /**
   * Cria uma nova Permisso no sistema. No se pode criar uma nova Permisso
   * com o mesmo nome de outra j existente. A Permisso criada  colocada na
   * cache.
   *
   * @param permission os dados da nova Permisso
   *
   * @return a Permisso criada ou null caso j exista uma Permisso com o mesmo
   *         nome.
   *
   * @throws Exception em caso de erro de comunicao.
   */
  public static Permission createPermission(Permission permission)
    throws Exception {
    if (permission == null) {
      throw new Exception(LNG.get("csbase.logic.EmptyPermission"));
    }
    if (getPermissionByName(permission.getName()) != null) {
      return null;
    }
    Permission result = ClientRemoteLocator.administrationService
      .createPermission(permission);
    if (result != null) {
      permissions.put(result.getId(), result);
    }
    return result;
  }

  /**
   * Solicita uma permisso que possui um determinado nome.
   *
   * @param name o nome da permisso procurada
   *
   * @return a permisso que possui o nome procurado ou null caso essa permisso
   *         no exista.
   *
   * @throws Exception em caso de erro de comunicao.
   */
  public static Permission getPermissionByName(String name) throws Exception {
    List<Permission> allPermissions = Permission.getAllPermissions();
    for (Permission permission : allPermissions) {
      if (permission.getName().equalsIgnoreCase(name)) {
        return permission;
      }
    }
    return null;
  }

  /**
   * Modifica uma permisso no sistema. No se pode modificar o nome de uma
   * permisso se j existir uma outra com esse nome. A permisso modificada 
   * colocada no cache.
   *
   * @param id o identificador da permisso a ser modificada
   * @param permission os dados da permisso a ser modificada
   *
   * @return a permisso modificada ou null caso j exista uma permisso com o
   *         mesmo nome ou no exista permisso com o id especificado
   *
   * @throws Exception em caso de erro de comunicao.
   */
  public static Permission modifyPermission(Object id, Permission permission)
    throws Exception {
    Permission oldPermission = getPermission(id);
    if (oldPermission == null) {
      return null;
    }
    if (!oldPermission.getName().equals(permission.getName())
      && existsAnotherPermission(id, permission.getName())) {
      return null;
    }
    Permission result = ClientRemoteLocator.administrationService
      .modifyPermission(id, permission);
    permissions.put(id, result);
    return result;
  }

  /**
   * Verifica se j existe uma outra permisso com o mesmo nome.
   *
   * @param id o identificador de uma permisso
   * @param name o nome sendo procurado
   *
   * @return true se existir uma outra permisso com o nome procurado ou false
   *         no caso de no existir
   *
   * @throws Exception em caso de erro de comunicao.
   */
  public static boolean existsAnotherPermission(Object id, String name)
    throws Exception {
    Permission anotherPermission = getPermissionByName(name);
    if ((anotherPermission != null) && !anotherPermission.getId().equals(id)) {
      return true;
    }
    return false;
  }

  /**
   * Remove uma permisso do sistema. A permisso removida  retirada do cache.
   *
   * @param id a identificao da permisso a ser removida do sistema.
   *
   * @throws Exception em caso de erro de comunicao.
   * @throws AdministrationDeleteException em caso de erro na operao.
   */
  public static void deletePermission(Object id) throws Exception,
    AdministrationDeleteException {
    ClientRemoteLocator.administrationService.deletePermission(id);
    permissions.remove(id);
  }

  /**
   * Devolve um vetor com todas as permisses da cache.
   *
   * @return um vetor com todas as permisses da cache
   */
  private static Vector<Permission> permissionsToVector() {
    Vector<Permission> allPermissions = new Vector<Permission>();
    for (Enumeration<Permission> e = permissions.elements(); e
      .hasMoreElements();) {
      allPermissions.add(e.nextElement());
    }
    return allPermissions;
  }

  /**
   * Adiciona um observador local da classe. Sempre que a classe  avisada pelo
   * servio de administrao de que alguma mudana ocorreu, os observadores
   * locais tambm so notificados.
   *
   * @param obs um observador local
   */
  public static void addObserver(Observer obs) {
    if (observable != null) {
      observable.addObserver(obs);
    }
  }

  /**
   * Remove um observador local da classe Platform.
   *
   * @param obs o observador a ser removido
   */
  public static void deleteObserver(Observer obs) {
    if (observable != null) {
      observable.deleteObserver(obs);
    }
  }

  /**
   * Esse mtodo  chamado quando um servio de administrao sofre alguma
   * alterao relativa a permisses. Atualiza o cache de permisses e notifica
   * seus observadores locais.
   *
   * @param event a ao que ocorreu no servio de administrao
   */
  public static void update(AdministrationEvent event) {
    Permission permission = (Permission) event.item;
    Object id = permission.getId();
    switch (event.type) {
      case AdministrationEvent.CREATE:
      case AdministrationEvent.MODIFY:
        permissions.put(id, permission);
        break;
      case AdministrationEvent.DELETE:
        permissions.remove(id);
        break;
    }
    observable.notifyObservers(event);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return "[" + id + "," + name + "," + description + "]";
  }

  static {
    RestartManager.getInstance().addListener(new RestartListener() {
      @Override
      public void restart() {
        Permission.hasAllPermissions = false;
        Permission.permissions = new Hashtable<Object, Permission>();
        Permission.observable.deleteObservers();
      }
    });
  }
}
