package csbase.logic;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

/**
 * Classe de tratamento de informaes de progresso do comando. Analisa o texto
 * indicado [proveniente de uma chave 'progresso...'] e extrai o progresso do
 * comando em dois formatos: valor texto de progresso e tambm o valor numrico
 * correspondente, entre 0 e 100%.
 * <ul>
 * <li>Exemplo: Texto: "1/10" --- Retorno: { "1/10", 10 }
 * </ul>
 * <p>
 * Se o texto no tiver um formato numrico conhecido, o valor fica Double.NaN.
 * <ul>
 * <li>Exemplo: Texto: "Passo 1: lendo arquivo" --- Retorno: {
 * "Passo 1: lendo arquivo", Double.NaN }
 * </ul>
 * <p>
 * Para que seja extrado um valor numrico de progresso, espera-se que
 * 'progressData' tenha um dos formatos abaixo:
 * <ul>
 * <li>1/12
 * <li>0.3 [entre 0 e 1]
 * <li>20% [entre 0 e 100]
 * <li><b>Opcionalmente um texto fixo pode ser indicado depois de um '#':</b>
 * <ul>
 * <li>1/12 #Um texto qualquer para o progresso atual
 * <li>10% #Um texto qualquer para o progresso atual
 * <li>0.3 #Um texto qualquer para o progresso atual.<br>
 * Nesses casos o texto retornado  o texto indicado, e no o texto que gerou o
 * valor.
 * <ul>
 * <li>Exemplo: Texto: "1/10 #Passo 1" --- Retorno: { "Passo 1", 10 }
 * </ul>
 * </ul>
 * </ul>
 */
public class ProgressDataParser {

  /**
   * O valor para a chave que indica o progresso do algoritmo, para uso no mapa
   * de informaes dinmicas
   */
  private static final String PROGRESS_KEY = "progresso";

  /**
   * Padro para chaves de progresso de algoritmos com sufixo numrico, para uso
   * no mapa de informaes dinmicas.
   */
  private static final String PROGRESS_NODE_KEY_PATTERN = "^" + PROGRESS_KEY
    + "\\.[0-9]+$";

  /**
   * Determina se h informao de progresso disponvel.
   * 
   * @param dynamicData Dados dinmicos do comando.
   * @return verdadeiro se h informao de progresso ou falso, caso contrrio.
   */
  public boolean hasProgressData(Map<String, String> dynamicData) {
    if (dynamicData.containsKey(ProgressDataParser.PROGRESS_KEY)) {
      return true;
    }
    for (String key : dynamicData.keySet()) {
      if (key.matches(ProgressDataParser.PROGRESS_NODE_KEY_PATTERN)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Obtm a informao de progresso geral do comando.
   * 
   * @param dynamicData Dados dinmicos do comando.
   * @param expectedPartialProgress Nmero de dados esperados de progresso.
   *        Utilizado para determinar se  possvel calcular a informao geral
   *        de progresso a partir dos valores individuais de progresso.
   * @return os dados gerais de progresso.
   */
  public ProgressData extractOverallProgressData(
    Map<String, String> dynamicData, int expectedPartialProgress) {

    if (dynamicData == null) {
      return null;
    }

    String progressData = dynamicData.get(ProgressDataParser.PROGRESS_KEY);
    if (progressData != null) {
      if (expectedPartialProgress == 1) {
        // h um registro para o progresso total ...  ele mesmo ! 
        return parseProgressData(progressData);
      }
      else {
        return null;
      }
    }

    /*
     * No h registro para o progresso total ... ver se temos chaves 'parciais'
     * ... Ex: "progresso.1", "progresso.2"
     */
    Map<Integer, ProgressData> map = extractProgressDataMap(dynamicData);

    /*
     * Se no houver chaves para todos os valores esperados, no h como
     * calcular o valor consolidado de progresso.
     */
    if (expectedPartialProgress == map.size()) {
      return calculateOverallProgress(map.values());
    }
    else {
      return null;
    }
  }

  /**
   * Retorna um mapa de dados de progresso por chave numrica.
   * 
   * @param dynamicData Dados dinmicos do comando.
   * 
   * @return Os dados de progresso mapeados pela chave numrica que os
   *         identificam.
   */
  public Map<Integer, ProgressData> extractProgressDataMap(
    Map<String, String> dynamicData) {
    Map<Integer, ProgressData> values = new HashMap<Integer, ProgressData>();

    if (dynamicData == null) {
      return values;
    }

    Set<Entry<String, String>> entrySet = dynamicData.entrySet();
    for (Entry<String, String> entry : entrySet) {
      String key = entry.getKey();
      if (key.matches(ProgressDataParser.PROGRESS_NODE_KEY_PATTERN)) {
        String[] keyParts = key.split("\\.");
        if (keyParts.length == 2) {
          ProgressData data = parseProgressData(entry.getValue());
          if (data != null) {
            values.put(Integer.valueOf(keyParts[1]), data);
          }
        }
      }
    }
    return values;
  }

  /**
   * Calcula a mdia dos progressos dos dados fornecidos.
   * 
   * @param data dados de progresso.
   * @return a mdia dos valores de progresso.
   */
  private ProgressData calculateOverallProgress(Collection<ProgressData> data) {
    if ((data == null) || data.isEmpty()) {
      // no temos informaes vlidas de 'progresso parcial'
      return null;
    }

    // temos N registros vlidos ... tirar a mdia:
    double total = 0;
    for (ProgressData progress : data) {
      Double value = progress.getValue();
      if ((value == null) || value.isNaN()) {
        return null;
      }
      total += value;
    }
    double percent = total / data.size();
    return new ProgressData(percent);
  }

  /**
   * Extrai o progresso do comando em dois formatos: valor texto de progresso e
   * tambm o valor numrico correspondente, entre 0 e 100%.
   * 
   * @param data string que corresponde ao valor do progresso desse algoritmo,
   *        como "1/10" ou "Passo 1".
   * @return O progresso do comando em dois formatos: valor texto original e
   *         tambm o valor numrico correspondente, entre 0 e 100%. <br>
   *         Ex: { "1/10", 10 }.
   */
  private ProgressData parseProgressData(String data) {

    if (data == null) {
      return null;
    }

    String progressData = data.trim();

    String valueText, markedStepText;
    int textMarkerIndex = progressData.indexOf('#');

    if (textMarkerIndex == 0) {
      // '#' no incio ...ento temos apenas texto ...
      return new ProgressData(progressData.substring(1).trim());
    }
    else if (textMarkerIndex > 0) {
      // temos um valor e um texto fixo ... "1/10 #texto"
      valueText = progressData.substring(0, textMarkerIndex).trim();
      markedStepText = progressData.substring(textMarkerIndex + 1).trim();
      Double value = parseProgressValue(valueText);
      if (value.isNaN()) {
        return new ProgressData(progressData);
      }
      else {
        return new ProgressData(markedStepText, value);
      }
    }

    // no temos um texto fixo marcado por '#' ... o texto ser o valor em %
    Double value = parseProgressValue(progressData);
    if (value.isNaN()) {
      return new ProgressData(progressData);
    }
    return new ProgressData(progressData, value);
  }

  /**
   * Extrai um valor numrico do texto indicado, que corresponde ao valor do
   * progresso desse algoritmo. O valor  convertido para porcentagem [entre 0 e
   * 100]. O texto pode ter um dos formatos abaixo:
   * <ul>
   * <li>"1/12"
   * <li>"0.3" [entre 0 e 1]
   * <li>"20%" [j entre 0 e 100]
   * </ul>
   * 
   * @param progressData string que corresponde ao valor do progresso do
   *        algoritmo.
   * @return o valor correspondente, ou Double.NaN para formatos desconhecidos.
   */
  private Double parseProgressValue(String progressData) {
    try {
      if (progressData.endsWith("%")) {
        String valueStr = progressData.substring(0, progressData.length() - 1);
        return Double.parseDouble(valueStr);
      }

      if (progressData.indexOf("/") > 0) {
        String[] stringValues = progressData.split("/");
        double v1 = Integer.parseInt(stringValues[0]);
        double v2 = Integer.parseInt(stringValues[1]);
        return (v1 / v2) * 100.0;
      }

      // no  '%' nem '/' ? Ento tem que ser um valor entre 0 e 1:
      double progress = Double.parseDouble(progressData);
      if ((progress >= 0) && (progress <= 1)) {
        return progress * 100.0;
      }
      else {
        return Double.NaN;
      }
    }
    catch (Exception e) {
      // qualquer erro no formato ... nmero fica NaN ...
      return Double.NaN;
    }
  }

}
