/*
 * $Author:$ $Date:$ $Release:$
 */
package csbase.logic.algorithms.parameters;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;
import java.text.MessageFormat;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.jexpression.JExpression;
import tecgraf.javautils.jexpression.util.CalculatorHandler;
import csbase.exception.ParseException;
import csbase.exception.algorithms.ExpressionFunctionExecutionException;
import csbase.exception.algorithms.ExpressionFunctionNotFoundException;
import csbase.exception.algorithms.FormulaCreationException;
import csbase.exception.algorithms.FormulaEvaluationException;
import csbase.exception.algorithms.ParameterIsOffException;
import csbase.exception.algorithms.ParameterNotFoundException;
import csbase.exception.algorithms.ParameterValueNotExistsException;

/**
 * Representa uma expresso que gera uma frmula a ser avaliada.
 */
public final class Expression implements Serializable {

  /** O configurador. */
  private SimpleAlgorithmConfigurator configurator;

  /** Indica se a expresso gera erro. */
  private boolean error;

  /** O texto que descreve a expresso. */
  private String expressionText;

  /** A frmula. */
  private transient JExpression formula;

  /** Os parmetros do configurador. */
  private Set<SimpleParameter<?>> parameters;

  /**
   * Construtor.
   * 
   * @param expressionText O texto que descreve a expresso.
   * 
   * @throws FormulaCreationException em caso de falha na criao da frmula.
   * @throws ParseException em caso de falha na leitura do texto.
   */
  public Expression(String expressionText) throws FormulaCreationException,
    ParseException {
    this.parameters = new HashSet<SimpleParameter<?>>();
    setExpressionText(expressionText);
    createFormula();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (obj == null) {
      return false;
    }
    if (!getClass().equals(obj.getClass())) {
      return false;
    }
    return toString().equals(obj.toString());
  }

  /**
   * Avalia a expresso.
   * 
   * @param algorithmConfigurator o configurador.
   * @return verdadeiro, se bem sucedido ou falso, caso contrrio.
   * @throws FormulaEvaluationException em caso de falha na criao da frmula.
   */
  public boolean evaluate(SimpleAlgorithmConfigurator algorithmConfigurator)
    throws FormulaEvaluationException {
    if (algorithmConfigurator == null) {
      throw new IllegalArgumentException(MessageFormat.format(LNG.get(
    		  "csbase.logic.algorithms.nullParameter"),
    		  "configurator"));
    }
    this.error = false;
    this.configurator = algorithmConfigurator;
    try {
      Boolean result = formula.eval(new ExpressionHandler(), Boolean.class);
      if (this.error) {
        return false;
      }
      return result;
    }
    catch (ParameterIsOffException e) {
      return true;
    }
    catch (ParameterValueNotExistsException e) {
      return false;
    }
    catch (FormulaEvaluationException e) {
      throw e;
    }
    catch (Exception e) {
      throw new FormulaEvaluationException(e,LNG.get(
		  "csbase.logic.algorithms.parameters.ExpressionEvalError"),
		  this,
		  e.getLocalizedMessage());
    }
    finally {
      this.configurator = null;
    }
  }

  /**
   * Obtm o configurador.
   * 
   * @return o configurador.
   */
  public SimpleAlgorithmConfigurator getConfigurator() {
    return this.configurator;
  }

  /**
   * Obtm o conjunto imutvel de parmetros do configurador.
   * 
   * @return o conjunto de parmetros.
   */
  public Set<SimpleParameter<?>> getParameters() {
    return Collections.unmodifiableSet(this.parameters);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return toString().hashCode();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return this.expressionText;
  }

  /**
   * Adiciona um parmetro.
   * 
   * @param parameter o parmetro.
   * @return verdadeiro, se bem sucedido ou falso, caso contrrio.
   */
  private boolean addParameter(SimpleParameter<?> parameter) {
    if (parameter == null) {
      throw new IllegalArgumentException(MessageFormat.format(LNG.get(
    		  "csbase.logic.algorithms.nullParameter"),
    		  "parameter"));
    }
    return this.parameters.add(parameter);
  }

  /**
   * Cria a frmula a partir do texto que descreve a expresso.
   * 
   * @throws ParseException em caso de falha na leitura da expresso.
   */
  private void createFormula() throws ParseException {
    try {
      formula = JExpression.compile(expressionText);
    }
    catch (Exception e) {
      throw new ParseException(e,LNG.get(
		  "csbase.logic.algorithms.parameters.FormulaCreationError"),
		  this, e.getLocalizedMessage());
    }
  }

  /**
   * Cria os atributos transientes.
   * 
   * @param inputStream Leitor de objetos
   * 
   * @throws IOException em caso de erro na leitura
   * @throws ClassNotFoundException se no encontrar a classe do objeto sendo
   *         lido.
   */
  private void readObject(ObjectInputStream inputStream) throws IOException,
    ClassNotFoundException {
    inputStream.defaultReadObject();
    try {
      createFormula();
    }
    catch (ParseException e) {
      IllegalStateException ise = new IllegalStateException();
      ise.initCause(e);
      throw ise;
    }
  }

  /**
   * Define o texto que descreve a expresso.
   * 
   * @param expressionText o texto que descreve a expresso.
   */
  private void setExpressionText(String expressionText) {
    if (expressionText == null) {
      throw new IllegalArgumentException(MessageFormat.format(LNG.get(
		  "csbase.logic.algorithms.nullParameter"),
		  "expressionText"));
    }
    this.expressionText = expressionText;
  }

  /**
   * Tratador de expresses.
   * 
   * @author Tecgraf
   */
  private class ExpressionHandler extends CalculatorHandler {

    /**
     * {@inheritDoc}
     */
    @Override
    public Object handleVar(String name) throws Exception {
      SimpleParameter<?> parameter = getConfigurator().getSimpleParameter(name);
      if (parameter == null) {
        throw new ParameterNotFoundException(name);
      }
      if (!parameter.isEnabled() || !parameter.isVisible()) {
        throw new ParameterIsOffException(parameter.getName());
      }
      addParameter(parameter);
      Object value = parameter.getExpressionValue();
      if (value == null) {
        throw new ParameterValueNotExistsException(name);
      }
      return value;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object handleFunctionCall(String name, List<Object> args)
      throws Exception {
      Object[] arguments = new Object[args.size()];
      for (int i = 0; i < args.size(); i++) {
        Object argument = args.get(i);
        if (argument instanceof Boolean) {
          arguments[i] = new Boolean((Boolean) argument);
        }
        else if (argument instanceof Double) {
          arguments[i] = new Double((Double) argument);
        }
        else if (argument instanceof String) {
          arguments[i] = argument.toString();
        }
        else {
          throw new IllegalArgumentException(LNG.get(
    		  "csbase.logic.algorithms.parameters.InvalidElementType"));
        }
      }

      try {
        return ExpressionFunctionExecutor.getInstance().execute(name,
          getConfigurator(), arguments);
      }
      catch (ExpressionFunctionExecutionException e) {
        throw new FormulaEvaluationException(e);
      }
      catch (ExpressionFunctionNotFoundException e) {
        throw new FormulaEvaluationException(e);
      }
    }
  }
}