package csbase.logic;

import tecgraf.javautils.core.lng.LNG;

/**
 * Classe base para implementaes de {@link CommandFinalizationInfo}, que
 * contm as informaes bsicas sobre o estado de finalizao de um comando.
 */
public abstract class AbstractCommandFinalizationInfo implements
  CommandFinalizationInfo {

  /**
   * Enumerao que identifica o que levou esse comando ao estado de
   * {@link CommandStatus#FINISHED} ou
   * {@link CommandFinalizationType#NOT_FINISHED} caso este comando ainda no
   * tenha terminado de ser executado.
   */
  protected CommandFinalizationType finalizationType;

  /**
   * Enumerao que identifica o que levou esse comando ao trmino com falha
   * {@link CommandFinalizationType#FAILED}.
   */
  protected FailureFinalizationType failureCause;

  /**
   * Cdigo de sada do comando.
   */
  protected Integer exitCode;

  /**
   * Indica se o comando gerou log com alertas durante sua execuo.
   */
  protected boolean hasWarnings;

  /**
   * Tipo de informao de finalizao disponvel para o comando, pode ser
   * {@link CommandFinalizationInfo.FinalizationInfoType#SIMPLE} ou
   * {@link CommandFinalizationInfo.FinalizationInfoType#EXTENDED}.
   */
  protected FinalizationInfoType infoType;

  /**
   * Construtor.
   * 
   * @param infoType Tipo de informao de finalizao disponvel para o
   *        comando, pode ser
   *        {@link CommandFinalizationInfo.FinalizationInfoType#SIMPLE} ou
   *        {@link CommandFinalizationInfo.FinalizationInfoType#EXTENDED}. (No
   *        aceita {@code null})
   * @param finalizationType Tipo de finalizao do comando. (No aceita
   *        {@code null}).
   * @param failureCause Causa da falha do comando. (Pode ser null).
   * @param hasWarnings Indica se o comando gerou log de alertas durante sua
   *        execuo.
   * @param exitCode Cdigo de sada do comando.
   */
  public AbstractCommandFinalizationInfo(FinalizationInfoType infoType,
    CommandFinalizationType finalizationType,
    FailureFinalizationType failureCause, boolean hasWarnings, Integer exitCode) {
    if (finalizationType == null) {
      throw new IllegalArgumentException(LNG.get(
        "csbase.logic.NullCommandFinalizationType"));
    }
    if (infoType == null) {
      throw new IllegalArgumentException(LNG.get(
        "csbase.logic.NullFinalizationInfo"));
    }
    this.finalizationType = finalizationType;
    this.failureCause = failureCause;
    this.hasWarnings = hasWarnings;
    this.exitCode = exitCode;
    this.infoType = infoType;
  }

  /**
   * Construtor.
   * 
   * @param infoType Tipo de informao de finalizao disponvel para o
   *        comando, pode ser
   *        {@link CommandFinalizationInfo.FinalizationInfoType#SIMPLE} ou
   *        {@link CommandFinalizationInfo.FinalizationInfoType#EXTENDED}. (No
   *        aceita {@code null})
   * @param finalizationType Tipo de finalizao do comando. (No aceita
   *        {@code null}).
   * @param failureCause Causa da falha do comando. (Pode ser null).
   * @param hasWarnings Indica se o comando gerou log de alertas durante sua
   *        execuo.
   */
  public AbstractCommandFinalizationInfo(FinalizationInfoType infoType,
    CommandFinalizationType finalizationType,
    FailureFinalizationType failureCause, boolean hasWarnings) {
    this(infoType, finalizationType, failureCause, hasWarnings, null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FinalizationInfoType getInfoType() {
    return infoType;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Integer getExitCode() {
    return exitCode;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CommandFinalizationType getFinalizationType() {
    return finalizationType;
  }

  /**
   * Indica se o comando gerou log de alertas durante sua execuo.
   *
   * @return verdadeiro caso o comando tenha gerado alertas ou falso, caso
   * contrrio.
   */
  public boolean hasWarnings() {
    return hasWarnings;
  }

  /**
   * Determina se o comando gerou log de alertas durante sua execuo.
   *
   * @param hasWarnings verdadeiro caso o comando tenha gerado alertas ou falso,
   * caso contrrio.
   */
  public void setHasWarnings(boolean hasWarnings) {
    this.hasWarnings = hasWarnings;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FailureFinalizationType getFailureCause() {
    return failureCause;
  }

  /**
   * Atribui o cdigo de sada do comando.
   * 
   * @param exitCode O cdigo de sada.
   */
  public void setExitCode(Integer exitCode) {
    this.exitCode = exitCode;
  }

  /**
   * Atribui o tipo de finalizao do comando.
   * 
   * @param finalizationType O tipo de finalizao.
   */
  public void setFinalizationType(CommandFinalizationType finalizationType) {
    this.finalizationType = finalizationType;
  }
}
