package csbase.logic.algorithms;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parsers.StringToEnumConverter;
import tecgraf.javautils.core.lng.LNG;

/**
 * Enumera os possveis locais de execuo de um algoritmo.
 */
public enum ExecutionLocation {

  /**
   * Diretrio de binrios do algoritmo.
   */
  BINARY_DIR,
  /**
   * Diretrio de persistncia do comando.
   */
  PERSISTENCY_DIR,
  /**
   * Sandbox de execuo do comando.
   */
  SANDBOX;

  /**
   * Converte strings para valores da enumerao ExecutionLocation.
   */
  public static class ExecutionLocationConverter implements
    StringToEnumConverter<ExecutionLocation> {

    /**
     * Retorna as strings que correspondem a cada um dos possveis valores da
     * enumerao {@link ExecutionLocation}.
     * 
     * @param value O valor da enumerao.
     * @return Um array com todas as strings que podem representar o valor.
     */
    public String[] getPossibleMatches(ExecutionLocation value) {
      switch (value) {
        case BINARY_DIR:
          return new String[] { "BIN", "BIN_DIR", "BINARY_DIR" };
        case PERSISTENCY_DIR:
          return new String[] { "CMD", "CMD_DIR", "PERSISTENCY_DIR" };
        case SANDBOX:
          return new String[] { "SANDBOX", "SANDBOX_DIR" };
        default:
          return new String[] {};
      }
    }

    /**
     * {@inheritDoc}
     * 
     */
    @Override
    public ExecutionLocation valueOf(String string) throws ParseException {
      if (string != null) {
        String upperCaseString = string.toUpperCase();
        for (ExecutionLocation location : ExecutionLocation.values()) {
          String[] possibleMatches = getPossibleMatches(location);
          for (String possibleMatch : possibleMatches) {
            if (possibleMatch.toUpperCase().equals(upperCaseString)) {
              return location;
            }
          }
        }
      }
      throw new ParseException(String.format(LNG.get(
		  "csbase.logic.algorithms.StringNotMappedInEnum"),
		  string,
		  ExecutionLocation.class.getName()));
    }
  }
}
