package csbase.logic.algorithms.parameters;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import csbase.exception.ParseException;
import tecgraf.javautils.core.lng.LNG;
import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.parameters.validators.EnumerationParameterValidator;
import csbase.logic.algorithms.parameters.validators.SimpleParameterValidator;

/**
 * Coluna de Enumerao.
 *
 * @author lmoreira
 */
public final class EnumerationColumn extends TableColumn<EnumerationItem> {

  /**
   * Os tems da enumerao.
   */
  private List<EnumerationItem> items;

  /**
   * Cria a coluna.
   *
   * @param id O identificador da coluna (No aceita {@code null}).
   * @param label O rtulo da coluna (No aceita {@code null}).
   * @param defaultItem O valor-padro da coluna (No aceita {@code null}).
   * @param items Os tens da enumerao (No aceita {@code null} e no pode
   *        estar vazio).
   * @param mustSortItems indica se a enumerao deve estar ordenada.
   */
  public EnumerationColumn(String id, String label, EnumerationItem defaultItem,
    List<EnumerationItem> items, boolean mustSortItems) {
    super(id, label, defaultItem, false, true);
    if (mustSortItems) {
      Collections.sort(items);
    }
    setItems(items);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getCommandValue(EnumerationItem itemValue,
    CommandLineContext context) {
    if (itemValue == null) {
      return null;
    }
    return itemValue.getCommandValue(context);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getItemValueAsText(EnumerationItem itemValue) {
    return itemValue.getId();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public EnumerationItem getItemValueFromText(String itemValue)
    throws ParseException {
    if (itemValue == null) {
      return null;
    }
    for (EnumerationItem item : items) {
      if (itemValue.equals(item.getId())) {
        return item;
      }
    }
    throw new ParseException(LNG.get(
    		"csbase.logic.algorithms.parameters.ItemNotFound"),
    		itemValue);
  }

  /**
   * <p>
   * Obtm os tens desta coluna.
   * </p>
   *
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   *
   * @return a lista.
   */
  public List<EnumerationItem> getItems() {
    return Collections.unmodifiableList(items);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getValueToExport(EnumerationItem itemValue) {
    if (itemValue == null) {
      return null;
    }
    return itemValue.getId();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public EnumerationItem getValueToImport(Object importedValue) {
    String enumerationItemId = (String) importedValue;
    if (enumerationItemId == null) {
      return null;
    }
    for (EnumerationItem item : items) {
      if (enumerationItemId.equals(item.getId())) {
        return item;
      }
    }
    throw new IllegalArgumentException(String.format(LNG.get(
		"csbase.logic.algorithms.parameters.InvalidEnumColumnValue"),
		enumerationItemId,
		getLabel()));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public SimpleParameterValidator<EnumerationItem> getValidator() {
    return new EnumerationParameterValidator(isOptional());
  }

  /**
   * Atribui os tems desta coluna.
   *
   * @param items Os tens da enumerao (No aceita {@code null} e no pode
   *        estar vazio).
   */
  private void setItems(List<EnumerationItem> items) {
    if (items == null) {
      throw new IllegalArgumentException(MessageFormat.format(LNG.get(
    		  "csbase.logic.algorithms.nullParameter"),
    		  "items"));
    }
    if (items.isEmpty()) {
      throw new IllegalArgumentException(MessageFormat.format(LNG.get(
    		  "csbase.logic.algorithms.EmptyParameter"),
    		  "items"));
    }
    this.items = new ArrayList<EnumerationItem>(items);
  }
}
