package csbase.logic.algorithms.parameters;

import java.text.MessageFormat;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import csbase.exception.ParseException;
import csbase.logic.algorithms.CommandLineBuilder;
import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.EnvironmentVariable;
import csbase.logic.algorithms.parameters.validators.URLParameterValidator;
import tecgraf.javautils.core.lng.LNG;

/**
 * Classe base para parmetro do tipo URL.
 *
 * @author Tecgraf
 */
public abstract class URLParameter extends AbstractFileParameter {

  /**
   * Chave utilizada para informar o <b>protocolo</b> da URL no mapa de
   * parmetros dos mtodos de importao ({@link #importValue(Map)}) e
   * exportao ({@link #exportValue()}).
   */
  private static final String PROTOCOL = ".protocol";

  /**
   * Chave utilizada para informar o <b>host</b> da URL no mapa de parmetros
   * dos mtodos de importao ({@link #importValue(Map)}) e exportao (
   * {@link #exportValue()}).
   */
  private static final String HOST = ".host";

  /**
   * Define o argumento na linha de comando informando o protoloco usado pelo
   * usurio.
   */
  private String localization;

  /** Conjunto de protocolos aceitos na URL. */
  private EnumSet<URLProtocol> allowedProtocols;

  /**
   * Construtor.
   *
   * @param name nome do parmetro (No aceita {@code null}).
   * @param label rtulo do parmetro (No aceita {@code null}).
   * @param description descrio do parmetro (No aceita {@code null}).
   * @param defaultValue valor-padro (Aceita {@code null}).
   * @param isOptional indica se o valor do parmetro  opcional.
   * @param isVisible indica se o parmetro deve ficar visvel.
   * @param commandLinePattern padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   * @param localization define o argumento na linha de comando informando o
   *        protoloco usado pelo usurio.
   * @param fileTypes tipo dos arquivos aceitos neste parmetro (Aceita
   *        {@code null}).
   * @param mode modo do parmetro (No aceita {@code null}).
   * @param usesPipe Indica se este parmetro pode aceitar pipe
   *        {@link FileParameterPipeAcceptance#TRUE}, no aceita pipe
   *        {@link FileParameterPipeAcceptance#FALSE} ou *s* aceita pipe
   *        {@link FileParameterPipeAcceptance#ALWAYS}.
   * @param mustExist indica se o arquivo deve existir.
   * @param allowedProtocols conjunto de protocolos aceitos na URL.
   */
  public URLParameter(String name, String label, String description,
    FileURLValue defaultValue, boolean isOptional, boolean isVisible,
    String commandLinePattern, String localization, String[] fileTypes,
    FileParameterMode mode, FileParameterPipeAcceptance usesPipe,
    boolean mustExist, EnumSet<URLProtocol> allowedProtocols) {
    super(name, label, description, defaultValue, isOptional, isVisible,
      commandLinePattern, fileTypes, mode, usesPipe, mustExist);
    if (mode == null) {
      throw new IllegalArgumentException(MessageFormat.format(LNG.get(
    		  "csbase.logic.algorithms.nullParameter"),
    		  "mode"));
    }
    this.localization = localization;
    this.allowedProtocols = allowedProtocols;
  }

  /**
   * Define o argumento na linha de comando informando o protoloco usado pelo
   * usurio.
   *
   * @return localizao.
   */
  public String getLocalization() {
    return localization;
  }

  /**
   * Obtm o conjunto de protocolos aceitos na URL.
   *
   * @return conjunto de protocolos.
   */
  public EnumSet<URLProtocol> getAllowedProtocols() {
    return allowedProtocols;
  }

  /**
   * Obtm o protocolo padro do parmetro (caso no tenha sido especificado
   * nenhum).
   *
   * @return protocolo padro.
   */
  public URLProtocol getDefaultProtocol() {
    if (allowedProtocols.size() == 0) {
      return FileURLValue.DEFAULT_PROTOCOL;
    }
    return allowedProtocols.iterator().next();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public URLParameterValidator createParameterValidator() {
    return new URLParameterValidator(isOptional());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, Object> exportValue() {
    if (getValue() == null) {
      return new HashMap<String, Object>();
    }

    List<Map<String, String>> exportableValues =
      new LinkedList<Map<String, String>>();
    Map<String, String> exportableValue = new HashMap<String, String>();

    FileURLValue value = getValue();
    exportableValue.put(PATH, value.getPath());
    exportableValue.put(TYPE, value.getType());

    URLProtocol protocol = value.getProtocol();
    if (protocol != null) {
      exportableValue.put(PROTOCOL, protocol.getType());
    }
    exportableValue.put(HOST, value.getHost());
    exportableValues.add(Collections.unmodifiableMap(exportableValue));

    Map<String, Object> result = new HashMap<String, Object>();
    result.put(getName(), Collections.unmodifiableCollection(exportableValues));
    return Collections.unmodifiableMap(result);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  @SuppressWarnings("unchecked")
  public void importValue(Map<String, Object> parameterValues) {
    Collection<Map<String, String>> exportableValues =
      (Collection<Map<String, String>>) parameterValues.get(getName());

    if (exportableValues == null || exportableValues.isEmpty()) {
      setValue(null);
      return;
    }

    Map<String, String> exportableValue = exportableValues.iterator().next();
    String path = exportableValue.get(PATH);
    String type = exportableValue.get(TYPE);
    String protocolName = exportableValue.get(PROTOCOL);
    String host = exportableValue.get(HOST);
    if (protocolName != null) {
      try {
        URLProtocol protocol = new URLProtocolConverter().valueOf(protocolName);
        setValue(new FileURLValue(path, type, protocol, host));
        return;
      }
      catch (ParseException e) {
        throw new IllegalArgumentException("Protocolo " + protocolName
          + " invlido");
      }
    }

    setValue(new FileURLValue(path, type));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getCommandValue(CommandLineContext context) {
    FileURLValue file = getValue();
    if (file == null) {
      return null;
    }

    String path = file.getPath();
    URLProtocol protocol = file.getProtocol();

    if (!path.startsWith(CommandLineBuilder.REFERENCE_VAR_CHAR)) {
      char fileSeparator = context.getFileSeparator();
      path = file.getPath(fileSeparator);
      switch (protocol) {
        case PROJECT:
          String filePath = file.getPath(fileSeparator);
          if (filePath.startsWith("/")) {
            filePath = filePath.substring(1);
          }
          path =
            CommandLineBuilder.makePathWithEnvironmentVariable(
              EnvironmentVariable.PROJECT_DIR, filePath, fileSeparator);
          break;
        default:
          String commandValue = protocol.getCommandValue();
          if (commandValue != null) {
            path = commandValue + path;
          }
          break;
      }
    }

    String f = "%s %s=%s";

    return String.format(f, path, getLocalization(), protocol.getType());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getValueAsText() {
    FileURLValue file = getValue();

    return FileURLValue.getStringValue(file);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setValueAsText(String parameterValue) {

    FileURLValue value = FileURLValue.getURLFromString(mode, parameterValue);
    setValue(value);
  }

  /**
   * Cria os atributos transientes.
   *
   * @param in stream com o objeto serializado.
   *
   * @throws IOException em caso de erro de I/O.
   * @throws ClassNotFoundException se a classe do objeto serializado no for
   *         encontrada.
   */
  private void readObject(java.io.ObjectInputStream in) throws IOException,
    ClassNotFoundException {
    in.defaultReadObject();
    listeners = new LinkedList<FileParameterListener>();
  }

}
