package csbase.logic.algorithms.parsers;

import java.util.LinkedList;
import java.util.List;

import csbase.exception.ParseException;
import csbase.exception.algorithms.FormulaCreationException;
import csbase.logic.algorithms.parameters.BooleanParameter;
import csbase.logic.algorithms.parameters.DoubleParameter;
import csbase.logic.algorithms.parameters.EnumerationItem;
import csbase.logic.algorithms.parameters.EnumerationParameter;
import csbase.logic.algorithms.parameters.IntegerParameter;
import csbase.logic.algorithms.parameters.Parameter;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.TextParameter;
import csbase.logic.algorithms.parameters.conditions.AndOperatorCondition;
import csbase.logic.algorithms.parameters.conditions.Condition;
import csbase.logic.algorithms.parameters.conditions.GenericCondition;
import csbase.logic.algorithms.parameters.conditions.NotOperatorCondition;
import csbase.logic.algorithms.parameters.conditions.OrOperatorCondition;
import csbase.logic.algorithms.parameters.conditions.SimpleCondition;
import tecgraf.javautils.core.lng.LNG;

/**
 * <p>
 * Analisador de {@link Condition}.
 * </p>
 * 
 * <p>
 * Este parser l os atributos que representam as condies como
 * {@link AndOperatorCondition}, {@link NotOperatorCondition} e
 * {@link OrOperatorCondition}. O elemento corrente do {@link XmlParser
 * analisador de XML} precisa ser um elemento {@link Condition}.
 * </p>
 */
public class ConditionParser {

  /**
   * <p>
   * O elemento {@value #CONDITION_ELEMENT}: indica a {@link Condition condio}
   * de um gatilho.
   * </p>
   * <p>
   *  elemento filho de um gatilho, do operador OR, do operator AND ou do
   * operador OR.
   * </p>
   */
  static final String CONDITION_ELEMENT = "condicao";

  /**
   * <p>
   * O elemento {@value #AND_ELEMENT}: indica que o operador AND ser usado em
   * uma condio.
   * </p>
   * <p>
   *  filho de qualquer elemento do tipo gatilho ou  filho do elementos:
   * {@link #AND_ELEMENT}, {@link #OR_ELEMENT} ou {@link #NOT_ELEMENT}.
   * </p>
   */
  static final String AND_ELEMENT = "e";

  /**
   * <p>
   * O elemento {@value #NOT_ELEMENT}: indica que o operador NOT ser usado em
   * uma condio.
   * </p>
   * <p>
   *  filho de qualquer elemento do tipo gatilho ou  filho do elementos:
   * {@link #AND_ELEMENT}, {@link #OR_ELEMENT} ou {@link #NOT_ELEMENT}.
   * </p>
   */
  static final String NOT_ELEMENT = "nao";

  /**
   * <p>
   * O elemento {@value #OR_ELEMENT}: indica que o operador OR ser usado em uma
   * condio.
   * </p>
   * <p>
   *  filho de qualquer elemento do tipo gatilho ou  filho do elementos:
   * {@link #AND_ELEMENT}, {@link #OR_ELEMENT} ou {@link #NOT_ELEMENT}.
   * </p>
   */
  static final String OR_ELEMENT = "ou";

  /**
   * <p>
   * O atributo {@value #CONDITION_ELEMENT_PARAMETER_ATTRIBUTE} do elemento
   * {@link #CONDITION_ELEMENT}: indica o nome do parmetro utilizado na
   * condio.
   * </p>
   */
  private static final String CONDITION_ELEMENT_PARAMETER_ATTRIBUTE =
    "parametro";

  /**
   * <p>
   * O atributo {@value #CONDITION_ELEMENT_PARAMETER_ATTRIBUTE} do elemento
   * {@link #CONDITION_ELEMENT}: indica o valor do parmetro utilizado na
   * condio.
   * </p>
   */
  private static final String CONDITION_ELEMENT_VALUE_ATTRIBUTE = "valor";

  /**
   * <p>
   * Carrega uma condio, que pode ser:
   * </p>
   * <ul>
   * <li>{@link SimpleCondition}</li>
   * <li>{@link GenericCondition}</li>
   * <li>{@link AndOperatorCondition}</li>
   * <li>{@link OrOperatorCondition}</li>
   * <li>{@link NotOperatorCondition}</li>
   * </ul>
   * 
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento ou {@link #CONDITION_ELEMENT} ou {@link #OR_ELEMENT} ou
   * {@link #AND_ELEMENT} ou {@link #NOT_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return O operador.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  public Condition loadCondition(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    String elementName = parser.getElementName();
    if (elementName.equals(CONDITION_ELEMENT)) {
      return loadConditionElement(parser, configurator);
    }
    if (elementName.equals(OR_ELEMENT)) {
      return loadOrCondition(parser, configurator);
    }
    if (elementName.equals(AND_ELEMENT)) {
      return loadAndCondition(parser, configurator);
    }
    if (elementName.equals(NOT_ELEMENT)) {
      return loadNotCondition(parser, configurator);
    }
    return null;
  }

  /**
   * <p>
   * Carrega um {@link SimpleCondition} ou {@link GenericCondition}.
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #CONDITION_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return O operador.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private Condition loadConditionElement(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    String expressionText = parser.getElementValue(null);
    if (expressionText != null) {
      try {
        return new GenericCondition(expressionText);
      }
      catch (FormulaCreationException e) {
        throw new ParseException(e, LNG.get(
          "csbase.logic.algorithms.parsers.ExpressionParsingError"),
          expressionText, e.getLocalizedMessage());
      }
    }
    String parameterName = parser.extractAttributeValue(
      CONDITION_ELEMENT_PARAMETER_ATTRIBUTE);
    Parameter<?> parameter = configurator.getSimpleParameter(parameterName);
    if (parameter == null) {
      throw new ParseException(LNG.get(
		  "csbase.logic.algorithms.parsers.ExpressionParsingError"),
		  new Object[] { parameterName,
		  CONDITION_ELEMENT, CONDITION_ELEMENT_PARAMETER_ATTRIBUTE });
    }
    Object value;
    if (parameter instanceof BooleanParameter) {
      value = new Boolean(parser.extractAttributeValueAsBoolean(
        CONDITION_ELEMENT_VALUE_ATTRIBUTE));
    }
    else if (parameter instanceof TextParameter) {
      value = parser.extractAttributeValue(CONDITION_ELEMENT_VALUE_ATTRIBUTE,
        null);
    }
    else if (parameter instanceof DoubleParameter) {
      value = parser.extractAttributeValueAsDouble(
        CONDITION_ELEMENT_VALUE_ATTRIBUTE, null);
    }
    else if (parameter instanceof IntegerParameter) {
      value = parser.extractAttributeValueAsInteger(
        CONDITION_ELEMENT_VALUE_ATTRIBUTE, null);
    }
    else if (parameter instanceof EnumerationParameter) {
      String itemId = parser.extractAttributeValue(
        CONDITION_ELEMENT_VALUE_ATTRIBUTE);
      EnumerationParameter enumerationParameter =
        (EnumerationParameter) parameter;
      EnumerationItem item = enumerationParameter.getItem(itemId);
      if (item == null) {
        throw new ParseException(LNG.get(
    		"csbase.logic.algorithms.parsers.UndefinedConditionItem"),
    		new Object[] { parameter, itemId });
      }
      value = item;
    }
    else {
      throw new ParseException(LNG.get(
        "csbase.logic.algorithms.parsers.ParamNotAllowed"),
        new Object[] { parameter,
            BooleanParameterFactory.BOOLEAN_PARAMETER_ELEMENT,
            TextParameterFactory.TEXT_PARAMETER_ELEMENT,
            DoubleParameterFactory.DOUBLE_PARAMETER_ELEMENT,
            IntegerParameterFactory.INTEGER_PARAMETER_ELEMENT,
            EnumerationParameterFactory.ENUMERATION_PARAMETER_ELEMENT });
    }
    parser.checkAttributes();
    parser.checkChildElements();
    return new SimpleCondition(parameter.getName(), value);
  }

  /**
   * <p>
   * Carrega um {@link AndOperatorCondition}.
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #AND_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return O operador.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private AndOperatorCondition loadAndCondition(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    parser.checkAttributes();
    if (!parser.goToFirstChild()) {
      throw new ParseException(LNG.get(
    		  "csbase.logic.algorithms.parsers.UndefinedCondition"));
    }

    List<Condition> conditions = new LinkedList<Condition>();

    do {
      Condition condition = loadCondition(parser, configurator);
      if (condition != null) {
        conditions.add(condition);
      }
    } while (parser.goToNextSibling());
    if (conditions.size() < 2) {
      throw new ParseException(String.format(LNG.get(
		  "csbase.logic.algorithms.parsers.OrMissingCondition"),
		  conditions.size()));
    }

    AndOperatorCondition andOperatorCondition = new AndOperatorCondition(
      conditions);
    parser.goToParent();
    return andOperatorCondition;
  }

  /**
   * <p>
   * Carrega um {@link NotOperatorCondition}.
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #NOT_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return O operador.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private Condition loadNotCondition(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    parser.checkAttributes();
    if (!parser.goToFirstChild()) {
      throw new ParseException(LNG.get(
    		  "csbase.logic.algorithms.parsers.UndefinedCondition"));
    }
    Condition condition = loadCondition(parser, configurator);
    if (condition == null) {
      throw new ParseException(LNG.get(
    		  "csbase.logic.algorithms.parsers.UndefinedCondition"));
    }
    NotOperatorCondition notOperatorCondition = new NotOperatorCondition(
      condition);
    parser.goToParent();
    return notOperatorCondition;
  }

  /**
   * <p>
   * Carrega um {@link OrOperatorCondition}.
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #OR_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   * 
   * @return O operador.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private Condition loadOrCondition(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    parser.checkAttributes();
    if (!parser.goToFirstChild()) {
      throw new ParseException(LNG.get(
    		  "csbase.logic.algorithms.parsers.UndefinedCondition"));
    }

    List<Condition> conditions = new LinkedList<Condition>();

    do {
      Condition condition = loadCondition(parser, configurator);
      if (condition != null) {
        conditions.add(condition);
      }
    } while (parser.goToNextSibling());
    if (conditions.size() < 2) {
      throw new ParseException(String.format(LNG.get(
		  "csbase.logic.algorithms.parsers.OrMissingCondition"),
		  conditions.size()));
    }
    OrOperatorCondition orOperatorCondition = new OrOperatorCondition(
      conditions);
    parser.goToParent();
    return orOperatorCondition;
  }
}
