package csbase.logic.algorithms.parsers;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.BooleanParameter;
import csbase.logic.algorithms.parameters.DoubleListParameter;
import csbase.logic.algorithms.parameters.DoubleParameter;
import csbase.logic.algorithms.parameters.EnumerationItem;
import csbase.logic.algorithms.parameters.EnumerationParameter;
import csbase.logic.algorithms.parameters.FormulaParameter;
import csbase.logic.algorithms.parameters.InputFileParameter;
import csbase.logic.algorithms.parameters.IntegerListParameter;
import csbase.logic.algorithms.parameters.IntegerParameter;
import csbase.logic.algorithms.parameters.OutputFileParameter;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.TableParameter;
import csbase.logic.algorithms.parameters.TextListParameter;
import csbase.logic.algorithms.parameters.TextParameter;

/**
 * Classe abstrata que serve de base para parsers de parmetros do tipo
 * {@link SimpleParameterParser}.
 * 
 * @param <T> Tipo do parmetro do qual essa classe faz parsing
 */
public abstract class SimpleParameterParser<T extends SimpleParameter<?>>
  extends ParameterParser implements ParameterFactory {

  /**
   * <p>
   * O atributo {@value #PARAMETER_ELEMENT_DEFAULT_VALUE_ATTRIBUTE} dos
   * elementos:
   * <ul>
   * <li>{@link BooleanParameter}</li>
   * <li>{@link DoubleParameter}</li>
   * <li>{@link EnumerationParameter}</li>
   * <li>{@link FormulaParameter}</li>
   * <li>{@link IntegerParameter}</li>
   * <li>{@link TextParameter}</li>
   * </ul>
   * <p>
   * Indica o valor-padro do parmetro.  opcional. O seu tipo dependente do
   * tipo do parmetro:<br>
   * <table>
   * <tr>
   * <td>{@link BooleanParameter}</td>
   * <td>booleano</td>
   * </tr>
   * <tr>
   * <td>{@link DoubleParameter}</td>
   * <td>real</td>
   * </tr>
   * <tr>
   * <td>{@link EnumerationParameter}</td>
   * <td>string (o identificador de {@link EnumerationItem item de enumerao})
   * </td>
   * </tr>
   * <tr>
   * <td>{@link FormulaParameter}</td>
   * <td>string</td>
   * </tr>
   * <tr>
   * <td>{@link IntegerParameter}</td>
   * <td>inteiro</td>
   * </tr>
   * <tr>
   * <td>{@link TextParameter}</td>
   * <td>texto</td>
   * </tr>
   * </table>
   * </p>
   */
  static final String PARAMETER_ELEMENT_DEFAULT_VALUE_ATTRIBUTE = "padrao";

  /**
   * <p>
   * O atributo {@value #PARAMETER_ELEMENT_HIDDEN_ATTRIBUTE} dos elementos:
   * <ul>
   * <li>{@link BooleanParameter}</li>
   * <li>{@link DoubleListParameter}</li>
   * <li>{@link DoubleParameter}</li>
   * <li>{@link EnumerationParameter}</li>
   * <li>{@link FormulaParameter}</li>
   * <li>{@link InputFileParameter}</li>
   * <li>{@link IntegerListParameter}</li>
   * <li>{@link IntegerParameter}</li>
   * <li>{@link OutputFileParameter}</li>
   * <li>{@link TableParameter}</li>
   * <li>{@link TextListParameter}</li>
   * <li>{@link TextParameter}</li>
   * </ul>
   * <p>
   * Indica se o parmetro deve ficar visvel.  opcional, o seu valor-padro 
   * {@link #PARAMETER_ELEMENT_HIDDEN_ATTRIBUTE_DEFAULT_VALUE} e o seu tipo 
   * booleano.
   * </p>
   */
  static final String PARAMETER_ELEMENT_HIDDEN_ATTRIBUTE = "oculto";

  /**
   * <p>
   * O valor-padro para o atributo {@link #PARAMETER_ELEMENT_HIDDEN_ATTRIBUTE}.
   * </p>
   * <p>
   * O seu valor  {@value #PARAMETER_ELEMENT_HIDDEN_ATTRIBUTE_DEFAULT_VALUE}.
   * </p>
   */
  static final boolean PARAMETER_ELEMENT_HIDDEN_ATTRIBUTE_DEFAULT_VALUE = false;

  /**
   * <p>
   * O atributo {@value #PARAMETER_ELEMENT_DESCRIPTION_ATTRIBUTE} dos elementos:
   * <ul>
   * <li>{@link BooleanParameter}</li>
   * <li>{@link DoubleListParameter}</li>
   * <li>{@link DoubleParameter}</li>
   * <li>{@link EnumerationParameter}</li>
   * <li>{@link FormulaParameter}</li>
   * <li>{@link InputFileParameter}</li>
   * <li>{@link IntegerListParameter}</li>
   * <li>{@link IntegerParameter}</li>
   * <li>{@link OutputFileParameter}</li>
   * <li>{@link TableParameter}</li>
   * <li>{@link TextListParameter}</li>
   * <li>{@link TextParameter}</li>
   * </ul>
   * <p>
   * Indica a descrio do parmetro.  obrigatria e o seu tipo  string.
   * </p>
   */
  static final String PARAMETER_ELEMENT_DESCRIPTION_ATTRIBUTE = "dica";

  /**
   * <p>
   * O atributo {@value #PARAMETER_ELEMENT_LABEL_ATTRIBUTE} dos elementos:
   * <ul>
   * <li>{@link BooleanParameter}</li>
   * <li>{@link DoubleListParameter}</li>
   * <li>{@link DoubleParameter}</li>
   * <li>{@link EnumerationParameter}</li>
   * <li>{@link FormulaParameter}</li>
   * <li>{@link InputFileParameter}</li>
   * <li>{@link IntegerListParameter}</li>
   * <li>{@link IntegerParameter}</li>
   * <li>{@link OutputFileParameter}</li>
   * <li>{@link TableParameter}</li>
   * <li>{@link TextListParameter}</li>
   * <li>{@link TextParameter}</li>
   * </ul>
   * <p>
   * Indica o rtulo do parmetro.  obrigatria e o seu tipo  string.
   * </p>
   */
  static final String PARAMETER_ELEMENT_LABEL_ATTRIBUTE = "rotulo";

  /**
   * <p>
   * O atributo {@value #PARAMETER_ELEMENT_NAME_ATTRIBUTE} dos elementos:
   * <ul>
   * <li>{@link BooleanParameter}</li>
   * <li>{@link DoubleListParameter}</li>
   * <li>{@link DoubleParameter}</li>
   * <li>{@link EnumerationParameter}</li>
   * <li>{@link FormulaParameter}</li>
   * <li>{@link InputFileParameter}</li>
   * <li>{@link IntegerListParameter}</li>
   * <li>{@link IntegerParameter}</li>
   * <li>{@link OutputFileParameter}</li>
   * <li>{@link TableParameter}</li>
   * <li>{@link TextListParameter}</li>
   * <li>{@link TextParameter}</li>
   * </ul>
   * <p>
   * Indica o nome do parmetro.  obrigatria e o seu tipo  string.
   * </p>
   */
  static final String PARAMETER_ELEMENT_NAME_ATTRIBUTE = "nome";

  /**
   * <p>
   * O atributo {@value #PARAMETER_ELEMENT_OPTIONAL_ATTRIBUTE} dos elementos:
   * <ul>
   * <li>{@link BooleanParameter}</li>
   * <li>{@link DoubleListParameter}</li>
   * <li>{@link DoubleParameter}</li>
   * <li>{@link EnumerationParameter}</li>
   * <li>{@link FormulaParameter}</li>
   * <li>{@link InputFileParameter}</li>
   * <li>{@link IntegerListParameter}</li>
   * <li>{@link IntegerParameter}</li>
   * <li>{@link OutputFileParameter}</li>
   * <li>{@link TableParameter}</li>
   * <li>{@link TextListParameter}</li>
   * <li>{@link TextParameter}</li>
   * </ul>
   * <p>
   * Indica se o valor do parmetro  opcional.  opcional, o seu valor-padro 
   * {@link #PARAMETER_ELEMENT_OPTIONAL_DEFAULT_VALUE} e o seu tipo  booleano.
   * </p>
   */
  static final String PARAMETER_ELEMENT_OPTIONAL_ATTRIBUTE = "opcional";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #PARAMETER_ELEMENT_OPTIONAL_ATTRIBUTE}.
   * </p>
   * <p>
   * O seu valor  {@value #PARAMETER_ELEMENT_OPTIONAL_DEFAULT_VALUE}.
   * </p>
   */
  static final boolean PARAMETER_ELEMENT_OPTIONAL_DEFAULT_VALUE = false;

  /**
   * <p>
   * O atributo {@value #PARAMETER_ELEMENT_IGNORE_IF_DISABLED_ATTRIBUTE} dos
   * elementos:
   * <ul>
   * <li>{@link BooleanParameter}</li>
   * <li>{@link DoubleListParameter}</li>
   * <li>{@link DoubleParameter}</li>
   * <li>{@link EnumerationParameter}</li>
   * <li>{@link FormulaParameter}</li>
   * <li>{@link InputFileParameter}</li>
   * <li>{@link IntegerListParameter}</li>
   * <li>{@link IntegerParameter}</li>
   * <li>{@link OutputFileParameter}</li>
   * <li>{@link TableParameter}</li>
   * <li>{@link TextListParameter}</li>
   * <li>{@link TextParameter}</li>
   * </ul>
   * <p>
   * Indica se o parmetro deve ser ignorado (no deve fazer parte da linha de
   * comando gerada), caso esteja desabilitado no configurador.  opcional, o
   * seu valor-padro 
   * {@link #PARAMETER_ELEMENT_IGNORE_IF_DISABLED_DEFAULT_VALUE} e o seu tipo 
   * booleano.
   * </p>
   */
  static final String PARAMETER_ELEMENT_IGNORE_IF_DISABLED_ATTRIBUTE =
    "ignorar_se_desabilitado";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #PARAMETER_ELEMENT_IGNORE_IF_DISABLED_ATTRIBUTE}.
   * </p>
   * <p>
   * O seu valor  {@value #PARAMETER_ELEMENT_IGNORE_IF_DISABLED_DEFAULT_VALUE}.
   * </p>
   */
  static final boolean PARAMETER_ELEMENT_IGNORE_IF_DISABLED_DEFAULT_VALUE =
    false;

  /**
   * <p>
   * O atributo {@value #PARAMETER_ELEMENT_IGNORE_IF_DISABLED_DEFAULT_VALUE} dos
   * elementos:
   * <ul>
   * <li>{@link BooleanParameter}</li>
   * <li>{@link DoubleListParameter}</li>
   * <li>{@link DoubleParameter}</li>
   * <li>{@link EnumerationParameter}</li>
   * <li>{@link FormulaParameter}</li>
   * <li>{@link InputFileParameter}</li>
   * <li>{@link IntegerListParameter}</li>
   * <li>{@link IntegerParameter}</li>
   * <li>{@link OutputFileParameter}</li>
   * <li>{@link TableParameter}</li>
   * <li>{@link TextListParameter}</li>
   * <li>{@link TextParameter}</li>
   * </ul>
   * <p>
   * Indica se o parmetro deve ser ignorado (no deve fazer parte da linha de
   * comando gerada), caso esteja invisvel(oculto) no configurador.  opcional,
   * o seu valor-padro 
   * {@link #PARAMETER_ELEMENT_IGNORE_IF_INVISIBLE_DEFAULT_VALUE} e o seu tipo 
   * booleano.
   * </p>
   */
  static final String PARAMETER_ELEMENT_IGNORE_IF_INVISIBLE_ATTRIBUTE =
    "ignorar_se_oculto";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #PARAMETER_ELEMENT_IGNORE_IF_INVISIBLE_ATTRIBUTE}.
   * </p>
   * <p>
   * O seu valor  {@value #PARAMETER_ELEMENT_IGNORE_IF_INVISIBLE_DEFAULT_VALUE}
   * .
   * </p>
   */
  static final boolean PARAMETER_ELEMENT_IGNORE_IF_INVISIBLE_DEFAULT_VALUE =
    false;

  /**
   * Cria uma instncia do parmetro de tipo {@code <T>}, a partir dos atributos
   * bsicos do parmetro. As subclasses devem implementar esse mtodo, fazendo
   * a extrao dos demais atributos do parmetro.
   * 
   * @param parser Parser xml do configurador.
   * @param name Nome do parmetro.
   * @param label Rtulo do parmetro.
   * @param description Dica do parmetro.
   * @param isOptional Indica se o parmetro  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern Padro da linha de comando do parmetro.
   * @param group grupo do parmetro.
   * @param configurator Configurador de algoritmo.
   * @return Uma instncia do parmetro com os atributos bsicos e especficos
   *         preenchidos.
   * @throws ParseException Caso no seja possvel criar a instncia do
   *         parmetro com os atributos especificados.
   */
  public abstract T createSimpleParameter(XmlParser parser, String name,
    String label, String description, boolean isOptional, boolean isVisible,
    String commandLinePattern, ParameterGroup group,
    SimpleAlgorithmConfigurator configurator) throws ParseException;

  /**
   * {@inheritDoc}
   */
  @Override
  public T createParameter(final XmlParser parser,
    final String defaultCommandLinePattern, ParameterGroup group,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    final String name = parser.extractAttributeValue(
      PARAMETER_ELEMENT_NAME_ATTRIBUTE);
    final String label = parser.extractAttributeValue(
      PARAMETER_ELEMENT_LABEL_ATTRIBUTE);
    final String description = parser.extractAttributeValue(
      PARAMETER_ELEMENT_DESCRIPTION_ATTRIBUTE);
    final boolean ignoreIfInvisible = parser.extractAttributeValueAsBoolean(
      PARAMETER_ELEMENT_IGNORE_IF_INVISIBLE_ATTRIBUTE,
      PARAMETER_ELEMENT_IGNORE_IF_INVISIBLE_DEFAULT_VALUE);
    final boolean ignoreIfDisabled = parser.extractAttributeValueAsBoolean(
      PARAMETER_ELEMENT_IGNORE_IF_DISABLED_ATTRIBUTE,
      PARAMETER_ELEMENT_IGNORE_IF_DISABLED_DEFAULT_VALUE);
    final boolean isOptional = parser.extractAttributeValueAsBoolean(
      PARAMETER_ELEMENT_OPTIONAL_ATTRIBUTE,
      PARAMETER_ELEMENT_OPTIONAL_DEFAULT_VALUE);
    final boolean isHidden = parser.extractAttributeValueAsBoolean(
      PARAMETER_ELEMENT_HIDDEN_ATTRIBUTE,
      PARAMETER_ELEMENT_HIDDEN_ATTRIBUTE_DEFAULT_VALUE);

    final String parameterCommandLinePattern = loadCommandLinePattern(parser,
      defaultCommandLinePattern);
    final T parameter = createSimpleParameter(parser, name, label, description,
      isOptional, !isHidden, parameterCommandLinePattern, group, configurator);
    parser.checkAttributes();
    parameter.setIgnoreIfDisabled(ignoreIfDisabled);
    parameter.setIgnoreIfInvisible(ignoreIfInvisible);
    return parameter;
  }

}
