package csbase.logic.algorithms.parsers.triggers;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.BooleanParameter;
import csbase.logic.algorithms.parameters.DoubleParameter;
import csbase.logic.algorithms.parameters.EnumerationItem;
import csbase.logic.algorithms.parameters.EnumerationParameter;
import csbase.logic.algorithms.parameters.IntegerParameter;
import csbase.logic.algorithms.parameters.Parameter;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.TextParameter;
import csbase.logic.algorithms.parameters.conditions.Condition;
import csbase.logic.algorithms.parameters.triggers.ChangeDefaultValueTrigger;
import csbase.logic.algorithms.parameters.triggers.Trigger;
import csbase.logic.algorithms.parsers.BooleanParameterFactory;
import csbase.logic.algorithms.parsers.DoubleParameterFactory;
import csbase.logic.algorithms.parsers.EnumerationParameterFactory;
import csbase.logic.algorithms.parsers.IntegerParameterFactory;
import csbase.logic.algorithms.parsers.TextParameterFactory;
import csbase.logic.algorithms.parsers.XmlParser;

/**
 * Fbrica de {@link ChangeDefaultValueTrigger}.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ChangeDefaultValueTriggerFactory extends
  AbstractSimpleTriggerFactory {
  /**
   * Atributo {@value #NEW_DEFAULT_VALUE_ATTRIBUTE}: indica o novo valor-padro,
   *  opcional para os parmetros do tipo texto, inteiro e real (na ausncia de
   * valor ser utilizado {@code null}), e obrigatrio para os parmetros do
   * tipo booleano e enumerao.
   */
  private static final String NEW_DEFAULT_VALUE_ATTRIBUTE = "novo_valor_padrao";

  /**
   * Cria a fbrica.
   */
  public ChangeDefaultValueTriggerFactory() {
    super("trocar_valor_padrao", false);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Trigger<?> doCreateTrigger(XmlParser parser, Condition condition,
    Parameter<?> parameter, SimpleAlgorithmConfigurator configurator)
      throws ParseException {
    if (parameter instanceof TextParameter) {
      TextParameter textParam = (TextParameter) parameter;
      String defaultValue = parser.extractAttributeValue(
        NEW_DEFAULT_VALUE_ATTRIBUTE, null);
      return new ChangeDefaultValueTrigger<String>(textParam, condition,
        defaultValue);
    }
    else if (parameter instanceof IntegerParameter) {
      IntegerParameter intParam = (IntegerParameter) parameter;
      Integer defaultValue = parser.extractAttributeValueAsInteger(
        NEW_DEFAULT_VALUE_ATTRIBUTE, null);
      return new ChangeDefaultValueTrigger<Integer>(intParam, condition,
        defaultValue);
    }
    else if (parameter instanceof DoubleParameter) {
      DoubleParameter doubleParam = (DoubleParameter) parameter;
      Double defaultValue = parser.extractAttributeValueAsDouble(
        NEW_DEFAULT_VALUE_ATTRIBUTE, null);
      return new ChangeDefaultValueTrigger<Double>(doubleParam, condition,
        defaultValue);
    }
    else if (parameter instanceof BooleanParameter) {
      BooleanParameter booleanParam = (BooleanParameter) parameter;
      Boolean defaultValue = new Boolean(parser.extractAttributeValueAsBoolean(
        NEW_DEFAULT_VALUE_ATTRIBUTE));
      return new ChangeDefaultValueTrigger<Boolean>(booleanParam, condition,
        defaultValue);

    }
    else if (parameter instanceof EnumerationParameter) {
      EnumerationParameter enumParameter = (EnumerationParameter) parameter;
      String itemId = parser.extractAttributeValue(NEW_DEFAULT_VALUE_ATTRIBUTE);
      EnumerationParameter enumerationParameter =
        (EnumerationParameter) parameter;
      EnumerationItem defaultValue = enumerationParameter.getItem(itemId);
      if (defaultValue == null) {
        throw new ParseException(
          "O item de enumerao {0} no est presente na enumerao {1}.",
          itemId, parameter);
      }
      return new ChangeDefaultValueTrigger<EnumerationItem>(enumParameter,
        condition, defaultValue);
    }
    else {
      throw new ParseException("O parmetro {0} no  de um tipo vlido.\n"
        + "Tipos permitidos:\n{1};\n{2};\n{3};\n{4};\n{5}.", parameter
          .getName(), TextParameterFactory.TEXT_PARAMETER_ELEMENT,
        IntegerParameterFactory.INTEGER_PARAMETER_ELEMENT,
        DoubleParameterFactory.DOUBLE_PARAMETER_ELEMENT,
        BooleanParameterFactory.BOOLEAN_PARAMETER_ELEMENT,
        EnumerationParameterFactory.ENUMERATION_PARAMETER_ELEMENT);
    }
  }
}
