/**
 * $Id: Statistics.java 154918 2014-08-01 00:49:36Z fpina $
 */
package csbase.console;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import csbase.logic.User;
import csbase.logic.UserOutline;
import csbase.logic.algorithms.ExecutionType;
import csbase.logic.diagnosticservice.CommandExecutionStatisticsInfo;
import csbase.logic.diagnosticservice.LoginStatisticsInfo;
import csbase.logic.diagnosticservice.UsersStatisticsInfo;
import csbase.remote.ClientRemoteLocator;

/**
 * Ferramenta para exibio de estatsticas armazenadas pelo servidor.
 * 
 * @author Tecgraf PUC-Rio
 */
public class Statistics extends AbstractConsoleApp {

  /**
   * Construtor.
   * 
   * @param args argumentos provenientes da linha de comando
   */
  public Statistics(String[] args) {
    super(args);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final String getLogin() {
    return (String) User.getAdminId();
  }

  /**
   * Exibe as estatsticas.
   * 
   * @throws RemoteException erro na comunicao RMI
   */
  private void showStats() throws RemoteException {
    showBasicInfo();
    /*
     * Nmero de usurios registrados e conectados
     */
    showUsersInfo();
    /*
     * top 10 logins e falhas de autenticao
     */
    showLoginStats(10);
    /*
     * Estatisticas de execues de comandos.
     */
    CommandExecutionStatisticsInfo executionStats =
      ClientRemoteLocator.diagnosticService.getCommandExecutionStatisticsInfo();
    /*
     * execues por tipo
     */
    showExeTypeStats(executionStats);
    /*
     * execues simples por algoritmo
     */
    showStats("Execues simples, por algoritmo:",
      executionStats.simpleAlgoExecutionStats);
    /*
     * execues de fluxos, por algoritmo
     */
    showStats("Execues de fluxos, por algoritmo:",
      executionStats.flowAlgoExecutionStats);
    /*
     * execues por usurio (top 10)
     */
    showUserExecutionStats(executionStats, 10);
    /*
     * execues por mquina
     */
    showStats("Execues por mquina:", executionStats.sgasExecutionStats);
    /*
     * execues por resultado (execues simples)
     */
    showStats("Resultados das execues simples:",
      executionStats.simpleAlgoResultsStats);
    /*
     * execues por resultado (fluxos)
     */
    showStats("Resultados das execues dos fluxos:",
      executionStats.flowAlgoResultsStats);

    println();
  }

  /**
   * Exibe o nmero de usurios cadastrados no sistema e o nmero de usurios
   * conectados. Esses nmeros no incluem o admin e a prpria sesso usada por
   * essa aplicao.
   * 
   * @throws RemoteException erro na comunicao RMI
   */
  private void showUsersInfo() throws RemoteException {
    UsersStatisticsInfo usersBasicInfo =
      ClientRemoteLocator.diagnosticService.getUsersStatisticsInfo();
    int numUsers = usersBasicInfo.numRegisteredUsers;
    /*
     * subtramos 1 para no incluir o admin
     */
    println("\nUsurios cadastrados: %d", numUsers - 1);
    UserOutline[] loggedUsers = usersBasicInfo.connectedUsers;
    /*
     * subtramos 1 para no considerar a sesso corrente
     */
    println("\nUsurios conectados: %d", loggedUsers.length - 1);
  }

  /**
   * Exibe o nmero de execues por usurio.
   * 
   * @param executionStats informaes estatsticas de execuo de comandos.
   * @param limit nmero mximo de itens a exibir
   */
  private void showUserExecutionStats(
    CommandExecutionStatisticsInfo executionStats, int limit) {
    showSortedMap("Execues por usurio", limit,
      executionStats.userExecutionStats);
  }

  /**
   * Exibe estatsticas sobre os tipos de execuo.
   * 
   * @param executionStats informaes estatsticas de execuo de comandos.
   * @return <code>true</code> se as estatsticas foram exibidas,
   *         <code>false</code> caso contrrio (no havia dados de execuo)
   */
  private boolean showExeTypeStats(CommandExecutionStatisticsInfo executionStats) {
    Map<ExecutionType, Integer> exeStats = executionStats.exeStats;
    String obs = null;
    int flowStats = executionStats.flowStats;
    if (flowStats > 0) {
      obs = "(" + flowStats + " fluxos)";
    }
    return showStats("Execues por tipo:", exeStats, obs);
  }

  /**
   * Exibe estatsticas de logins bem-sucedidos e de falhas de autenticao,
   * ordenados de forma decrescente pelo nmero de ocorrncias.
   * 
   * @param limit nmero mximo de itens a exibir
   * @throws RemoteException erro na comunicao RMI
   */
  private void showLoginStats(int limit) throws RemoteException {
    LoginStatisticsInfo loginInfo =
      ClientRemoteLocator.diagnosticService.getLoginStatisticsInfo();
    showSortedMap("Usurios autenticados com sucesso", limit,
      loginInfo.succeededLogins);
    if (!loginInfo.failedLogins.isEmpty()) {
      showSortedMap("Erros de autenticao", limit, loginInfo.failedLogins);
    }
  }

  /**
   * Exibe um mapa ordenado de forma descrescente, pelos seus valores.
   * 
   * @param <T> o tipo da chave do mapa
   * @param header informao a ser exibida antes do mapa
   * @param limit nmero mximo de itens a exibir
   * @param statsMap mapa
   * @return <code>true</code> se o mapa foi exibido, <code>false</code> se o
   *         mapa estava vazio e no foi exibido
   */
  private <T> boolean showSortedMap(String header, int limit,
    Map<T, Integer> statsMap) {
    if (statsMap.isEmpty()) {
      return false;
    }
    println("\n%s (top %d):", header, limit);
    return showSortedMap(limit, statsMap);
  }

  /**
   * Exibe um mapa ordenado de forma decrescente pelos seus valores.
   * 
   * @param <T> tipo das chaves do mapa
   * @param limit nmero mximo de itens a exibir
   * @param statsMap mapa
   * @return <code>true</code> se o mapa foi exibido, <code>false</code> se o
   *         mapa estava vazio e no foi exibido
   */
  private <T> boolean showSortedMap(int limit, Map<T, Integer> statsMap) {
    if (statsMap.isEmpty()) {
      return false;
    }
    List<T> sortedKeys = sortMapByValues(statsMap, true);
    int maxItems = limit > 0 ? limit : sortedKeys.size();
    int maxLen = getMaxStrLen(sortedKeys);
    String format = "%s%-" + maxLen + "s : %d";
    for (T key : sortedKeys) {
      println(format, IDENT, key, statsMap.get(key));
      if (--maxItems == 0) {
        break;
      }
    }
    return true;
  }

  /**
   * Dado um mapa com contadores, exibe as chaves com seus respectivos valores,
   * alm de um total.
   * 
   * @param <T> tipo das chaves do mapa
   * @param header cabealho a ser exibido
   * @param map mapa de contadores
   * @return <code>true</code> se o mapa foi exibido, <code>false</code> se o
   *         mapa estava vazio e no foi exibido
   * 
   * @see #showStats(String, Map, String)
   */
  private <T> boolean showStats(String header, Map<T, Integer> map) {
    return showStats(header, map, null);
  }

  /**
   * Dado um mapa com contadores, exibe as chaves com seus respectivos valores,
   * alm de um total. Exibe tambm uma informao adicional ao lado do total.
   * 
   * @param <T> tipo das chaves do mapa
   * @param header cabealho a ser exibido
   * @param map mapa de contadores
   * @param totalObs informao a ser exibida ao lado do total (se for
   *        <code>null</code> no ser exibida)
   * @return <code>true</code> se o mapa foi exibido, <code>false</code> se o
   *         mapa estava vazio e no foi exibido
   * 
   * @see #showStats(String, Map)
   */
  private <T> boolean showStats(String header, Map<T, Integer> map,
    String totalObs) {
    if (map.isEmpty()) {
      return false;
    }
    println('\n' + header);
    int total = 0;
    int maxLen = getMaxStrLen(map.keySet());
    String formatStr = "%s%-" + maxLen + "s : %d";
    for (Entry<T, Integer> entry : map.entrySet()) {
      Integer v = entry.getValue();
      println(formatStr, IDENT, entry.getKey(), v);
      total += v;
    }
    if (totalObs == null) {
      println(formatStr, IDENT, "TOTAL", total);
    }
    else {
      println(formatStr + " %s", IDENT, "TOTAL", total, totalObs);
    }
    return true;
  }

  /**
   * Retorna uma lista contendo as chaves em uma ordem que corresponde 
   * ordenao do mapa pelos seus valores.
   * 
   * @param <T> tipo das chaves
   * @param map mapa
   * @param descending <code>true</code> se a ordenao  desrescente
   * 
   * @return lista contendo as chaves do mapa, ordenada de acordo com os valores
   *         deste
   */
  private static <T> List<T> sortMapByValues(final Map<T, Integer> map,
    final boolean descending) {
    Comparator<T> comparator = new Comparator<T>() {
      @Override
      public int compare(T o1, T o2) {
        if (descending) {
          return map.get(o2).compareTo(map.get(o1));
        }
        return map.get(o1).compareTo(map.get(o2));
      }
    };
    List<T> result = new ArrayList<T>(map.keySet());
    Collections.sort(result, comparator);
    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected BasicParams createParams() {
    return new BasicParams();
  }

  /**
   * @param args argumentos provenientes da linha de comando
   */
  public static void main(String[] args) {
    Statistics stats = null;
    try {
      stats = new Statistics(args);
      if (stats.login()) {
        stats.showStats();
      }
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    finally {
      if (stats != null) {
        stats.logout();
      }
    }
  }
}
