package csbase.console;

import java.util.Arrays;

import tecgraf.javautils.configurationmanager.ConfigurationManager;

/**
 * Superclasse para os clientes console correspondentes a operaes sobre
 * algoritmos.
 *
 * @author Tecgraf
 */
public class CSBaseConsole {

  /**
   * Executa o cliente de importao de uma verso de algoritmo.
   *
   * @param args parmetros da linha de comando
   * @throws Exception em caso de erro na parametrizao.
   */
  public static void main(String[] args) throws Exception {
    if (args.length == 0 || args[0].trim().equalsIgnoreCase("-h")) {
      printHelp();
      // Comando no informado.
      System.exit(ExitCode.FAILURE.getCode());
      return;
    }
    String command = args[0];
    ConsoleOperation operation = ConsoleOperation.getOperationFromName(command);
    if (operation == null) {
      System.out.println("Comando " + command + " no identificado.");
      printHelp();
      // Comando no identificado.
      System.exit(ExitCode.FAILURE.getCode());
      return;
    }

    ConfigurationManager.createInstance();

    String[] opArgs = args.length > 1 ? Arrays.copyOfRange(args, 1, args.length)
      : new String[0];
    AbstractConsoleApp console = null;

    switch (operation) {
      case ADMIN:
        console = new Admin(opArgs);
        break;
      case DIAGNOSTICS:
        console = new Diagnostics(opArgs);
        break;
      case CHECK_BUNDLES:
        console = new CheckBundlesApp(opArgs);
        break;
      case STATISTICS:
        console = new Statistics(opArgs);
        break;
      case ADD_VERSION:
        console = new ImportAlgoVersion(opArgs);
        break;
      case CREATE_ALGORITHM:
        console = new CreateAlgorithm(opArgs);
        break;
      case LIST_VERSIONS:
        console = new ListAlgoVersions(opArgs);
        break;
      case REMOVE_ALGORITHM:
        console = new RemoveAlgorithm(opArgs);
        break;
      case REMOVE_VERSION:
        console = new RemoveAlgoVersion(opArgs);
        break;
      case LIST_CATEGORIES:
        console = new ListAlgoCategories(opArgs);
        break;
      default:
        System.out.println(
          "Tratamento de comando " + command + " ainda no disponvel.");
        // Comando indisponvel.
        System.exit(ExitCode.FAILURE.getCode());
        return;
    }

    if (console.login()) {
      System.out.println("Execuo de comando " + command + "...");
      ExitCode exitCode = console.execute();
      console.logout();
      System.exit(exitCode.getCode());
    }

    // Falha de login
    System.exit(ExitCode.FAILURE.getCode());
  }

  /**
   * Exibe mensagem de ajuda na sada padro.
   */
  private static void printHelp() {
    System.out.println("Comandos disponveis:");
    for (ConsoleOperation op : ConsoleOperation.values()) {
      System.out
        .println( " - " + op.getCommandName() + ": " + op.getCommandDescription());
    }
    System.out.println("\n");
  }

}
