package br.com.petrobras.security.consume.soap.utilities.mounters;

import br.com.petrobras.security.exception.InvalidOperationException;
import br.com.petrobras.security.model.Area;
import br.com.petrobras.security.model.i18n.SupportedLanguage;
import br.com.petrobras.security.server.service.soap.AreaDTO;

import javax.xml.datatype.DatatypeConfigurationException;
import java.util.List;
import java.util.Locale;

/**
 * Montador de rea.
 * 
 * @since 4.3
 *
 */
public class AreaMounter {

	/**
	 * Monta uma rea do modelo a partir de uma rea de transferncia.
	 * 
	 * @param areaDTO rea de transferncia.
	 * @param preferredUserLanguage Idioma preferido do usurio.
	 * @param supportedApplicationLanguages Idiomas suportados pelo sistema.
	 * @return rea do modelo.
	 */
	public static Area mountLocalObject(AreaDTO areaDTO, Locale preferredUserLanguage, List<SupportedLanguage> supportedApplicationLanguages)
    {
    	if ((areaDTO != null) && (((areaDTO.getParentId() == null) || (areaDTO.getParentId().equals(""))) == false)) {
    		Area parent = new Area(areaDTO.getParentId()); 
    		return mountLocalObject(areaDTO, preferredUserLanguage, supportedApplicationLanguages, parent);
    	} else {
    		return mountLocalObject(areaDTO, preferredUserLanguage, supportedApplicationLanguages, null);
    	}
    }

	/**
	 * Monta uma rea do modelo a partir de uma rea de transferncia.
	 * 
	 * @param areaDTO rea de transferncia.
	 * @param preferredUserLanguage Idioma preferido do usurio.
	 * @param supportedApplicationLanguages Idiomas suportados pelo sistema.
	 * @param parent Pai da rea.
	 * @return rea do modelo.
	 */
    public static Area mountLocalObject(AreaDTO areaDTO, Locale preferredUserLanguage, List<SupportedLanguage> supportedApplicationLanguages, Area parent)
    {
        if (areaDTO == null)
        {
            return null;
        }
        else
        {
        	Area area = new Area();

            //Simple.
        	area.setUid(areaDTO.getUid());
        	area.setId(areaDTO.getId());
        	area.setCreationDate(DateMounter.mountLocalObject(areaDTO.getCreationDate()));
        	area.setParent(parent);

            //Complex.
        	area.setApplicationEnvironment(ApplicationEnvironmentMounter.mountLocalObject(areaDTO.getApplicationEnvironment(), preferredUserLanguage));
            
        	if (areaDTO.getChildren() != null)
            {
                for(AreaDTO areaChild : areaDTO.getChildren())
                {
                    area.getChildren().add(AreaMounter.mountLocalObject(areaChild, preferredUserLanguage, supportedApplicationLanguages, area));
                }
            }

            //I18n.
            area.setNameTranslations(I18nStringMounter.mountModelObject(areaDTO.getNameTranslations(), preferredUserLanguage, supportedApplicationLanguages));
            area.setDescriptionTranslations(I18nStringMounter.mountModelObject(areaDTO.getDescriptionTranslations(), preferredUserLanguage, supportedApplicationLanguages));
            
            return area;
        }
    }
    
    /**
     * Monta uma rea de transferncia a partir de uma rea do modelo.
     * 
     * @param area rea do modelo.
     * @param supportedApplicationLanguages Idiomas suportados pelo sistema.
     * @return rea de transferncia.
     */
    public static AreaDTO mountTransferObject(Area area, List<SupportedLanguage> supportedApplicationLanguages)
    {        
         if ((area != null ) && (area.getParent() != null))
         { 
        	 AreaDTO parentDTO = new AreaDTO();
             parentDTO.setId(area.getParent().getId());

             return mountTransferObject (area, supportedApplicationLanguages, parentDTO );
         }
         else
         {
             return mountTransferObject (area, supportedApplicationLanguages, null );
         }
    }

    /**
     * Monta uma rea de transferncia a partir de uma rea do modelo.
     * 
     * @param area rea do modelo.
     * @param supportedApplicationLanguages Idiomas suportados pelo sistema.
     * @param parent Pai da rea.
     * @return rea de transferncia.
     */
    public static AreaDTO mountTransferObject
        (Area area, List<SupportedLanguage> supportedApplicationLanguages, AreaDTO parent) {
        
        if (area == null) {
            return null;
        }
        else {
            /*if (area.getNameTranslations() == null || area.getNameTranslations().getCount() < 1) {
                throw new IllegalArgumentException("A lista de tradues de nome da rea est nula ou vazia.");
            }

            if (area.getDescriptionTranslations() == null || area.getDescriptionTranslations().getCount() < 1) {
                throw new IllegalArgumentException("A lista de tradues de descrio da rea est nula ou vazia.");
            }*/

            AreaDTO areaDTO = new AreaDTO();

            //Simple.
            areaDTO.setUid(area.getUid());
            areaDTO.setId(area.getId());
            
            try {
            	areaDTO.setCreationDate(DateMounter.mountTransferObject(area.getCreationDate()));
            } catch (DatatypeConfigurationException dtce) {
                throw new InvalidOperationException(
                        "No foi possvel converter a data de cadastro da rea para enviar ao servio. "
                                + dtce.getMessage());
            }
            
            if (parent != null) {
            	areaDTO.setParentId(parent.getId());
            } else {
            	areaDTO.setParentId(null);
            }

            //Complex.
            areaDTO.setApplicationEnvironment(ApplicationEnvironmentMounter.mountTransferObject(area.getApplicationEnvironment()));
            
            for (Area areaChild : area.getChildren()) {
            	areaDTO.getChildren().add(AreaMounter.mountTransferObject(areaChild, supportedApplicationLanguages, areaDTO));
            }

            //I18n.
            areaDTO.setNameTranslations(I18nStringMounter.mountTransferObject(area.getNameTranslations(), supportedApplicationLanguages));
            areaDTO.setDescriptionTranslations(I18nStringMounter.mountTransferObject(area.getDescriptionTranslations(), supportedApplicationLanguages));
            
            return areaDTO;
        }
    }

}
