package csbase.rest.adapter.algorithm.v1;

import csbase.logic.AlgorithmExecutionPermission;
import csbase.logic.CategoryAlgorithmsExecutionPermission;
import csbase.logic.User;
import csbase.logic.algorithms.*;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.remote.AlgorithmServiceInterface;
import csbase.remote.ClientRemoteLocator;
import ibase.common.ServiceAdapter;
import ibase.exception.InternalServiceException;
import ibase.rest.api.algorithm.v1.adapter.AlgorithmNotFoundException;
import ibase.rest.api.algorithm.v1.adapter.AlgorithmServiceAdapter;
import ibase.rest.api.algorithm.v1.adapter.VersionNotFoundException;
import ibase.rest.model.algorithm.v1.Algorithm;
import ibase.rest.model.algorithm.v1.AlgorithmConfiguration;
import ibase.rest.model.algorithm.v1.AlgorithmVersion;

import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.Response;
import java.rmi.RemoteException;
import java.util.*;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * Created by mjulia on 08/06/16.
 */
public class CSBaseAlgorithmServiceAdapter implements AlgorithmServiceAdapter {

	private final static String CURRENT_SYSTEM_ID = ".";

	private ParameterMapper mapper = new ParameterMapper();

	@Override
	public void setLocale(Locale locale) {
		ClientRemoteLocator.administrationService.setLocale(locale);
	}

	@Override
	public boolean existsAlgorithm(String algorithmId) {
		AlgorithmServiceInterface service = ClientRemoteLocator.algorithmService;
		return getAlgorithmInfo(service, algorithmId) != null;
	}

	@Override
	public boolean existsAlgorithmVersion(String algorithmId, String versionId) throws AlgorithmNotFoundException {
		AlgorithmServiceInterface service = ClientRemoteLocator.algorithmService;
		try {
			AlgorithmInfo algoInfo = getAlgorithmInfo(service, algorithmId);
			if (algoInfo == null)
				throw new AlgorithmNotFoundException();
			AlgorithmVersionId algoVersionId = AlgorithmVersionId.create(versionId);
			AlgorithmVersionInfo version = algoInfo.getVersionInfo(algoVersionId);
			return version != null;
		} catch (Throwable e) {
			throw new InternalServiceException(e);
		}
	}

	@Override
	public List<Algorithm> getAlgorithms(Predicate<Algorithm> predicate) {
		AlgorithmServiceInterface service = ClientRemoteLocator.algorithmService;
		try {
			AlgorithmInfo[] algoInfos = service.getAllInfo();
			List<Algorithm> algorithms = new ArrayList<Algorithm>();
			csbase.logic.User user = User.getUserByLogin(ServiceAdapter.getCurrenUser());
			Arrays.asList(algoInfos).forEach(i -> {
				if (userHasPermissionToExecuteAlgorithm(user, i)) {
					Algorithm a = buildAlgorithm(i);
					if (predicate == null)
						algorithms.add(a);
					else if (predicate.test(a))
						algorithms.add(a);
				}
			});
			return algorithms;
		} catch (Throwable e) {
			throw new InternalServiceException(e);
		}
	}

	@Override
	public Optional<Algorithm> getAlgorithm(String algorithmId) {
		AlgorithmServiceInterface service = ClientRemoteLocator.algorithmService;
		try {
			Optional<Algorithm> algorithm = Optional.empty();
			AlgorithmInfo algoInfo = getAlgorithmInfo(service, algorithmId);
			if (algoInfo != null) 
				algorithm = Optional.of(buildAlgorithm(algoInfo));
			return algorithm;
		} catch (Throwable e) {
			throw new InternalServiceException(e);
		}
	}

	@Override
	public List<AlgorithmVersion> getAlgorithmVersions(String algorithmId) throws AlgorithmNotFoundException {
		System.out.println("getAlgorithmVersions: " + algorithmId);
		AlgorithmServiceInterface service = ClientRemoteLocator.algorithmService;
		try {
			AlgorithmInfo algoInfo = getAlgorithmInfo(service, algorithmId);
			if (algoInfo == null) 
				throw new AlgorithmNotFoundException();
			List<AlgorithmVersionInfo> versionInfos = algoInfo.getVersions();
			if (versionInfos != null) {
				List<AlgorithmVersion> versions = versionInfos.stream().map(v -> buildAlgorithmVersion(algoInfo, v))
						.collect(Collectors.toList());
				return versions;
			}
		} catch (Throwable e) {
			throw new InternalServiceException(e);
		}
		return null;
	}

	@Override
	public Optional<AlgorithmVersion> getAlgorithmVersion(String algorithmId, String versionId) throws AlgorithmNotFoundException {
		System.out.println("getAlgorithmVersion: " + algorithmId + " - " + versionId);
		AlgorithmServiceInterface service = ClientRemoteLocator.algorithmService;
		AlgorithmVersionId algoVersionId = AlgorithmVersionId.create(versionId);
		try {
			AlgorithmInfo algoInfo = getAlgorithmInfo(service, algorithmId);
			if (algoInfo == null) 
				throw new AlgorithmNotFoundException();
			Optional<AlgorithmVersion> algorithmVersion = Optional.empty();
			AlgorithmVersionInfo version = algoInfo.getVersionInfo(algoVersionId);
			if (version != null) 
				algorithmVersion = Optional.of(buildAlgorithmVersion(algoInfo, version));
			return algorithmVersion;
		} catch (Throwable e) {
			throw new InternalServiceException(e);
		}
	}

	@Override
	public AlgorithmConfiguration getAlgorithmConfiguration(String algorithmId, String versionId) throws AlgorithmNotFoundException, VersionNotFoundException {
		System.out.println("getAlgorithmConfiguration: " + algorithmId + " - " + versionId);
		if (!existsAlgorithm(algorithmId))
			throw new AlgorithmNotFoundException();
		if (!existsAlgorithmVersion(algorithmId, versionId))
			throw new VersionNotFoundException();
		AlgorithmConfiguration algorithmConfig = new AlgorithmConfiguration();
		AlgorithmServiceInterface service = ClientRemoteLocator.algorithmService;
		AlgorithmVersionId algoVersionId = AlgorithmVersionId.create(versionId);
		try {
			AlgorithmConfigurator configurator = service.createAlgorithmConfigurator(algorithmId, algoVersionId);
			if (SimpleAlgorithmConfigurator.class.isInstance(configurator)) {
				SimpleAlgorithmConfigurator simpleConfigurator = SimpleAlgorithmConfigurator.class.cast(configurator);
				List<csbase.logic.algorithms.parameters.ParameterGroup> groups = simpleConfigurator.getGroups();
				algorithmConfig
						.setGroups(groups.stream().map(g -> buildParameterGroup(g)).collect(Collectors.toList()));
				algorithmConfig.setCommand(simpleConfigurator.getCommandBinaryName());
				algorithmConfig.setExecutionType(getExecutionType(simpleConfigurator.getExecutionType()));
			}
		} catch (Throwable e) {
			throw new InternalServiceException(e);
		}
		return algorithmConfig;
	}

	/**
	 * Build a JSON algorithm object model.
	 * 
	 * @param info
	 *            The CSBase algorithm info
	 * @return the JSON object representing the algorithm
	 */
	private Algorithm buildAlgorithm(AlgorithmInfo info) {
		Algorithm algorithm = new Algorithm();
		algorithm.setName(info.getName());
		algorithm.setId(info.getId());
		algorithm.setDescription(info.getDescription());
		String owner = info.getOwner();
		if (owner != null)
			algorithm.setWhoCreated(buildRestUser(owner));
		List<AlgorithmVersionInfo> versions = info.getVersions();
		if (versions != null) {
			algorithm.setVersions(
					versions.stream().map(v -> buildAlgorithmVersion(info, v)).collect(Collectors.toList()));
		}
		return algorithm;
	}

	private ibase.rest.model.algorithm.v1.User buildRestUser(String userId) {
		Client client = ClientBuilder.newClient();
		WebTarget target = client.target(ServiceAdapter.getURI());
		Response response = target.path("users").path(userId).request("application/json;charset=UTF-8")
				.header(HttpHeaders.AUTHORIZATION, ServiceAdapter.getAutheticationHeader()).get();
		if (response.getStatus() != 200) {
			return null;
		}
		ibase.rest.model.algorithm.v1.User restUser = response.readEntity(ibase.rest.model.algorithm.v1.User.class);
		return restUser;
	}

	private ibase.rest.model.algorithm.v1.AlgorithmVersion buildAlgorithmVersion(AlgorithmInfo algoInfo,
			AlgorithmVersionInfo versionInfo) {
		AlgorithmVersion version = new AlgorithmVersion();
		version.setId(versionInfo.getId().toString());
		try {
			AlgorithmServiceInterface service = ClientRemoteLocator.algorithmService;
			AlgorithmConfigurator configurator = service.createAlgorithmConfigurator(algoInfo.getId(), versionInfo.getId());
			version.setDescription(configurator.getDescription());
		} catch (RemoteException e) {
			// Nothing to do -- it is not a remote call
		}
		RequirementsBuilder mapper = new RequirementsBuilder(algoInfo.getId(), versionInfo.getId(),
				ClientRemoteLocator.administrationService.getCurrentLocale());
		version.setRequirements(mapper.requirements);
		return version;
	}

	private ibase.rest.model.algorithm.v1.ParameterGroup buildParameterGroup(
			csbase.logic.algorithms.parameters.ParameterGroup group) {
		ibase.rest.model.algorithm.v1.ParameterGroup parameterGroup = new ibase.rest.model.algorithm.v1.ParameterGroup();
		parameterGroup.setParameters(group.getSimpleParameters().stream().map(p -> mapper.getFactoryFor(p.getType()).buildParameter(p))
				.collect(Collectors.toList()));
		parameterGroup.setId(group.getName());
		parameterGroup.setLabel(group.getLabel());
		parameterGroup.setCollapsable(group.isCollapsible());
		return parameterGroup;
	}

	private ibase.rest.model.algorithm.v1.AlgorithmConfiguration.ExecutionTypeEnum getExecutionType(
			ExecutionType type) {
		switch (type) {
		case SIMPLE:
			return AlgorithmConfiguration.ExecutionTypeEnum.SIMPLE;
		case MULTIPLE:
			return AlgorithmConfiguration.ExecutionTypeEnum.MULTIPLE;
		}
		return null;
	}

	private AlgorithmInfo getAlgorithmInfo(AlgorithmServiceInterface service, String algorithmId)
			throws InternalServiceException {
		try {
			AlgorithmInfo algoInfo = service.getInfo((Object) algorithmId);
			return (algoInfo != null
					&& !userHasPermissionToExecuteAlgorithm(User.getUserByLogin(ServiceAdapter.getCurrenUser()),
							algoInfo)) ? null : algoInfo;
		} catch (Throwable e) {
			throw new InternalServiceException(e);
		}
	}

	private boolean userHasPermissionToExecuteAlgorithm(csbase.logic.User user, AlgorithmInfo algoInfo) {
		try {
			String algorithmName = algoInfo.getName();
			CategorySet allCategories = ClientRemoteLocator.algorithmService
					.getAllCategories();
			List<String> categoriesFullNames = allCategories
					.getAlgorithmCategoriesFullNames(algoInfo);
			return user.isAdmin() || AlgorithmExecutionPermission.checkSystemAndAlgorithmExecPermission(user,
					CURRENT_SYSTEM_ID, algorithmName) || CategoryAlgorithmsExecutionPermission
					.checkSystemAndCategoriesExecPermission(User.getLoggedUser(), CURRENT_SYSTEM_ID,
							categoriesFullNames);
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			return false;
		}
	}
}
