package csbase.rest.adapter.drmaa2.v1.DAO;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import ibase.rest.api.drmaa2.v1.adapter.JobDAO;
import ibase.rest.model.drmaa2.v1.Job;
import ibase.rest.model.drmaa2.v1.JobSession;

/**
 * Created by mjulia on 11/10/16.
 */
public class DefaultJobDAO implements JobDAO{

  /**
   * Mapa de sesses por session name.
   */
  private Map<String, JobSession> jobSessions = new HashMap<>();

  /**
   * Mapa dos jobs por jobId.
   */
  private Map<String, Job> jobs = new HashMap<>();

  /**
   * Jobs em cada sesso por session name.
   */
  private Map<String, List<Job>> jobsBySession = new HashMap<>();

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean containsJob(String jobId) {
    return jobs.containsKey(jobId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Job> findAllJobs() {
    return jobs.values().stream().collect(Collectors.toList());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Job findJobById(String jobId) {
    return jobs.get(jobId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Job> findJobs(String projectId, Long modifiedAfter) {
    List<Job> filteredJobs = jobs.values().stream().filter(j -> {
      if (projectId != null && !j.getProjectId().equals(projectId)) {
        return false;
      }
      if (modifiedAfter != null && LocalDateTime.parse(j.getLastModifiedTime())
        .isBefore(LocalDateTime.ofInstant(Instant.ofEpochMilli(modifiedAfter),
          ZoneId.systemDefault()))) {
        return false;
      }
      return true;
    }).collect(Collectors.toList());
    return filteredJobs;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Job> findJobsBySession(String jobSessionId) {
    return jobsBySession.get(jobSessionId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean insertJob(Job job) {
    jobs.put(job.getJobId(), job);
    List<Job> jobs = jobsBySession.get(job.getSessionId());
    if (jobs==null) {
      jobs = new ArrayList<>();
      jobsBySession.put(job.getSessionId(), jobs);
    }
    jobs.add(job);
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean updateJob(Job job) {
    Job oldJob = jobs.get(job.getJobId());
    jobs.put(job.getJobId(), job);
    List<Job> sessionJobs = jobsBySession.get(job.getSessionId());
    sessionJobs.remove(oldJob);
    sessionJobs.add(job);
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean containsJobSession(String sessionId) {
    return jobSessions.containsKey(sessionId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<JobSession> findAllJobSessions() {
    return jobSessions.values().stream().collect(Collectors.toList());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public JobSession findJobSessionById(String sessionId) {
    return jobSessions.get(sessionId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void insertJobSession(JobSession jobSession) {
    jobSessions.put(jobSession.getSessionName(), jobSession);
  }


}
