package csbase.rest.monitor;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.ConnectException;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.logging.ConsoleHandler;
import java.util.logging.Level;
import java.util.logging.LogManager;
import java.util.logging.Logger;

import javax.ws.rs.ProcessingException;

import Requests.Algorithms.Algorithm;
import Requests.Algorithms.AlgorithmExplorer;
import Requests.Authentication.Authentication;
import Requests.Authentication.InvalidLoginOrPasswordException;
import Requests.Authentication.LoginOrPasswordNotProvidedException;
import Requests.Authentication.Token;
import Requests.Jobs.JobInfo;
import Requests.Jobs.JobInfo.ExitStatusEnum;
import Requests.Jobs.JobServices;
import Requests.Projects.FileExplorer;
import Requests.Projects.PermissionException;
import Requests.Projects.Project;
import Requests.Projects.ProjectExplorer;
import Requests.Projects.ProjectOrFileNotFoundException;

/**
 * Exemplo de submissão programática da execução de um algoritmo em um servidor
 * CSBase.
 *
 * Compatível com a versão 1.4.6 do iBase.
 *
 * @author Tecgraf/PUC-Rio
 */
public class RESTMonitor {

  private static Logger logger;

  private static final String ALGORITHM = "zip";
  private static final String REMOTE_INPUT_FILE_1 = "input1.txt";
  private static final String REMOTE_INPUT_FILE_2 = "input2.txt";
  private static final String REMOTE_OUTPUT_FILE = "output.zip";

	/**
	 * Nome do arquivo com as propriedades para a demo
	 */
	private static String configFileName = "config.properties";

	/*
	 * Assertivas:
	 * - usuário já cadastrado
	 * - projeto previamente criado
	 * - usuário é o dono do projeto
	 * - algoritmo zip cadastrado
	 * - arquivos de entrada disponiveis localmente
	 * - caminho válido para arquivo de saída a ser criado localmente
	 * - alguma maquina remota disponivel para executar
	 */
	public static void main(String[] args) {

    // get the global logger to configure it
    System.setProperty("java.util.logging.SimpleFormatter.format",
      "[%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS] [%4$s] %5$s%6$s%n");
    logger = Logger.getGlobal();
    Arrays.stream(LogManager.getLogManager().getLogger("").getHandlers()).forEach(h -> {
      if (h instanceof ConsoleHandler) {
        h.setLevel(Level.ALL);
      }
    });
    logger.info("-- Inicio da demo");

		Properties props = new Properties();
    InputStream input = null;
    try {
      if (args.length > 0) {
        configFileName = args[0];
        input = new FileInputStream(configFileName);
      }
      else {
        input = RESTMonitor.class.getClassLoader().getResourceAsStream(configFileName);
      }
      props.load(input);
      if (input == null) {
        logger.severe("FALHA: Arquivo de configuração " + configFileName + " inexistente");
        System.exit(1);
      }
      logger.info("Propriedades carregadas de " + configFileName);
    }
    catch (Exception e) {
      logger.log(Level.SEVERE, "FALHA: Erro na leitura do arquivo " + configFileName, e);
      System.exit(1);
    }
    finally {
      if (input != null) {
        try {
          input.close();
        }
        catch (IOException e) {
          //pode ignorar.
        }
      }
		}
    String logLevel = props.getProperty("java.util.logging.ConsoleHandler.level");
    logger.setLevel(Level.parse(logLevel));

		String host = props.getProperty("host");
		if (host==null || host.isEmpty()) {
      logger.severe("FALHA: A propriedade host é obrigatória");
			System.exit(1);
		}
    logger.fine("Endereço do servidor: " + host);
		String username = props.getProperty("username");
    if (username == null || username.isEmpty()) {
      logger.severe("FALHA: A propriedade username é obrigatória");
			System.exit(1);
		}
    logger.fine("Usuário de acesso: " + username);
		String password = props.getProperty("password");
    if (password == null || password.isEmpty()) {
      logger.severe("FALHA: A propriedade password é obrigatória");
			System.exit(1);
		}
		String projectName = username + "/" + props.getProperty("project");
    if (projectName.equals(username + "/" + null) || projectName.equals(username + "/")) {
      logger.severe("FALHA: A propriedade project é obrigatória");
			System.exit(1);
		}
    logger.fine("Projeto: " + projectName);
		String projectFolderPath = props.getProperty("projectFolderPath");
    if (projectFolderPath == null || projectFolderPath.isEmpty()) {
      logger.severe("FALHA: A propriedade projectFolderPath é obrigatória");
			System.exit(1);
		}
    logger.fine("Diretório destino: " + projectFolderPath);
		String toUploadFilePath1 = props.getProperty("input_file_1");
    if (toUploadFilePath1 == null || toUploadFilePath1.isEmpty()) {
      logger.severe("FALHA: A propriedade input_file_1 é obrigatória");
			System.exit(1);
		}
    logger.fine("Primeiro arquivo de entrada: " + toUploadFilePath1);
		String toUploadFilePath2 = props.getProperty("input_file_2");
    if (toUploadFilePath2 == null || toUploadFilePath2.isEmpty()) {
      logger.severe("FALHA: A propriedade input_file_2 é obrigatória");
			System.exit(1);
		}
    logger.fine("Segundo arquivo de entrada: " + toUploadFilePath2);
		String toSaveFilePath = props.getProperty("output_file");
    if (toSaveFilePath == null || toSaveFilePath.isEmpty()) {
      logger.severe("FALHA: A propriedade output_file é obrigatória");
			System.exit(1);
		}

		try {
			/*
			 * Autentica o usuario com login e senha configurados na demo
			 */
			Token token = Authentication.authenticate(host, username, password);


			/*
			 * Procura o projeto com o nome configurado na demo
			 */
			Project project = ProjectExplorer.findProjectByName(host, token, projectName);
			if (project==null) {
        logger.severe("FALHA: Projeto " + projectName + " não encontrado");
				System.exit(1);
			}
      logger.info("Projeto " + projectName + " encontrado");

			/*
			 * Procura o algoritmo com o nome configurado na demo
			 */
			Algorithm algorithm = AlgorithmExplorer.findAlgorithmByName(host, token, ALGORITHM);
			if (algorithm==null) {
        logger.severe("FALHA: Algoritmo " + ALGORITHM + " não encontrado");
				System.exit(1);
			}
      logger.info("Algoritmo " + ALGORITHM + " encontrado");

			/*
			 * Faz o upload dos dados de entrada na raiz do projeto
			 *
			 * Os arquivos que serão necessários para o algoritmo precisam ser enviados para o projeto, pois é de
			 * lá onde o algoritmo consegue ler os arquivos.
			 */
      logger.info("Enviando arquivos que serão entrada do algoritmo " + ALGORITHM);
			FileExplorer.uploadFile(host, token, project, projectFolderPath, toUploadFilePath1, REMOTE_INPUT_FILE_1);
			FileExplorer.uploadFile(host, token, project, projectFolderPath, toUploadFilePath2, REMOTE_INPUT_FILE_2);

			/*
			 * Cria a lista que será utilizada como parâmetro do algoritmo.
			 *
			 * O algoritmo Zip recebe como parâmetro de entrada uma lista de arquivos (que
			 * pode conter somente um arquivo). No exemplo, utilizamos dois arquivos.
			 */
      List<String> inputFiles = new ArrayList<>();
      inputFiles.add(REMOTE_INPUT_FILE_1);
      inputFiles.add(REMOTE_INPUT_FILE_2);

      /*
       * O nome dos parâmetros (aqui no exemplo "ENTRADA" e "SAIDA") são definidos pelo próprio algoritmo, dentro de seu arquivo config.xml
       * Acesse os links abaixo para mais informações sobre o config.xml
       * https://jira.tecgraf.puc-rio.br/confluence/pages/viewpage.action?pageId=36504138
       * http://webserver2.tecgraf.puc-rio.br/ftp_pub/csbase/1.5.5/manualConfiguradorXML.pdf
       */
      Map<String, Object> commandArgs = new HashMap<>();
      commandArgs.put("ENTRADA", inputFiles);
      commandArgs.put("SAIDA", REMOTE_OUTPUT_FILE);

      /*
       * Executa o algoritmo Zip (obtido previamente)
       */
			String jobId = JobServices.submitJob(host, token, project, algorithm, algorithm.getVersions().get(0),
					"demo rest java", 0, false, new String[0], commandArgs);
			if (jobId==null) {
        logger.severe("FALHA: FALHA: na submissão do algoritmo " + ALGORITHM);
				System.exit(1);
			}
      logger.info("Execução de algoritmo " + ALGORITHM + " submetida com sucesso");

			/*
			 * Obtém as informações do job criado para a execução do algoritmo.
			 */
			JobInfo jinfo = JobServices.getJobInfo(host, token, jobId);
			if (jinfo==null) {
        logger.severe("FALHA: Job " + jobId + " não foi encontrado");
				System.exit(1);
			}
      logger.info("Job " + jobId + " encontrado");

      logger.info("Aguardando término da execução do algoritmo: " + ALGORITHM);
      /*
       * Aguarda o job terminar sua execução
       */
			JobServices.awaitJobEnd(host, token, jinfo);

      jinfo = JobServices.getJobInfo(host, token, jobId);
      if (jinfo.getExitStatus() != ExitStatusEnum.SUCCESS) {
        logger.severe("FALHA: FALHA: na execução do algoritmo " + ALGORITHM + ": " + jinfo.getExitStatus());
        System.exit(1);
      } else {
        logger.info("Algoritmo " + ALGORITHM + " executado com sucesso.");
      }

      /*
       * Faz download do arquivo (quando o arquivo é pequeno e somente
       * texto)
       *
       * obs: como o arquivo de saída é um binário (zip), esse exemplo está
       * sendo feito com um dos arquivos de entrada que foi enviado para o
       * servidor, mas basta alterar o nome do arquivo na chamada para obter o
       * arquivo pretendido.
       */
      logger.info("Obtendo arquivo texto.");
      String textResult = FileExplorer.downloadTextFile(host, token, project, projectFolderPath, REMOTE_INPUT_FILE_1);

      logger.fine("Resultado em " + REMOTE_INPUT_FILE_1 + ":\n" + textResult);

      /*
       * Faz download de arquivo. (quando o arquivo é pequeno e qualquer formato)
       *
       * obs: neste caso fazemos download do arquivo de saída.
       */
      logger.info("Obtendo arquivos binários.");
      InputStream fileStream = FileExplorer.downloadBinaryFile(host, token, project, projectFolderPath, REMOTE_OUTPUT_FILE);
      if (fileStream == null) {
        logger.severe("FALHA: Erro ao fazer download de arquivo binário.");
        System.exit(1);
        return; //desativa warnings de nullpointer do fileStream.
      }

      try {
        long fileSize = Files.copy(fileStream, new File(toSaveFilePath).toPath(), StandardCopyOption.REPLACE_EXISTING);
        fileStream.close();
        logger.fine("Resultado em " + REMOTE_OUTPUT_FILE + ": " + fileSize + " bytes ");
      }
      catch (IOException e) {
        logger.log(Level.WARNING, "Não foi possível gerar arquivo binário obtido via download.", e);
      }

			/*
			 * Faz download do arquivo de saída (quando o arquivo é grande)
			 *
			 * obs: colocamos um prefixo no nome do arquivo a ser salvo para diferenciar do arquivo obtido no exemplo acima.
			 */
      FileExplorer.downloadLargeFile(host, token, project, projectFolderPath, REMOTE_OUTPUT_FILE, "OUTRO" + toSaveFilePath);

      logger.fine("Resultado em " + "OUTRO" + toSaveFilePath + ": " + new File("OUTRO" + toSaveFilePath).length()
        + " bytes ");

      logger.info("Removendo arquivos enviados e arquivo de saída.");
      logger.fine("Removendo REMOTE_INPUT_FILE_1: " + REMOTE_INPUT_FILE_1);
      FileExplorer.removeFile(host, token, project, projectFolderPath, REMOTE_INPUT_FILE_1);
      logger.fine("Removendo REMOTE_INPUT_FILE_2: " + REMOTE_INPUT_FILE_2);
      FileExplorer.removeFile(host, token, project, projectFolderPath, REMOTE_INPUT_FILE_2);
      logger.fine("Removendo REMOTE_OUTPUT_FILE: " + REMOTE_OUTPUT_FILE);
      FileExplorer.removeFile(host, token, project, projectFolderPath, REMOTE_OUTPUT_FILE);

      logger.fine("Arquivos removidos com sucesso.");

      logger.info("SUCESSO: Servidor funcionando corretamente.");

      System.exit(0);

		} catch (LoginOrPasswordNotProvidedException e) {
      logger.log(Level.SEVERE, "FALHA: Usuário ou senha não informados corretamente.", e);
		} catch (InvalidLoginOrPasswordException e) {
      logger.log(Level.SEVERE, "FALHA: Erro de autenticação.", e);
		} catch (ProjectOrFileNotFoundException e) {
      logger.log(Level.SEVERE, "FALHA: Projeto ou arquivo não encontrado.", e);
		} catch (PermissionException e) {
      logger.log(Level.SEVERE, "FALHA: Usuário não possui permissão de acesso ao projeto ou arquivo.", e);
    } catch(ProcessingException e) {
      String errorMsg = "Erro no acesso ao servidor.";
      if (e.getCause() != null) {
        if (e.getCause() instanceof ConnectException) {
          errorMsg = "Servidor fora do ar.";
        } else if (e.getCause() instanceof FileNotFoundException) {
          errorMsg = "Arquivo de entrada inexistente.";
        }
      }
      logger.log(Level.SEVERE, errorMsg, e);
    }

    // Se chegou aqui, caiu em alguma exceção e aí deu erro.
    System.exit(1);
	}
}
