package csbase.rest.adapter.project.v1;

import java.io.File;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.function.Predicate;

import csbase.exception.InfoException;
import csbase.exception.ServiceFailureException;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.CommonProjectInfo;
import csbase.logic.ProjectAdminInfo;
import csbase.logic.ProjectPermissions;
import csbase.logic.User;
import csbase.logic.UserProjectInfo;
import csbase.remote.AdministrationServiceInterface;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;
import ibase.common.NotFoundException;
import ibase.exception.InternalServiceException;
import ibase.exception.InvalidParameterException;
import ibase.rest.api.project.v1.adapter.Project;
import ibase.rest.api.project.v1.adapter.ProjectFile;
import ibase.rest.api.project.v1.adapter.ProjectInfo;
import ibase.rest.api.project.v1.adapter.ProjectService;
import ibase.rest.api.project.v1.adapter.SharingType;

/**
 * Implementação do adaptador CSBase para o serviço de projetos.
 *
 * @author Tecgraf/PUC-Rio
 */
public class CSBaseProjectServiceImpl implements ProjectService {

    /**
     * {@inheritDoc}
     */
    @Override
    public Project createProject(String name, String userId, String description,
                                 String type, SharingType sharingType) {
        ProjectServiceInterface service = ClientRemoteLocator.projectService;
        CommonProjectInfo projectInfo = new CommonProjectInfo();
        projectInfo.userId = userId;
        projectInfo.description = description;
        if (type != null) {
            projectInfo.setAttribute(CSBaseProject.PROJECT_TYPE, type);
        }
        projectInfo.name = name;
        try {
            CommonClientProject project = service.createProject(projectInfo);
            if (sharingType != null) {
                setCSBaseSharingType(project.getId(), sharingType, project.getUsersRO(), project.getUsersRW());
            }
            return new CSBaseProject(project);
        } catch (InfoException e) {
            throw new InvalidParameterException(e.getMessage());
        } catch (Throwable e) {
            throw new InternalServiceException(e);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Project updateProject(String projectId, String description,
                                 String type, SharingType sharingType) {
        try {
            ProjectServiceInterface service = ClientRemoteLocator.projectService;
            CommonClientProject project = service.openProject(projectId, false);
            if (description != null) {
                project.setDescription(description);
                project.modify();
            }
            if (type != null) {
                project.setAttribute(CSBaseProject.PROJECT_TYPE, type);
                project.modify();
            }
            if (sharingType != null) {
                Set<Object> usersRO = project.getUsersRO();
                Set<Object> usersRW = project.getUsersRW();
                setCSBaseSharingType(project.getId(), sharingType, usersRO, usersRW);
            }
            return new CSBaseProject(project);
        } catch (InfoException e) {
            throw new InvalidParameterException(e.getMessage());
        } catch (ServiceFailureException e) {
            throw new NotFoundException(e.getMessage());
        } catch (Throwable e) {
            e.printStackTrace();
            throw new InternalServiceException(e);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Project getProject(String projectId) {
        ProjectServiceInterface service = ClientRemoteLocator.projectService;
        try {
            CommonClientProject project = service.openProject(projectId, false);
            return new CSBaseProject(project);
        } catch (InfoException e) {
            throw new InvalidParameterException(e.getMessage());
        } catch (ServiceFailureException e) {
            throw new NotFoundException(e.getMessage());
        } catch (Throwable e) {
            throw new InternalServiceException(e);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setLocale(Locale locale) {
        ClientRemoteLocator.administrationService.setLocale(locale);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ProjectFile getProjectFile(String projectId, String fileId) {
        ProjectServiceInterface service = ClientRemoteLocator.projectService;
        ClientProjectFile file;
        try {
            file = service.getChild(projectId, new String[]{fileId});
            return new CSBaseProjectFile(file);
        } catch (InfoException e) {
            throw new InvalidParameterException(e.getMessage());
        } catch (ServiceFailureException e) {
            throw new NotFoundException(e.getMessage());
        } catch (Throwable e) {
            throw new InternalServiceException(e);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<ProjectInfo> getProjectsSharedWith(String userId) {
        ProjectServiceInterface service = ClientRemoteLocator.projectService;
        try {
            List<UserProjectInfo> infos = service.getProjectsSharedWithUser(userId);
            return filter(infos);
        } catch (InfoException e) {
            throw new InvalidParameterException(e.getMessage());
        } catch (ServiceFailureException e) {
            throw new NotFoundException(e.getMessage());
        } catch (Throwable e) {
            throw new InternalServiceException(e);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<ProjectInfo> getAllProjects(String userId) {
        ProjectServiceInterface projectService = ClientRemoteLocator.projectService;
        AdministrationServiceInterface adminService =
                ClientRemoteLocator.administrationService;
        try {
            List<User> users = adminService.getAllUsers();
            List<UserProjectInfo> allProjects = new ArrayList<UserProjectInfo>();
            for (User user : users) {
                allProjects.addAll(projectService.getProjectsFromUser(user.getId()));
            }
            return filter(allProjects);
        } catch (InfoException e) {
            throw new InvalidParameterException(e.getMessage());
        } catch (ServiceFailureException e) {
            throw new NotFoundException(e.getMessage());
        } catch (Throwable e) {
            throw new InternalServiceException(e);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<ProjectInfo> getProjectsCreatedBy(String userId) {
        ProjectServiceInterface service = ClientRemoteLocator.projectService;
        try {
            List<UserProjectInfo> infos = service.getProjectsFromUser(userId);
            return filter(infos);
        } catch (InfoException e) {
            throw new InvalidParameterException(e.getMessage());
        } catch (ServiceFailureException e) {
            throw new NotFoundException(e.getMessage());
        } catch (Throwable e) {
            throw new InternalServiceException(e);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void removeProject(String projectId) {
        ProjectServiceInterface service = ClientRemoteLocator.projectService;
        try {
            service.removeProject(projectId);
        } catch (InfoException e) {
            throw new InvalidParameterException(e.getMessage());
        } catch (ServiceFailureException e) {
            throw new NotFoundException(e.getMessage());
        } catch (Throwable e) {
            throw new InternalServiceException(e);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean existsProjectFile(String projectId, String fileId) {
        ProjectServiceInterface service = ClientRemoteLocator.projectService;
        try {
            String[] path = fileId.split(File.separator);
            return service.existsFile(projectId, path);
        } catch (InfoException e) {
            throw new InvalidParameterException(e.getMessage());
        } catch (ServiceFailureException e) {
            throw new NotFoundException(e.getMessage());
        } catch (Throwable e) {
            throw new InternalServiceException(e);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void copyFile(String sourceProjectId, String sourceFileId,
                         String targetProjectId, String targetFileId) {
        ProjectServiceInterface service = ClientRemoteLocator.projectService;
        try {
            String[] sourceFilePath = sourceFileId.split(File.separator);
            String[] targetDirPath = targetFileId.split(File.separator);
            service.copyFile(sourceProjectId, sourceFilePath, targetProjectId,
                    targetDirPath);
        } catch (InfoException e) {
            throw new InvalidParameterException(e.getMessage());
        } catch (Throwable e) {
            throw new InternalServiceException(e);

        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void moveFile(String sourceProjectId, String sourceFileId,
                         String targetProjectId, String targetFileId) {
        ProjectServiceInterface service = ClientRemoteLocator.projectService;
        try {
            String[] sourceFilePath = sourceFileId.split(File.separator);
            String[] targetDirPath = targetFileId.split(File.separator);
            service.moveFile(sourceProjectId, sourceFilePath, targetProjectId,
                    targetDirPath);
        } catch (InfoException e) {
            throw new InvalidParameterException(e.getMessage());
        } catch (Throwable e) {
            throw new InternalServiceException(e);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ProjectFile renameFile(String projectId, String fileId, String name) {
        ProjectServiceInterface service = ClientRemoteLocator.projectService;
        try {
            String[] path = fileId.split(File.separator);
            service.renameFile(projectId, path, name);
            path[path.length - 1] = name;
            ClientProjectFile newFile = service.getChild(projectId, path);
            return new CSBaseProjectFile(newFile);
        } catch (ServiceFailureException e) {
            throw new NotFoundException(e.getMessage());
        } catch (InfoException e) {
            throw new InvalidParameterException(e.getMessage());
        } catch (Throwable e) {
            throw new InternalServiceException(e);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ProjectFile createFolder(String projectId, String fileId, String folderName) {
        ProjectServiceInterface service = ClientRemoteLocator.projectService;
        try {
            ibase.rest.api.project.v1.adapter.ProjectFile targetFolder = getProjectFile(
                    projectId, fileId);
            String folderPath = String.join(File.separator, fileId, folderName);
            service.createDirectory(projectId, folderPath.split(File.separator));
            ClientProjectFile child = service.getChild(projectId, folderPath.split(File.separator));
            return new CSBaseProjectFile(child);
        } catch (InfoException e) {
            throw new InvalidParameterException(e.getMessage());
        } catch (Throwable e) {
            throw new InternalServiceException(e);
        }
    }

    /////////////////////
    // Auxiliar Methods
    /////////////////////

    @SuppressWarnings("javadoc")
    private List<ProjectInfo> filter(List<UserProjectInfo> infos) {
        List<ProjectInfo> projectInfos = new ArrayList<ProjectInfo>();
        infos.stream().filter(isAvailable()).forEach(i -> projectInfos.add(
                new CSBaseProjectInfo(i)));
        return projectInfos;
    }

    @SuppressWarnings("javadoc")
    private static Predicate<UserProjectInfo> isAvailable() {
        return new Predicate<UserProjectInfo>() {
            @Override
            public boolean test(UserProjectInfo info) {
                ProjectServiceInterface service = ClientRemoteLocator.projectService;
                ProjectAdminInfo projectAdminInfo;
                try {
                    projectAdminInfo = service.getProjectAdminInfo(info.getProjectId());
                    return !(projectAdminInfo != null && (projectAdminInfo.isLocked()
                            || projectAdminInfo.isWaitingAreaFree()));
                } catch (RemoteException e) {
                    return false;
                }
            }
        };
    }

    @SuppressWarnings("javadoc")
    private void setCSBaseSharingType(Object projectId, SharingType type, Set<Object> usersRO, Set<Object> usersRW) {
        ProjectServiceInterface service = ClientRemoteLocator.projectService;
        try {
            switch (type) {
                case PRIVATE:
                    service.updateUsers(projectId, ProjectPermissions.SharingType.PRIVATE,
                            null, null);
                    break;
                case PUBLIC_RO:
                    service.updateUsers(projectId, ProjectPermissions.SharingType.ALL_RO,
                            null, null);
                    break;
                case PUBLIC_RW:
                    service.updateUsers(projectId, ProjectPermissions.SharingType.ALL_RW,
                            null, null);
                    break;
                case SELECTIVE:
                    service.updateUsers(projectId, ProjectPermissions.SharingType.PARTIAL, usersRO,
                            usersRW);
                    break;
            }
        } catch (RemoteException e) {
        }
    }

}