package csbase.rest.adapter.project.v1;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.stream.Stream;

import csbase.exception.InfoException;
import csbase.exception.PermissionException;
import csbase.exception.project.FileLockedException;
import csbase.logic.ClientProjectFile;
import csbase.logic.NoHiddenFileFilter;
import csbase.logic.ProjectFileType;
import csbase.logic.ProjectFileTypeInfo;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;
import ibase.common.NotFoundException;
import ibase.exception.InternalServiceException;
import ibase.exception.InvalidParameterException;
import ibase.rest.api.project.v1.adapter.ProjectFile;
import tecgraf.javautils.core.io.FileUtils;

/**
 * Implementao do Gateway para um arquivo/pasta de projeto no CSBase.
 *
 * @author Tecgraf/PUC-Rio
 */
public class CSBaseProjectFile implements ProjectFile {

  /**
   * O arquivo/pasta no CSBase
   */
  private ClientProjectFile file;

  /**
   * Construtor.
   * 
   * @param file o arquivo/pasta no CSBase
   */
  CSBaseProjectFile(ClientProjectFile file) {
    this.file = file;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getId() {
    try {
      return FileUtils.joinPath("/", file.getPath());
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getParentId() {
    try {
      String[] path = file.getPath();
      if (path.length == 0) {
        return null;
      }
      String[] parentPath = Arrays.copyOf(path, path.length - 1);
      return FileUtils.joinPath("/", parentPath);
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String whoCreated() {
    try {
      return (String) file.whoCreated();
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Long getCreationDate() {
    try {
      return file.getCreationDate();
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getDescription() {
    try {
      return file.getDescription();
    }
    catch (InfoException e) {
      throw new NotFoundException(e);
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Boolean isDirectory() {
    try {
      return file.isDirectory();
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getName() {
    try {
      return file.getName();
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Boolean isUnderConstruction() {
    try {
      return file.isUnderConstruction();
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Boolean isLocked() {
    try {
      return file.isLocked();
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Long getModificationDate() {
    try {
      return file.getModificationDate();
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getNumberOfChildren(boolean showHiddenFiles) {
    try {
      return getChildren(showHiddenFiles).length;
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getMimeType() {
    ProjectServiceInterface projectService = ClientRemoteLocator.projectService;
    ProjectFileTypeInfo fileTypeInfo;
    try {
      fileTypeInfo = projectService.getFileType(file.getType());
      return fileTypeInfo.getMimeType();
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ProjectFile[] getChildren(boolean showHiddenFiles) {
    try {
      NoHiddenFileFilter noDotFileFilter = NoHiddenFileFilter.getInstance();
      List<ProjectFile> children = new ArrayList<ProjectFile>();
      Stream.of(file.getChildren()).filter(c -> noDotFileFilter.accept(c) || showHiddenFiles).forEach(c -> children.add(
        new CSBaseProjectFile(c)));
      return children.toArray(new ProjectFile[0]);
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public InputStream getInputStream() throws IOException {
    return file.getInputStream();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public OutputStream getOutputStream() throws IOException {
    return file.getOutputStream();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ProjectFile getChild(String fileName) {
    try {
      ClientProjectFile child = file.getChild(fileName);
      if (child == null) {
        throw new InternalServiceException("The file " + fileName + " was not found in path " + file.getStringPath());

      }
      return new CSBaseProjectFile(child);
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean containsChild(String fileName) {
    try {
      return file.getChild(fileName) != null;
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ProjectFile createFile(String fileName, Locale locale) {
    try {
      ProjectFileType.loadFileTypes(locale);
      ProjectFileType fileType = ProjectFileType.getProjectFileTypeFromExtension(FileUtils.getFileExtension(fileName),
        false);
      String fileTypeExtension = (fileType != null) ? fileType.getCode() : ProjectFileType.UNKNOWN;
      file.createFile(fileName, fileTypeExtension);
      ClientProjectFile newFile = file.getChild(fileName);
      if (newFile == null) {
        throw new InternalServiceException("The file " + fileName + " was not successfully created in path " + file
          .getStringPath());
      }
      return new CSBaseProjectFile(newFile);
    }
    catch (PermissionException e) {
      throw new ibase.exception.PermissionException(e.getMessage());
    }
    catch (InfoException e) {
      throw new InvalidParameterException(e.getMessage());
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void open(boolean readOnly) {
    try {
      file.open(readOnly);
    }
    catch (PermissionException e) {
      throw new ibase.exception.PermissionException(e.getMessage());
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(byte[] array, int i, int chunck_size, Long rangeStart) throws IOException {
    try {
      file.write(array, i, chunck_size, rangeStart);
    }
    catch (PermissionException e) {
      throw new ibase.exception.PermissionException(e.getMessage());
    }
    catch (FileLockedException e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void rename(String fileName) {
    try {
      ProjectFileType.loadFileTypes(Locale.getDefault());
      ProjectFileType fileType = ProjectFileType.getProjectFileTypeFromExtension(FileUtils.getFileExtension(fileName),
        false);
      String fileTypeExtension = (fileType != null) ? fileType.getCode() : ProjectFileType.UNKNOWN;
      file.rename(fileName);
      file.setName(fileName);
      file.changeType(fileTypeExtension);
    }
    catch (PermissionException e) {
      throw new ibase.exception.PermissionException(e.getMessage());
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void remove() {
    try {
      file.remove();
    }
    catch (PermissionException e) {
      throw new ibase.exception.PermissionException(e.getMessage());
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long size() {
    try {
      if (file.isDirectory()) {
        return 0;
      }
      else {
        return file.size();
      }
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void close(boolean force) throws IOException {
    try {
      file.close(force);
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }
}
