/*
 * $Id$
 */
package csbase.server.services.schedulerservice;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.Vector;

import csbase.exception.PermissionException;
import csbase.logic.CommandInfo;
import csbase.logic.FailureFinalizationType;
import csbase.logic.Priority;
import csbase.logic.SGASet;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.server.Server;
import csbase.server.Service;
import csbase.server.services.sgaservice.ProjectNotFoundException;
import csbase.server.services.sgaservice.SGANotAvailableException;
import csbase.server.services.sgaservice.SGAService;

/**
 * Classe responsvel por submeter os comandos ao <code>SGAService</code>.
 * 
 * Para submeter um comando para um servidor, o <code>Scheduler</code> considera
 * os seguintes critrios: - peso do algoritmo (CPU e memria); -
 * disponibilidade da mquina; - permisso do usurio;
 * 
 * Na configurao da execuo de um comando, o usurio escolhe manualmente os
 * servidores para execuo ou deixa que o <code>Scheduler</code> tome a deciso
 * de onde o comando ser executado.
 */
public class Scheduler {
  /** Thread do escalonador */
  private Thread schedulerThread;
  /** Indica o trmino do processamento da fila de comandos. */
  private boolean exitSchedulerThread;
  /** Fila que controla a prioridade dos processos */
  private final PriorityQueue commandQueue;
  /** Referncia para o servio. */
  private final SchedulerService service;
  /** SGAService */
  private final SGAService sgaService;
  /** Mapeamento dos sgas que acabaram de receber comandos a uma data */
  private final HashMap<String, Long> recentUsedServers;
  /**
   * Tempo que o SGA fica de quarentena depois que recebe um comando. O valor
   * default  zero. Neste caso no h quarentena.
   */
  private final int sgaQuarantine;
  /**
   * Intervalo de processamento da fila (milisegundos). O valor default  duas
   * vezes o intervalo de atualizao das informaes do SGA.
   */
  private final long queueProcessingInterval;

  /** Objeto que executa a poltica de escalonamento ativa */
  private SchedulerPolicyInterface schedPolicyInterface;

  /**
   * Identificador da poltica de escalonamento que deve ser empregada na
   * escolha dos melhores SGAs ativos
   */
  private final String schedPolicyName;

  /** Controlador de recursos dos sgas. */
  private SchedulerResourcesController schedulerResourcesController;

  /**
   * Constri a representao do Scheduler.
   * 
   * @param commandQueue A fila de comandos a ser processada.
   * @param queueProcessingInterval Intervalo de processamento da fila.
   * @param sgaQuarantine Tempo de quarentena dos SGAs.
   * @param schedPolicyName Poltica de escalonamento ativa.
   * @param resourcesControl Indica se deve fazer o controle de recursos de cada
   *        sga.
   * @param backupFilePath Caminho do arquivo de backup dos recursos de cada
   *        sga.
   */
  protected Scheduler(PriorityQueue commandQueue, long queueProcessingInterval,
    int sgaQuarantine, String schedPolicyName, boolean resourcesControl,
    String backupFilePath) {
    this.commandQueue = commandQueue;
    this.queueProcessingInterval = queueProcessingInterval;
    this.sgaQuarantine = sgaQuarantine;
    this.schedPolicyName = schedPolicyName;
    this.sgaService = SGAService.getInstance();
    this.service = SchedulerService.getInstance();
    this.recentUsedServers = new HashMap<String, Long>();
    this.schedulerResourcesController =
      new SchedulerResourcesController(resourcesControl, backupFilePath);
  }

  /**
   * Cria a thread de processamento da fila de comandos. A thread  acordada a
   * cada novo comando submetido ou no intervalo de
   * <code>queueProcessingInterval</code>. Para cada comando, escolhe o servidor
   * para execuo e submete o comando no servidor escolhido. Coloca o servidor
   * em quarentena, depois que ele  escolhido. Caso no encontre servidor
   * disponvel para executar, o comando permanece na fila.
   */
  protected void startScheduler() {
    loadSchedulerPolicy();

    schedulerThread = new Thread(new Runnable() {
      @Override
      public void run() {
        while (!exitSchedulerThread) {
          Server.logFineMessage("Incio do processamento, se houver comandos");

          CommandInfo[] cmds = commandQueue.getAndWaitForCommands();
          List<String> sgasNames = null;
          sgasNames = sgaService.getAllSGANames();
          clearQuarantine();

          int numOfCommandsToSchedule = cmds.length;
          try {
            for (int i = 0; i < cmds.length && !exitSchedulerThread; i++) {
              CommandInfo cmd = cmds[i];

              // Indica qual usurio est executando o comando.
              Service.setUserId(cmd.getUserId());

              List<String> serversChoosed = chooseServers(cmd, sgasNames);
              if (serversChoosed == null) {
                if (cmd.getPriority() == Priority.ROOT) {
                  // Retira da lista
                  commandQueue.remove(cmd.getId());
                  numOfCommandsToSchedule--;
                  Server
                    .logWarningMessage("Falha ao disparar comando com prioridade "
                      + "ROOT. No foi encontrado um SGA disponvel para atender "
                      + "a plataforma "
                      + cmd.getPlatformFilter()
                      + " do algoritmo usado no comando: " + cmd.getId());
                  // Notifica que no disparou!
                  service.failCommand(cmd,
                    FailureFinalizationType.NO_SGA_AVAILABLE_TO_ROOT_COMMAND);
                }
                continue;
              }

              // Para simples, s h um servidor escolhido.
              String serverChoosed = serversChoosed.get(0);
              Server.logFineMessage("Servidor aprovado: " + serverChoosed);

              // Submete o comando!
              try {
                cmd.setSGAName(serverChoosed);
                schedulerResourcesController.makeReservation(cmd);
                /* Submete o comando para execuo */
                //TODO mjulia REVISAR
                // Estou sincronizando na fila e testando se j houve um cancelamento que removeu.
                // Isso porque o comando por ter sido removido da fila aps o array
                // de comandos ser recuperado aqui.
                synchronized (commandQueue) {
                  if (commandQueue.getCommand(cmd.getId()) == null) {
                    Server
                      .logWarningMessage("O comando "
                        + cmd.getId()
                        + " no vai ser disparado porque j foi removido da fila");
                  }
                  else {
                    sgaService.executeCommand(cmd);
                    commandQueue.remove(cmd.getId());
                    // Coloca os servidores na tabela de recm utilizado
                    for (String server : serversChoosed) {
                      recentUsedServers.put(server, System.currentTimeMillis());
                    }
                  }
                }
                numOfCommandsToSchedule--;
              }
              catch (SGANotAvailableException e) {
                Server.logSevereMessage(String.format(
                  "Erro ao submeter o comando %s - %s.", cmd.getId(), cmd
                    .getTip()), e);
                // Retira da lista
                commandQueue.remove(cmd.getId());
                numOfCommandsToSchedule--;
                // Notifica que no disparou!
                service.failCommand(cmd,
                  FailureFinalizationType.SGA_IS_NOT_AVAILABLE);
              }
              catch (ProjectNotFoundException e) {
                Server.logSevereMessage(String.format(
                  "Erro ao submeter o comando %s - %s.", cmd.getId(), cmd
                    .getTip()), e);
                // Retira da lista
                commandQueue.remove(cmd.getId());
                numOfCommandsToSchedule--;
                // Notifica que no disparou!
                service.failCommand(cmd,
                  FailureFinalizationType.PROJECT_NOT_FOUND);
              }
              catch (PermissionException e) {
                Server.logSevereMessage(String.format(
                  "Erro ao submeter o comando %s - %s.", cmd.getId(), cmd
                    .getTip()), e);
                // Retira da lista
                commandQueue.remove(cmd.getId());
                numOfCommandsToSchedule--;
                // Notifica que no disparou!
                service
                  .failCommand(
                    cmd,
                    FailureFinalizationType.USER_WITHOUT_PERMISSION_FOR_EXECUTION);
              }
              catch (Exception e) {
                Server.logSevereMessage(String.format(
                  "Erro ao submeter o comando %s - %s.", cmd.getId(), cmd
                    .getTip()), e);
                // Retira da lista
                commandQueue.remove(cmd.getId());
                numOfCommandsToSchedule--;
                // Notifica que no disparou!
                FailureFinalizationType.UNKNOWN.changeDescription(e
                  .getMessage());
                service.failCommand(cmd, FailureFinalizationType.UNKNOWN);
              }
            }
          }
          finally {
            Service.setUserId(null);
          }

          // Espera um tempo para reprocessar os elementos que ainda esto
          // na fila. Mas, nesse meio tempo podem ter entrado outros processos.
          if ((commandQueue.size() > 0)
            && (commandQueue.size() == numOfCommandsToSchedule)) {
            Server
              .logFineMessage("Ainda h comandos para processar. Fila em sleep.");
            commandQueue.sleep(queueProcessingInterval);
          }
        }
      }
    });
    schedulerThread.setName(this.getClass().getSimpleName() + "::"
      + "SchedulerThread");
    schedulerThread.start();
  }

  /**
   * Interrompe a execuo da <code>Thread</code>.
   */
  protected void stopScheduler() {
    this.exitSchedulerThread = true;
  }

  /**
   * Instancia um objeto contendo a poltica de escalonamento ativa.
   * 
   * @return Objeto com a poltica de escalonamento ativa.
   */
  private SchedulerPolicyInterface loadSchedulerPolicy() {
    try {
      Class<?> c = Class.forName(schedPolicyName);
      schedPolicyInterface = (SchedulerPolicyInterface) c.newInstance();
    }
    catch (Exception e) {
      schedPolicyInterface = new PercCPUPolicy();
      Server.logSevereMessage(String.format(
        "Erro ao ativar a poltica de escalonamento %s.", schedPolicyName), e);
    }

    Server.logInfoMessage("Classe de escalonamento instanciada: "
      + schedPolicyInterface.getClass().getName() + ".java.");

    return schedPolicyInterface;
  }

  /**
   * Escolhe, dentre os servidores disponveis, um conjunto para processar o
   * comando. Quando o comando  simples, retornamos uma lista com apenas um
   * servidor. XXX - Ainda temos o caso de retornar mais de um sga?
   * 
   * @param command Comando para execuo
   * @param serversNames Lista com todos os servidores disponveis no servidor.
   * 
   * @return Uma lista com os servidores escolhidos para execuo do comando.
   */
  private List<String> chooseServers(CommandInfo command,
    List<String> serversNames) {
    try {
      AlgorithmConfigurator configurator = command.getConfigurator();
      // Atualiza informaes sobre o algoritmo.
      configurator.updateAlgorithmVersion();
      Set<String> platformsSet = configurator.getPlatforms();
      String platformRestriction = command.getPlatformFilter();
      Vector<String> platforms = new Vector<String>();
      if (platformRestriction == null) {
        platforms.addAll(platformsSet);
      }
      else {
        if (platformsSet.contains(platformRestriction)) {
          platforms.add(platformRestriction);
        }
        else {
          return null;
        }
      }
      float cpuAmount = configurator.getCpuAmount();
      float memoryAmount = configurator.getMemoryAmount();
      Set<String> requirements = configurator.getRequirements();

      Server.logFineMessage("Processando o comando: " + command);
      Server.logFineMessage("Tipo do comando: "
        + configurator.getExecutionType());
      Server
        .logFineMessage("Unidade de processamento necessria: " + cpuAmount);
      Server.logFineMessage("Memria necessria: " + memoryAmount);
      for (String requirement : requirements) {
        Server.logFineMessage("Requisito necessrio: " + requirement);
      }
      if (command.isAutomatic()) {
        Server.logFineMessage("Seleo automtica de servidores");
      }
      else {
        Server.logFineMessage("Seleo manual. Servidor escolhido: "
          + command.getSGAName());
      }

      // Escolhe o servidor de acordor com o seu tipo do configurador.
      List<String> serversChoosed = null;
      // Simples e mltipla
      String serverChoosed =
        chooseServerForSimpleCommand(command, serversNames, platforms,
          cpuAmount, memoryAmount, requirements);
      if (serverChoosed != null) {
        serversChoosed = new ArrayList<String>();
        serversChoosed.add(serverChoosed);
      }
      return serversChoosed;
    }
    catch (Exception e) {
      Server.logSevereMessage(String.format(
        "Erro ao processar o comando %s - %s.", command.getId(), command
          .getTip()), e);
      return null;
    }
  }

  /**
   * Obtm o sga para execuo do comando simples.
   * 
   * @param command Comando para execuo.
   * @param servers Lista ordenada de servidores.
   * @param platforms Lista das plataformas em que o algoritmo pode ser
   *        executado.
   * @param cpuAmount Unidade de processamento do algoritmo, ou seja, quanto de
   *        CPU o algoritmo precisa.
   * @param memoryAmount Quantidade de memria (em Mb) necessria para executar
   *        o comando.
   * @param requirements Requisitos necessrios para executar o comando.
   * @return sga escolhido para submisso do comando.
   */
  private String chooseServerForSimpleCommand(CommandInfo command,
    List<String> servers, Vector<String> platforms, float cpuAmount,
    float memoryAmount, Set<String> requirements) {
    // Execuo Automtica
    if (command.isAutomatic()) {
      List<String> candidateServers = new ArrayList<String>();
      for (String serverName : servers) {
        Server.logFineMessage("Tentando sga: " + serverName);
        if (checkServer(serverName, platforms, cpuAmount, memoryAmount,
          requirements)) {
          candidateServers.add(serverName);
        }
      }
      return schedPolicyInterface.chooseServer(command, candidateServers);
    }
    // Execuo Manual
    String serverName = command.getSGAName();
    Server.logFineMessage("Verificando sga: " + serverName);
    if (checkServer(serverName, platforms, cpuAmount, memoryAmount,
      requirements)) {
      return serverName;
    }
    return null;
  }

  /**
   * Obtm o <code>SGASet</code> para o nome dado.
   * 
   * @param serverName Nome do SGA.
   * @return O <code>SGASet</code>, ou null caso no haja SGASet para o nome
   *         dado.
   */
  private SGASet getSGASet(String serverName) {
    SGASet sgaSet = null;
    // O mtodo getSGASet j verifica se o usurio possui permisso para
    // executar comandos. Caso no possua permisso,
    // o SGASet fica desabilitado.
    sgaSet = sgaService.getSGASet(serverName);
    if (sgaSet == null) {
      Server.logSevereMessage("Falha na obteno do sga " + serverName);
    }
    return sgaSet;
  }

  /**
   * Faz todas as verificaes necessrias para saber se o servidor pode
   * executar o comando.
   * 
   * @param server Nome do servidor
   * @param platforms Lista das plataformas em que o comando pode ser executado
   * @param cpuAmount Unidade de processamento do algoritmo, ou seja, quanto de
   *        CPU o algoritmo precisa.
   * @param memoryAmount Quantidade de memria (em Mb) necessria para executar
   *        o comando.
   * @param requirements Requisitos necessrios para executar o comando.
   * @return true se o servidor estiver apto a executar o comando, ou false caso
   *         contrrio.
   */
  private boolean checkServer(String server, Vector<String> platforms,
    float cpuAmount, float memoryAmount, Set<String> requirements) {
    Server.logFineMessage("SGA[" + server + "] Obtendo SGASet.");
    SGASet sgaSet = getSGASet(server);
    if (sgaSet == null) {
      Server.logFineMessage("SGA[" + server + "] SGASet == null");
      return false;
    }
    Server.logFineMessage("SGA[" + server + "] SGASet obtido "
      + (sgaSet.getAlive() ? "vivo" : "morto"));
    // Poderamos retornar diretamente a chamada dos mtodos com o operador "&&"
    // Fizemos desta maneira para colocar logs.
    boolean inQuarantine = isInQuarantine(server);
    Server
      .logFineMessage("SGA[" + server + "] isInQuarantine: " + inQuarantine);
    if (inQuarantine) {
      return false;
    }

    boolean checkServerRestrictions =
      checkServerRestrictions(sgaSet, platforms);
    Server.logFineMessage("SGA[" + server + "] checkServerRestrictions: "
      + checkServerRestrictions);
    if (!checkServerRestrictions) {
      return false;
    }

    boolean checkServerResources =
      checkServerResources(sgaSet, cpuAmount, memoryAmount, requirements);
    Server.logFineMessage("SGA[" + server + "] checkServerResources: "
      + checkServerResources);
    if (!checkServerResources) {
      return false;
    }

    boolean checkResourcesController =
      schedulerResourcesController.checkControllerResources(sgaSet, cpuAmount,
        memoryAmount);
    Server.logFineMessage("SGA[" + server + "] checkResourcesController: "
      + checkResourcesController);
    if (!checkResourcesController) {
      return false;
    }

    return true;
  }

  /**
   * Retira os SGAs da quarentena se seus tempos j expiraram.
   */
  private void clearQuarantine() {
    Iterator<Long> iterator = recentUsedServers.values().iterator();
    while (iterator.hasNext()) {
      Long quarantineExpirationTime = iterator.next();
      // Verifica se a quarentena j expirou. Nesse caso, retira
      // o SGA da lista e retorna true.
      long now = System.currentTimeMillis();
      if (now - quarantineExpirationTime > sgaQuarantine) {
        iterator.remove();
      }
    }
  }

  /**
   * Verifica se o servidor est na quarentena.
   * 
   * @param sgaName Nome do servidor.
   * @return <code>true</code> se o servidor est na quarentena,
   *         <code>false</code> caso contrrio.
   */
  private boolean isInQuarantine(String sgaName) {
    if (!recentUsedServers.containsKey(sgaName)) {
      return false;
    }
    // Verifica se a quarentena j expirou. Nesse caso, retira
    // o SGA da lista e retorna true.
    long now = System.currentTimeMillis();
    if ((sgaQuarantine > 0)
      && (now - recentUsedServers.get(sgaName) <= sgaQuarantine)) {
      return true;
    }
    recentUsedServers.remove(sgaName);
    return false;
  }

  /**
   * Verifica se o servidor est disponvel, se o usurio possui permisso e se
   * o algoritmo possui binrios na plataforma do servidor.
   * 
   * @param sgaSet SGA
   * @param platforms Lista das plataformas em que o algoritmo pode ser
   *        executado.
   * @return <code>true</code> se o servidor puder executar o algoritmo,
   *         <code>false</code> caso contrrio.
   */
  private boolean checkServerRestrictions(SGASet sgaSet,
    Vector<String> platforms) {
    if (sgaSet.isPlatformSupported(platforms) && sgaSet.mayExecuteCommand()) {
      return true;
    }
    return false;
  }

  /**
   * Verifica se o server possui CPU e memria suficientes para a execuo.
   * 
   * @param sgaSet SGA
   * @param cpuAmount Quantidade de CPU requerida
   * @param memoryAmount Quantidade de memria requerida.
   * @param requirements Requisitos necessrios para executar o comando.
   * @return <code>true</code> se houver recursos para executar o algoritmo,
   *         <code>false</code> caso contrrio.
   */
  private boolean checkServerResources(SGASet sgaSet, float cpuAmount,
    float memoryAmount, Set<String> requirements) {
    if (cpuAmount <= 0 && memoryAmount <= 0 && requirements.size() == 0) {
      return true;
    }

    double localNumProcessors =
      sgaSet.getNumProcessors()
        - (sgaSet.getCPULoad1() * sgaSet.getNumProcessors() / 100);
    Server.logFineMessage("CPU livre: " + localNumProcessors);
    Server.logFineMessage("Memria (Mb) livre: " + sgaSet.getRAMFreeMemoryMb());
    boolean hasAllRequirements = true;
    /*
     * Mesmo quando um falhar, continua o lao para saber quais os requisitos
     * que o n no possui.
     */
    for (String requirement : requirements) {
      boolean hasRequirement = sgaSet.hasRequirement(requirement);
      hasAllRequirements = hasAllRequirements && hasRequirement;
      Server.logFineMessage("Requisito " + requirement + ": " + hasRequirement);
    }

    if ((localNumProcessors >= cpuAmount)
      && (sgaSet.getRAMFreeMemoryMb() >= memoryAmount) && hasAllRequirements) {
      return true;
    }

    return false;
  }
}
