package csbase.server.services.opendreamsservice.opendreams.v1_7;

import java.util.HashMap;
import java.util.Map;

import csbase.exception.OperationFailureException;
import csbase.logic.CommonClientProject;
import csbase.server.services.projectservice.ProjectService;

/**
 * Gerencia as sesses ativas para um determinado usurio. Cada usurio somente
 * pode ter uma sesso ativa. A sesso est associada a um nico projeto.
 * 
 * @author Tecgraf PUC-Rio
 * 
 */
public class SessionManager {

  /**
   * Informaes que identificam univocamente uma sesso.
   * <ul>
   * <li>Usurio executando a chamada que requisitou a sesso
   * <li>Nome do projeto para o qual a sesso est sendo recuperada
   * </ul>
   */
  private static class SessionInfo {
    /** Usurio executando a chamada corrente */
    public final String user;
    /** Nome do projeto para o qual a sesso est sendo recuperada */
    public final String projectName;

    /**
     * Constri as informaes que identificam univocamente uma sesso.
     * 
     * @param user usurio executando a chamada corrente
     * @param projectName nome do projeto para o qual a sesso est sendo
     *        recuperada
     */
    public SessionInfo(String user, String projectName) {
      this.user = user;
      this.projectName = projectName;
    }

    /**
     * {@inheritDoc}
     * 
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object o) {
      if (this == o) {
        return true;
      }
      if (o == null || this.getClass() != o.getClass()) {
        return false;
      }
      SessionInfo other = SessionInfo.class.cast(o);
      return this.user.equals(other.user)
        && this.projectName.equals(other.projectName);
    }

    /**
     * {@inheritDoc}
     * 
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
      final int prime = 31;
      int result = 1;
      result =
        prime * result + ((projectName == null) ? 0 : projectName.hashCode());
      result = prime * result + ((user == null) ? 0 : user.hashCode());
      return result;
    }
  }

  /**
   * Sesses ativas por usuario/projeto.
   */
  private Map<SessionInfo, OpenDreamsSession> sessions;

  /**
   * Constri um gerente de sesses para o OpenDreams.
   */
  protected SessionManager() {
    sessions = new HashMap<SessionInfo, OpenDreamsSession>();
  }

  /**
   * Obtm uma sesso para o usurio. Cada usurio somente pode ter uma sesso
   * ativa por projeto. Se ainda no houver uma sesso ativa, cria essa sesso.
   * 
   * @param userId o identificador do usurio que est requisitando uma sesso
   * @param projectName o nome do projeto
   * @return uma sesso <code>OpenDreamsSession</code>
   * 
   * @throws OperationFailureException se houver algum erro na execuo dessa
   *         operao.
   */
  public Session getSession(String userId, String projectName)
    throws OperationFailureException {
    try {
      OpenDreamsSession session =
        sessions.get(new SessionInfo(userId, projectName));
      if (session == null) {
        session = new OpenDreamsSession();
      }
      ProjectService projectService = ProjectService.getInstance();
      Object projectId = projectService.getProjectId(userId, projectName);
      CommonClientProject project =
        projectService.openProject(projectId, false);
      session.setProject(project);
      session.setUser(userId);
      return session;
    }
    catch (Throwable e) {
      throw new OperationFailureException(e.getMessage());
    }
  }
}
