/*
 * $Id: ProjectDataService.java 128667 2012-05-03 20:40:27Z mjulia $
 */
package csbase.server.services.projectservice.v1_01;

import java.io.FileNotFoundException;
import java.io.UnsupportedEncodingException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;

import org.omg.CORBA.Any;
import org.omg.CORBA.ORB;

import tecgraf.ftc.common.exception.FailureException;
import tecgraf.ftc.common.exception.MaxClientsReachedException;
import tecgraf.ftc.common.exception.PermissionException;
import tecgraf.ftc.utils.Utils;
import tecgraf.openbus.data_service.core.v1_01.AbsentViews;
import tecgraf.openbus.data_service.core.v1_01.DataAccessDenied;
import tecgraf.openbus.data_service.core.v1_01.DataAlreadyExist;
import tecgraf.openbus.data_service.core.v1_01.DataDescription;
import tecgraf.openbus.data_service.core.v1_01.DataKeyWrapper;
import tecgraf.openbus.data_service.core.v1_01.DataNotFound;
import tecgraf.openbus.data_service.core.v1_01.DefaultView;
import tecgraf.openbus.data_service.core.v1_01.IDataService;
import tecgraf.openbus.data_service.core.v1_01.InvalidDataKey;
import tecgraf.openbus.data_service.core.v1_01.Metadata;
import tecgraf.openbus.data_service.core.v1_01.ServiceFailure;
import tecgraf.openbus.data_service.core.v1_01.UnavailableDataService;
import tecgraf.openbus.data_service.core.v1_01.UnstructuredDataView;
import tecgraf.openbus.data_service.core.v1_01.UnstructuredDataViewHelper;
import tecgraf.openbus.data_service.core.v1_01.UnsupportedView;
import tecgraf.openbus.data_service.hierarchical.v1_01.IHierarchicalTransferDataServiceOperations;
import tecgraf.openbus.data_service.hierarchical.v1_01.InvalidPrototype;
import tecgraf.openbus.data_service.hierarchical.v1_01.UnsupportedOperation;
import tecgraf.openbus.data_service.hierarchical.v1_02.InvalidContainer;
import tecgraf.openbus.data_service.project.v1_01.PATH_SEPARATOR;
import tecgraf.openbus.data_service.project.v1_01.ProjectDataViewFactory;
import tecgraf.openbus.data_service.project.v1_01.ProjectDataViewHelper;
import tecgraf.openbus.data_service.project.v1_01.ProjectItemDataView;
import tecgraf.openbus.data_service.project.v1_01.ProjectItemDataViewFactory;
import tecgraf.openbus.data_service.project.v1_01.ProjectItemDataViewHelper;
import tecgraf.openbus.data_service.project.v1_01.ProjectItemDataViewImpl;
import csbase.logic.ClientProjectFile;
import csbase.logic.SyncRemoteFileChannel;
import csbase.server.Server;
import csbase.server.Service;
import csbase.server.services.openbusservice.OpenBusService;
import csbase.server.services.projectservice.ProjectService;
import csbase.server.services.projectservice.UpdatableFileInfo;

/**
 * Representa um Servio de Dados de Projeto.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ProjectDataService implements
  IHierarchicalTransferDataServiceOperations {

  /**
   * Array vazio de descritores de dados. Utilizado principalmente para retorno.
   */
  private static final DataDescription[] EMPTY_DATA_DESCRIPTION_ARRAY =
    new DataDescription[0];
  /**
   * O nome do metadado que representa o caminho absoluto de um dado.
   */
  private static final String ABSOLUTE_PATH_METADATUM_NAME = "ABSOLUTE_PATH";
  /**
   * O valor do campo SourceId na gerao das chaves
   */
  protected String SourceId = null;

  /**
   * A instncia nica do Servio de Dados de Projeto.
   */
  private static ProjectDataService instance;

  /**
   * Cria um Servico de Dados de Projeto.
   */
  private ProjectDataService() {
    SourceId = ProjectService.getInstance().getSourceId();
    org.omg.CORBA_2_3.ORB orb =
      (org.omg.CORBA_2_3.ORB) OpenBusService.getInstance().getORB();
    orb.register_value_factory(ProjectDataViewHelper.id(),
      new ProjectDataViewFactory());
    orb.register_value_factory(ProjectItemDataViewHelper.id(),
      new ProjectItemDataViewFactory());
  }

  /**
   * Obtm a instncia nica do Servio de Dados de Projeto.
   * 
   * @return A instncia nica do Servio de Dados de Projeto.
   */
  public static ProjectDataService getInstance() {
    if (instance == null) {
      instance = new ProjectDataService();
    }
    return instance;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public byte[] copyDataFrom(byte[] fSourceKey, byte[] fParentKey)
    throws ServiceFailure, DataAccessDenied, InvalidDataKey, DataNotFound,
    AbsentViews, UnavailableDataService, DataAlreadyExist, UnsupportedOperation {
    DataKeyWrapper parentKey = new DataKeyWrapper(fParentKey);
    DataValidation.checkDataKey(parentKey);

    OpenBusService.getInstance().joinChain();

    try {
      byte[] fParentKeyV1_02;
      try {
        fParentKeyV1_02 = DataBridge.convertDataKeyToV1_02(fParentKey);
      }
      catch (tecgraf.openbus.data_service.core.v1_02.InvalidDataKey e1) {
        throw new InvalidDataKey(new byte[][] { fParentKey });
      }

      tecgraf.openbus.data_service.core.v1_02.IDataServiceOperations dataServicev1_02 =
        csbase.server.services.projectservice.v1_02.ProjectDataService
          .getInstance();
      DataDescription parentDescription;
      try {
        parentDescription =
          DataBridge.convertDataDescriptionToV1_01(dataServicev1_02
            .getDataDescription(fParentKeyV1_02));
      }
      catch (tecgraf.openbus.data_service.core.v1_02.ServiceFailure e1) {
        throw new ServiceFailure(e1.fMessage);
      }
      catch (tecgraf.openbus.data_service.core.v1_02.DataAccessDenied e1) {
        throw new DataAccessDenied(new byte[][] { fParentKey });
      }
      catch (tecgraf.openbus.data_service.core.v1_02.DataNotFound e1) {
        throw new DataNotFound(new byte[][] { fParentKey });
      }
      catch (tecgraf.openbus.data_service.core.v1_02.InvalidDataKey e1) {
        throw new InvalidDataKey(new byte[][] { fParentKey });
      }

      String parentAbsolutePath = null;
      for (int i = 0; i < parentDescription.fMetadata.length; i++) {
        if (parentDescription.fMetadata[i].fName
          .equals(ABSOLUTE_PATH_METADATUM_NAME)) {
          Any any = parentDescription.fMetadata[i].fValue;
          parentAbsolutePath = any.extract_string();
        }
      }

      DataKeyWrapper sourceDataKey = new DataKeyWrapper(fSourceKey);
      IDataService sourceDataService = FindService.find(sourceDataKey);
      if (sourceDataService == null) {
        String userLogin = Service.getUser().getLogin();
        String msg =
          MessageFormat
            .format(
              "Ocorreu um erro na solicitao do usurio {0}: error ao copiar o dado: servio de dados da origem no disponvel: {1} {2}",
              new Object[] { userLogin, sourceDataKey.getSystemDeploymentId(),
                  sourceDataKey.getDataSourceId() });
        Server.logSevereMessage(msg);
        throw new UnavailableDataService(msg);
      }

      DataDescription sourceDataDescription =
        sourceDataService.getDataDescription(fSourceKey);

      ProjectItemDataView sourceProjItemDataView =
        ProjectItemDataView.class.cast(sourceDataService.getDataView(
          fSourceKey, ProjectItemDataViewHelper.id()));

      String absolutePath =
        parentAbsolutePath + PATH_SEPARATOR.value + sourceDataDescription.fName;
      Metadata[] metadata = createMetadataList(absolutePath);
      long date = System.currentTimeMillis();

      String userLogin = Service.getUser().getLogin();

      ProjectItemDataView prototypeProjItemDataView =
        new ProjectItemDataViewImpl(null, userLogin,
          sourceProjItemDataView.fDescription, absolutePath,
          sourceProjItemDataView.fType, 0, sourceProjItemDataView.fIsContainer,
          sourceProjItemDataView.fCanRead, sourceProjItemDataView.fCanWrite,
          date, date);

      DefaultView defaultView =
        new DefaultView(ProjectItemDataViewHelper.id(),
          prototypeProjItemDataView);

      DataDescription prototype =
        new DataDescription(null, sourceDataDescription.fName, defaultView,
          null, metadata);

      tecgraf.openbus.data_service.core.v1_02.DataDescription prototypeV1_02 =
        DataBridge.convertDataDescriptionToV1_02(prototype);

      tecgraf.openbus.data_service.hierarchical.v1_02.IHierarchicalManagementDataServiceOperations managementDataServiceV1_02 =
        csbase.server.services.projectservice.v1_02.ProjectDataService
          .getInstance();

      byte[] key = null;
      try {
        key =
          DataBridge.convertDataKeyToV1_01(managementDataServiceV1_02
            .createData(prototypeV1_02, fParentKeyV1_02));
      }
      catch (tecgraf.openbus.data_service.core.v1_02.InvalidDataKey e) {
        throw new InvalidDataKey(new byte[][] { fParentKey });
      }
      catch (tecgraf.openbus.data_service.core.v1_02.ServiceFailure e) {
        throw new ServiceFailure(e.fMessage);
      }
      catch (tecgraf.openbus.data_service.hierarchical.v1_02.UnsupportedOperation e) {
        throw new UnsupportedOperation();
      }
      catch (tecgraf.openbus.data_service.hierarchical.v1_02.InvalidPrototype e) {
        throw new InvalidPrototype();
      }
      catch (tecgraf.openbus.data_service.core.v1_02.DataAlreadyExist e) {
        throw new DataAlreadyExist();
      }
      catch (tecgraf.openbus.data_service.core.v1_02.DataNotFound e) {
        throw new DataNotFound(new byte[][] { fParentKey });
      }
      catch (tecgraf.openbus.data_service.core.v1_02.DataAccessDenied e) {
        throw new DataAccessDenied(new byte[][] { fParentKey });
      }

      this.updateDataFrom(key, fSourceKey);

      return key;
    }
    catch (ServiceFailure e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao copiar o dado: {1}",
            new Object[] { userLogin, e.fMessage });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
    // Verso 1.2 lana esta exceo que  convertidad para a ServiceFailure na verso 1.1
    catch (InvalidContainer e) {
      throw new ServiceFailure("Dado pai no  um diretrio.");
    }
    catch (InvalidPrototype e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao copiar o dado: prottipo invlido.",
            new Object[] { userLogin });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
    catch (UnsupportedView e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao copiar o dado: viso no suportada.",
            new Object[] { userLogin });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
    finally {
      OpenBusService.getInstance().exitChain();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void updateDataFrom(byte[] fKey, byte[] fSourceKey)
    throws ServiceFailure, DataAccessDenied, InvalidDataKey, DataNotFound,
    UnavailableDataService, AbsentViews {
    DataKeyWrapper dataKey = new DataKeyWrapper(fKey);
    DataValidation.checkDataKey(dataKey);

    byte[] fKeyV1_02;
    tecgraf.openbus.data_service.core.v1_02.DataKeyWrapper dataKeyV1_02 = null;
    try {
      fKeyV1_02 = DataBridge.convertDataKeyToV1_02(fKey);
      dataKeyV1_02 =
        new tecgraf.openbus.data_service.core.v1_02.DataKeyWrapper(fKeyV1_02);
    }
    catch (tecgraf.openbus.data_service.core.v1_02.InvalidDataKey e2) {
      throw new InvalidDataKey(new byte[][] { fKey });
    }

    ClientProjectFile file;
    try {
      file =
        csbase.server.services.projectservice.v1_02.ProjectDataService
          .getProjectFile(dataKeyV1_02);
    }
    catch (tecgraf.openbus.data_service.core.v1_02.InvalidDataKey e1) {
      throw new InvalidDataKey(new byte[][] { fKey });
    }
    catch (tecgraf.openbus.data_service.core.v1_02.ServiceFailure e1) {
      throw new ServiceFailure(e1.fMessage);
    }
    catch (tecgraf.openbus.data_service.core.v1_02.DataNotFound e1) {
      throw new DataNotFound(new byte[][] { fKey });
    }
    catch (tecgraf.openbus.data_service.core.v1_02.DataAccessDenied e1) {
      throw new DataAccessDenied(new byte[][] { fKey });
    }

    DataKeyWrapper sourceDataKey = new DataKeyWrapper(fSourceKey);

    OpenBusService.getInstance().joinChain();

    //Origem e destino iguais no h nada a fazer
    if (dataKey.getSystemDeploymentId().equals(
      sourceDataKey.getSystemDeploymentId())) {
      if (dataKey.getDataSourceId().equals(sourceDataKey.getDataSourceId())) {
        if (dataKey.getDataId().equals(sourceDataKey.getDataId())) {
          return;
        }
      }
    }

    IDataService sourceDataService = FindService.find(sourceDataKey);
    if (sourceDataService == null) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao copiar o dado: servio de dados da origem no disponvel: {1} {2}",
            new Object[] { userLogin, sourceDataKey.getSystemDeploymentId(),
                sourceDataKey.getDataSourceId() });
      Server.logSevereMessage(msg);
      throw new UnavailableDataService(msg);
    }

    UnstructuredDataView sourceView;
    try {
      sourceView =
        UnstructuredDataView.class.cast(sourceDataService.getDataView(
          sourceDataKey.getKey(), UnstructuredDataViewHelper.id()));
    }
    catch (UnsupportedView e) {
      throw new AbsentViews(e.fKeys, new String[] { UnstructuredDataViewHelper
        .id() });
    }

    UnstructuredDataView view = null;
    try {
      tecgraf.openbus.data_service.core.v1_02.IDataServiceOperations dataServicev1_02 =
        csbase.server.services.projectservice.v1_02.ProjectDataService
          .getInstance();
      tecgraf.openbus.data_service.core.v1_02.UnstructuredDataView viewV1_02 =
        null;
      try {
        viewV1_02 =
          tecgraf.openbus.data_service.core.v1_02.UnstructuredDataView.class
            .cast(dataServicev1_02.getDataView(fKeyV1_02, DataBridge
              .convertInterfaceNameToV1_02(UnstructuredDataViewHelper.id())));
      }
      catch (UnsupportedView e) {
        String userLogin = Service.getUser().getLogin();
        String msg =
          MessageFormat
            .format(
              "Ocorreu um erro na solicitao do usurio {0}: error ao atualizar o dado: erro ao converter UnstructuredDataView.",
              new Object[] { userLogin, dataKey.getDataId() });
        Server.logSevereMessage(msg, e);
        throw new ServiceFailure(msg);
      }
      view =
        UnstructuredDataView.class.cast(DataBridge
          .convertDataViewToV1_01(viewV1_02));
    }
    catch (tecgraf.openbus.data_service.core.v1_02.UnsupportedView e) {
      throw new AbsentViews(new byte[][] { fKey },
        new String[] { UnstructuredDataViewHelper.id() });
    }
    catch (tecgraf.openbus.data_service.core.v1_02.ServiceFailure e) {
      throw new ServiceFailure(e.fMessage);
    }
    catch (tecgraf.openbus.data_service.core.v1_02.DataAccessDenied e) {
      throw new DataAccessDenied(new byte[][] { fKey });
    }
    catch (tecgraf.openbus.data_service.core.v1_02.DataNotFound e) {
      throw new DataNotFound(new byte[][] { fKey });
    }
    catch (tecgraf.openbus.data_service.core.v1_02.InvalidDataKey e) {
      throw new InvalidDataKey(new byte[][] { fKey });
    }

    SyncRemoteFileChannel channel;
    try {
      channel =
        new SyncRemoteFileChannel(dataKey.getDataId().getBytes(
          Utils.CHARSET_ENCODING), view.fWritable, view.fHost, view.fPort,
          view.fAccessKey);
    }
    catch (UnsupportedEncodingException e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao atualizar o dado: erro ao converter o identificador do dado de destino ({1}) para bytes.",
            new Object[] { userLogin, dataKey.getDataId() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }

    SyncRemoteFileChannel sourceChannel;
    try {
      sourceChannel =
        new SyncRemoteFileChannel(sourceDataKey.getDataId().getBytes(
          Utils.CHARSET_ENCODING), sourceView.fWritable, sourceView.fHost,
          sourceView.fPort, sourceView.fAccessKey);
    }
    catch (UnsupportedEncodingException e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao atualizar o dado: erro ao converter o identificador do dado de origem ({1}) para bytes.",
            new Object[] { userLogin, sourceDataKey.getDataId() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }

    try {
      channel.open(false);
    }
    catch (PermissionException e) {
      Server.logSevereMessage(e.getMessage(), e);
      throw new DataAccessDenied(new byte[][] { fKey });
    }
    catch (FileNotFoundException e) {
      Server.logSevereMessage(e.getMessage(), e);
      throw new DataNotFound(new byte[][] { fKey });
    }
    catch (FailureException e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao atualizar o dado: falha na transferncia do arquivo: {1}",
            new Object[] { userLogin, e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
    catch (MaxClientsReachedException e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao atualizar o dado: quantidade mxima de clientes atingida: {1}",
            new Object[] { userLogin, e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }

    try {
      sourceChannel.open(true);
    }
    catch (PermissionException e) {
      Server.logSevereMessage(e.getMessage(), e);
      throw new DataAccessDenied(new byte[][] { fSourceKey });
    }
    catch (FileNotFoundException e) {
      Server.logSevereMessage(e.getMessage(), e);
      throw new DataNotFound(new byte[][] { fSourceKey });
    }
    catch (FailureException e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao atualizar o dado: falha na transferncia do arquivo: {1}",
            new Object[] { userLogin, e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
    catch (MaxClientsReachedException e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao atualizar o dado: quantidade mxima de clientes atingida: {1}",
            new Object[] { userLogin, e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }

    try {
      ProjectItemDataView sourceProjectView =
        (ProjectItemDataView) sourceDataService.getDataView(sourceDataKey
          .getKey(), ProjectItemDataViewHelper.id());

      channel.syncTransferFrom(sourceChannel, 0, sourceProjectView.fSize);
    }
    catch (PermissionException e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao atualizar o dado: arquivo aberto somente para escrita: {1}",
            new Object[] { userLogin, e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
    catch (FailureException e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao atualizar o dado: falha na transferncia do arquivo: {1}",
            new Object[] { userLogin, e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
    catch (tecgraf.ftc.common.exception.FileLockedException e) {
      String userLogin = Service.getUser().getLogin();
      String msg =
        MessageFormat
          .format(
            "Ocorreu um erro na solicitao do usurio {0}: error ao atualizar o dado: arquivo bloqueado para escrita: {1}",
            new Object[] { userLogin, e.getMessage() });
      Server.logSevereMessage(msg, e);
      throw new ServiceFailure(msg);
    }
    catch (UnsupportedView e) {
      throw new AbsentViews(e.fKeys, new String[] { ProjectItemDataViewHelper
        .id() });
    }
    finally {
      try {
        OpenBusService.getInstance().exitChain();
        channel.close();
      }
      catch (FailureException e) {
      }
      finally {
        try {
          sourceChannel.close();
        }
        catch (FailureException e) {
        }
      }
    }

    if (!ProjectService.getInstance().setUpdatableFileInfo(
      file,
      new UpdatableFileInfo(OpenBusProjectFileUpdater.class, sourceDataKey
        .getKey()))) {
      Server.logWarningMessage(MessageFormat.format(
        "A informao sobre atualizao do arquivo {0} no foi definida",
        new Object[] { file.getStringPath() }));
    }
  }

  /**
   * Atualiza um arquivo a partir de um dado localizado em um outro servidor.
   * 
   * @param projectId O identificador do projeto.
   * @param pathArray O caminho para o arquivo.
   * @param fSourceKey A chave unvoca do dado de origem, ou seja, o dado
   *        armazenado no outro servidor.
   * 
   * @throws ServiceFailure Caso ocorra uma falha no servio.
   * @throws DataAccessDenied Caso o usurio solicitante no tenha acesso ao
   *         dado de origem
   * @throws InvalidDataKey Caso a chave do dado de origem no seja vlida.
   * @throws DataNotFound Caso o dado de origem no exista.
   * @throws UnavailableDataService Caso o servio de dados da origem esteja
   *         indisponvel
   * @throws AbsentViews Caso as vises oferecidas pelo dado de origem no sejam
   *         conhecidas deste servidor.
   */
  public void updateDataFrom(Object projectId, String[] pathArray,
    byte[] fSourceKey) throws ServiceFailure, DataAccessDenied, InvalidDataKey,
    DataNotFound, UnavailableDataService, AbsentViews {
    String dataId = generateDataId(projectId, pathArray);
    DataKeyWrapper dataKey = this.generateDataKey(dataId);
    this.updateDataFrom(dataKey.getKey(), fSourceKey);
  }

  /**
   * Cria uma lista de metadados para os dados oferecidos por este servio.
   * 
   * @param absolutePath O caminho absoluto do dado.
   * 
   * @return A lista de metadados.
   */
  private static Metadata[] createMetadataList(String absolutePath) {
    List<Metadata> metadataList = new ArrayList<Metadata>();

    ORB orb = OpenBusService.getInstance().getORB();
    Any any;

    if (absolutePath != null) {
      any = orb.create_any();
      any.insert_string(absolutePath);
      metadataList.add(new Metadata(ABSOLUTE_PATH_METADATUM_NAME, any));
    }

    return metadataList.toArray(new Metadata[0]);
  }

  /**
   * Gera um identificador para um arquivo de projeto.
   * 
   * @param projectId O identificador do arquivo.
   * @param pathArray O caminho para o arquivo de projeto.
   * @return O identificador.
   */
  private static String generateDataId(Object projectId, String[] pathArray) {
    String dataId = (String) projectId;
    for (String path : pathArray) {
      dataId += ProjectService.FILE_ID_SEPARATOR + path;
    }
    return dataId;
  }

  /**
   * Gera um identificador unvoco a partir de um identificador de um arquivo de
   * projeto.
   * 
   * @param dataId O identificador do arquivo de projeto.
   * 
   * @return O identificador unvoco.
   * 
   * @throws InvalidDataKey Caso ocorra algum erro ao criar o identificador
   *         unvoco.
   */
  private DataKeyWrapper generateDataKey(String dataId) throws InvalidDataKey {
    String systemDeploymentId = OpenBusService.getInstance().getEntityName();
    DataKeyWrapper dataKey =
      new DataKeyWrapper(systemDeploymentId, SourceId, dataId);
    return dataKey;
  }
}
