/*
 * $Id:$
 */

package csbase.server.services.openurlservice;

import java.net.MalformedURLException;
import java.net.URL;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;

import org.apache.commons.codec.binary.Base64;

import csbase.exception.CSBaseException;
import csbase.exception.ServiceFailureException;
import csbase.logic.ServerURI;
import csbase.logic.openbus.OpenBusLoginToken;
import csbase.logic.openurlservice.PredefinedURLSystem;
import csbase.logic.url.URLParameters;
import csbase.remote.OpenURLServiceInterface;
import csbase.remote.ServerEntryPoint;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.services.loginservice.ReferedServerCache;

/**
 * Servio de busca de URL de sistemas CSBASE.
 *
 * @author Tecgraf/PUC-Rio
 */
public class OpenURLService extends Service implements OpenURLServiceInterface {
  /**
   * Sistemas prdefinidos.
   */
  final private List<PredefinedURLSystem> predefinedURLSystems =
    new ArrayList<PredefinedURLSystem>();

  /**
   * Monta a URL
   *
   * @param token token para conexo com o OpenBus
   * @param hostName host (nome)
   * @param hostPort porta do servidor
   * @param locale locale desejado de login
   * @param currentClientId identificador do cliente que chamou a ao.
   * @param newClientId identificador para o novo cliente a ser lanado.
   * @param startVisible indicativo se o dekstop deve aparecer visvel.
   * @return a URL
   */
  private URL buildURL(final OpenBusLoginToken token, final String hostName,
    final int hostPort, final Locale locale, final String currentClientId,
    final String newClientId, final boolean startVisible) {
    if (locale == null) {
      final String err = getString("internal.null.locale.error");
      throw new ServiceFailureException(err);
    }

    final ServerEntryPoint serverEntryPoint;
    final ReferedServerCache referedServerCache =
      ReferedServerCache.getInstance();
    final ServerURI uri = ServerURI.create(hostName, hostPort);
    try {
      serverEntryPoint = referedServerCache.getServer(uri);
      final String uriFmt = "Servidor/Porta %s:%d definido com URI: %s";
      final String uriMsg = String.format(uriFmt, hostName, hostPort, uri);
      Server.logInfoMessage(uriMsg);

    }
    catch (final CSBaseException e) {
      final String fmt = getString("not.monitored.server.error");
      final String err = String.format(fmt, uri);
      throw new ServiceFailureException(err);
    }

    if (serverEntryPoint == null) {
      final String fmt = getString("no.server.error");
      final String err = String.format(fmt, uri);
      throw new ServiceFailureException(err);
    }

    // Montagem da URL bsica.
    String urlText;
    try {
      urlText = serverEntryPoint.getSystemURLWithRMIPort();
    }
    catch (final RemoteException e) {
      final String fmt = getString("comm.server.error");
      final String err = String.format(fmt, uri);
      throw new ServiceFailureException(err, e);
    }

    // Parametrizao: Visibilidade do desktop
    final String visParamName = URLParameters.DESKTOP_VISIBLE_PARAMETER;
    final String visValue = Boolean.toString(startVisible);
    urlText = URLParameters.addUrlParam(urlText, visParamName, visValue);

    // Parametrizao: Locale
    final String locParamName = URLParameters.LOCALE_PARAMETER;
    final String locValue = locale.toString();
    urlText = URLParameters.addUrlParam(urlText, locParamName, locValue);

    // Parametrizao: Client original (id)
    final String orParamName = URLParameters.SOURCE_CLIENT_IDENTIFIER_PARAMETER;
    urlText = URLParameters.addUrlParam(urlText, orParamName, currentClientId);

    // Parametrizao: Novo cliente (id)
    final String newcliParamName =
      URLParameters.CURRENT_CLIENT_IDENTIFIER_PARAMETER;
    urlText = URLParameters.addUrlParam(urlText, newcliParamName, newClientId);

    // Log sem id da credencial.
    final String urlFmt = "Servidor/Porta %s:%d gerou URL (sem id): [%s]";
    final String urlMsg = String.format(urlFmt, hostName, hostPort, urlText);
    Server.logInfoMessage(urlMsg);

    // Parametrizao: o usurio do token
    final String userParamName = URLParameters.OPENBUS_TOKEN_USER_PARAMETER;
    final String userValue = token.user;
    urlText = URLParameters.addUrlParam(urlText, userParamName, userValue);

    // Parametrizao: O segredo do token
    final String secretParamName = URLParameters.OPENBUS_TOKEN_SECRET_PARAMETER;
    final byte[] secretValue = token.secret;
    urlText =
      URLParameters.addUrlParam(urlText, secretParamName, Base64
        .encodeBase64String(secretValue));

    try {
      final URL url = new URL(urlText);
      return url;
    }
    catch (final MalformedURLException e) {
      final String fmt = getString("bad.url.error");
      final String err = String.format(fmt, uri);
      throw new ServiceFailureException(err, e);
    }
  }

  /**
   * Lana exceo se servio estiver inibido.
   */
  private void checkEnabled() {
    if (!isEnabled()) {
      final String err = getString("disabled.service.error");
      throw new ServiceFailureException(err);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<PredefinedURLSystem> getPredefinedURLSystems() {
    final List<PredefinedURLSystem> list = new ArrayList<PredefinedURLSystem>();
    list.addAll(predefinedURLSystems);
    Collections.unmodifiableList(list);
    return list;
  }

  /**
   * Busca um sistema pr-definido com base em um rtulo.
   *
   * @param systemLabel label
   * @return a definio do sistema.
   */
  private PredefinedURLSystem getSystem(final String systemLabel) {
    for (final PredefinedURLSystem sys : predefinedURLSystems) {
      if (sys.getLabel().equals(systemLabel)) {
        return sys;
      }
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public URL getSystemURL(final Locale locale, final OpenBusLoginToken token,
    final String systemLabel, final String currentClientId,
    final String newClientId, final boolean startVisible) {
    checkEnabled();

    if (!hasSystem(systemLabel)) {
      final String fmt = getString("no.system.error");
      final String err = String.format(fmt, systemLabel);
      throw new ServiceFailureException(err);
    }
    final Locale loc = (locale == null ? Locale.getDefault() : locale);

    final String owner = token.user;
    final String logFmt = "Pedido de URL pelo usurio: %s (%s)- tag [%s]";
    final String locText = loc.toString();
    final String logMsg = String.format(logFmt, owner, locText, systemLabel);
    Server.logInfoMessage(logMsg);

    final PredefinedURLSystem system = getSystem(systemLabel);
    final String hostAddress = system.getHostAddress();
    final int hostPort = system.getHostPort();

    final String reqFmt = "Pedido de [%s]/[%s] direcionado para: %s - porta %d";
    final String reqMsg =
      String.format(reqFmt, owner, systemLabel, hostAddress, hostPort);
    Server.logInfoMessage(reqMsg);

    final URL url =
      buildURL(token, hostAddress, hostPort, loc, currentClientId, newClientId,
        startVisible);
    return url;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public URL getURL(final Locale locale, final OpenBusLoginToken token,
    final String currentClientId, final String newClientId,
    final boolean startVisible) throws RemoteException {
    checkEnabled();
    final Server server = Server.getInstance();
    final String hostName = server.getHostName();
    final int hostPort = server.getRegistryPort();

    final Locale loc = (locale == null ? Locale.getDefault() : locale);

    final URL url =
      buildURL(token, hostName, hostPort, loc, currentClientId, newClientId,
        startVisible);
    return url;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean has2Update(final Object arg, final Object event) {
    return false;
  }

  /**
   * Indica se existe um sistema de rtulo
   *
   * @param systemLabel label
   * @return indicativo
   */
  private boolean hasSystem(final String systemLabel) {
    final PredefinedURLSystem system = getSystem(systemLabel);
    return system != null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initService() throws ServerException {
    loadPredefinedSystems();
  }

  /**
   * Faz a carga de conexes predefinidas
   */
  private void loadPredefinedSystems() {
    final String prefix = "predefined.systems.";
    final List<String> labels = getStringListProperty(prefix + "label");
    int i = 1;
    for (final String label : labels) {
      if (hasSystem(label)) {
        final String fmt = "Definio de [%s] duplicada! Descartando...";
        final String err = String.format(fmt, label);
        Server.logSevereMessage(err);
      }
      else {
        final String hostAdressTag = prefix + "host.address" + "." + i;
        final String hostPortTag = prefix + "host.port" + "." + i;

        final String hostAddress = getStringProperty(hostAdressTag);
        final int hostPort = getIntProperty(hostPortTag);

        final PredefinedURLSystem system =
          new PredefinedURLSystem(label, hostAddress, hostPort);
        final String fmt = "Sistema predefinido encontrado: %s - %s:%d";
        final String msg = String.format(fmt, label, hostAddress, hostPort);

        Server.logInfoMessage(msg);
        predefinedURLSystems.add(system);
      }
      i = i + 1;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void shutdownService() throws ServerException {
  }

  /**
   * Constri a instncia do servio de gerncia de SGAs
   *
   * @throws ServerException em caso de erro.
   */
  public static void createService() throws ServerException {
    new OpenURLService();
  }

  /**
   * Obtm uma referncia para a instncia nica (singleton) do servio de
   * algoritmos.
   *
   * @return referncia para a instncia nica do servio de algoritmos.
   */
  public static OpenURLService getInstance() {
    final String serviceName = OpenURLServiceInterface.SERVICE_NAME;
    return (OpenURLService) Service.getInstance(serviceName);
  }

  /**
   * Construtor
   *
   * @throws ServerException em caso de erro.
   */
  protected OpenURLService() throws ServerException {
    super(OpenURLServiceInterface.SERVICE_NAME);
    this.setEnabled(this.getBooleanProperty("enabled"));
  }
}
