/*
 * RoleIO.java
 * 
 * $Id: RoleIO.java 175407 2016-08-10 20:31:03Z clinio $
 */
package csbase.server.services.administrationservice;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Hashtable;
import java.util.List;
import java.util.Vector;
import java.util.Map.Entry;

import csbase.logic.IdFactory;
import csbase.logic.Permission;
import csbase.logic.Role;
import csbase.logic.RoleInfo;
import csbase.server.Server;

/**
 * A classe <code>RoleIO</code>  responsvel pela persistncia dos registros
 * dos perfis de acesso (objetos Role). Todos os objetos so mantidos em um
 * arquivo criptografado que  atualizado sempre que as funes Role.createRole,
 * Role.modifyRole e Role.removeRole so invocadas. A chave simtrica usada na
 * criptografia  gerada a partir da senha do administrador do sistema.
 */
public class RoleIO {
  /** O gerador de identificadores dos perfis */
  private IdFactory idFactory;
  /** Chave do nome do perfil na Hashtable persistida */
  private static final String NAME = "name";
  /** Chave da descrio do perfil na Hashtable persistida */
  private static final String DESCRIPTION = "description";
  /** Chave das permisses do perfil na Hashtable persistida */
  private static final String PERMISSIONS = "permissions";
  /** Nome da propriedade que possui o valor do nome do arquivo de perfis */
  private static String FILENAME_PROPERTY = "RoleIO.filename";
  /** Nome do arquivo de roles, usado no caso da propriedade acima no existir */
  private static String FILENAME = "role.dat";
  /** Nome do arquivo de roles, usado no caso da propriedade acima no existir */
  private static String fileName = FILENAME;

  /**
   * Imprime na tela o contedo do arquivo de dados
   * 
   * @param args argumentos
   * 
   * @throws Exception no caso de falha.
   */
  public static void main(String[] args) throws Exception {
    printDataFile();
  }

  /**
   * Constri o objeto responsvel pelo arquivo de perfis.
   */
  public RoleIO() {
    final Server server = Server.getInstance();
    AdministrationService admSrv = AdministrationService.getInstance();
    fileName = server.getPersistencyRootDirectoryName() + File.separator
        + admSrv.getStringProperty(FILENAME_PROPERTY);
    Server.logInfoMessage("RoleIO.RoleIO: Arquivo de perfis: " + fileName);
  }

  /**
   * Obtm todos os os perfis existentes. Caso exista algum perfil com alguma
   * permissao inexistente, grava os perfis apenas com as permissoes validas.
   * 
   * @return o vetor de perfis.
   * 
   * @throws Exception erro durante a leitura dos perfis.
   */
  public synchronized List<Role> readAll() throws Exception {
    AdministrationService srv = AdministrationService.getInstance();
    List<Role> allRoles = new Vector<Role>();
    File rolesFile = new File(fileName);
    if (!rolesFile.exists()) {
      /*
       * arquivo de perfis ainda no existe, registramos e retornamos uma lista
       * vazia
       */
      Server.logInfoMessage("Criando novo arquivo de perfis " + fileName);
      return allRoles;
    }
    // L o arquivo de perfis.
    ObjectInputStream in = null;
    try {
      in =
        new ObjectInputStream(new DataInputStream(new BufferedInputStream(
          new FileInputStream(rolesFile))));
      Hashtable<Object, Hashtable<String, Object>> roles =
        (Hashtable<Object, Hashtable<String, Object>>) in.readObject();
      boolean allRead = true;
      for (Entry<Object, Hashtable<String, Object>> roleEntry : roles
        .entrySet()) {
        Object id = roleEntry.getKey();
        Hashtable<String, Object> roleHash = roleEntry.getValue();
        String name = (String) roleHash.get(NAME);
        String desc = (String) roleHash.get(DESCRIPTION);
        Object[] permissions = (Object[]) roleHash.get(PERMISSIONS);
        Role role = new Role(id, new RoleInfo(name, desc, permissions));
        if (!checkPermissions(role))
          allRead = false;
        allRoles.add(role);
      }
      if (!allRead) {
        try {
          Server.logInfoMessage("Persistindo os perfis apenas com permisses "
            + "vlidas.");
          writeAll(allRoles);
        }
        catch (Exception e) {
          Server.logSevereMessage("Falha na persistencia dos perfis com apenas "
            + "permisses vlidas.", e);
        }
      }
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro durante a leitura e descriptografia dos "
        + "perfis de usurios", e);
    }
    finally {
      if (in != null) {
        in.close();
      }
    }
    return allRoles;
  }

  /**
   * Verifica se as permisses do perfil esto consistentes. As que no
   * existirem mais sero removidas e ser gerado um registro (log) do ocorrido.
   * Esse teste  feito para evitar que um perfil seja "perdido" caso ocorra
   * algum erro na gerncia de permisses.
   * 
   * @param role perfil
   * @return <code>true</code> se todas as permisses foram lidas,
   *         <code>false</code> se alguma permisso foi descartada por no
   *         existir mais
   */
  private boolean checkPermissions(Role role) {
    AdministrationService srv = AdministrationService.getInstance();
    Object[] ids = role.getPermissionIds();
    boolean result = true;
    int lostPermissions = 0;
    for (int i = 0; i < ids.length; i++) {
      Permission p = null;
      try {
        p = Permission.getPermission(ids[i]);
      }
      catch (Exception e) {
        Server.logSevereMessage("Erro ao obter permisso " + ids[i], e);
      }
      if (p == null) {
        Server.logSevereMessage("Permisso inexistente sendo removida do perfil "
          + role.getName() + ": " + ids[i]);
        ids[i] = null;
        lostPermissions++;
      }
    }
    if (lostPermissions > 0) {
      Object[] newIds = new Object[ids.length - lostPermissions];
      int n = 0;
      for (int i = 0; i < ids.length; i++) {
        if (ids[i] != null) {
          newIds[n++] = ids[i];
        }
      }
      role.setPermissionIds(newIds);
      result = false;
    }
    return result;
  }

  /**
   * Obtm o perfil que possui o identificador especificado
   * 
   * @param id identificador da perfil procurado
   * 
   * @return perfil encontrado ou null caso esse perfil no exista
   * 
   * @throws Exception erro durante a obteno desse perfil
   */
  public synchronized Role read(Object id) throws Exception {
    List<Role> roles = Role.getAllRoles();
    for (Role role : roles) {
      if (role.getId().equals(id)) {
        return role;
      }
    }
    return null;
  }

  /**
   * Cria um novo perfil e grava no arquivo.
   * 
   * @param info as informaes do novo pefil
   * 
   * @return o perfil criado
   * 
   * @throws Exception erro durante a criao/gravao do perfil
   */
  public synchronized Role writeNew(RoleInfo info) throws Exception {
    try {
      List<Role> roles = Role.getAllRoles();
      if (idFactory == null) {
        idFactory = new IdFactory(roles);
      }
      Object id = idFactory.next();
      Role role = new Role(id, info);
      roles.add(role);
      /* Insere o perfil no espelho local e refaz o arquivo em disco */
      writeAll(roles);
      return role;
    }
    catch (Exception e) {
      throw new IOException("Arquivo de perfis corrompido");
    }
  }

  /**
   * Modifica um perfil e grava no arquivo.
   * 
   * @param id o identificador do perfil a ser modificado
   * @param info as novas informaes do perfil
   * 
   * @return o perfil modificado
   * 
   * @throws Exception se o perfil no existir ou se ocorrer algum erro de
   *         gravao
   */
  public synchronized Role write(Object id, RoleInfo info) throws Exception {
    try {
      Role role = Role.getRole(id);
      List<Role> roles = Role.getAllRoles();
      if (!roles.remove(role)) {
        throw new Exception("Perfil no existe");
      }
      /* Constri o novo perfil e salva no espelho local */
      role = new Role(id, info);
      roles.add(role);
      /* Refaz o arquivo do disco */
      writeAll(roles);
      return role;
    }
    catch (Exception e) {
      throw new IOException(e.getMessage());
    }
  }

  /**
   * Remove um perfil do arquivo.
   * 
   * @param id o identificador do perfil a ser removido
   * 
   * @throws Exception se o perfil no est no arquivo ou se ocorrer algum erro
   *         de gravao
   */
  public synchronized void delete(Object id) throws Exception {
    List<Role> roles;
    Role role;
    try {
      role = Role.getRole(id);
      roles = Role.getAllRoles();
      if (idFactory == null) {
        idFactory = new IdFactory(roles);
      }
    }
    catch (Exception e) {
      throw new IOException();
    }
    if (!roles.remove(role)) {
      throw new Exception("Perfil no existe");
    }
    writeAll(roles);
    idFactory.free(id);
  }

  /**
   * Grava o espelho de perfis no arquivo de dados.
   * 
   * @param roleList
   * 
   * @throws Exception Erro durante a escrita dos perfis.
   */
  private synchronized void writeAll(List<Role> roleList) throws Exception {
    Hashtable<Object, Hashtable<String, Object>> roles =
      new Hashtable<Object, Hashtable<String, Object>>();
    for (Role role : roleList) {
      Hashtable<String, Object> roleHash = new Hashtable<String, Object>();
      roleHash.put(NAME, role.getName());
      roleHash.put(DESCRIPTION, role.getDescription());
      roleHash.put(PERMISSIONS, role.getPermissionIds());
      roles.put(role.getId(), roleHash);
    }
    ObjectOutputStream out =
      new ObjectOutputStream(new DataOutputStream(new BufferedOutputStream(
        new FileOutputStream(fileName))));
    try {
      out.writeObject(roles);
    }
    finally {
      out.close();
    }
  }

  /**
   * Imprime na tela o contedo do arquivo de perfis.
   */
  public static void printDataFile() {
  }
}
