/*
 * $Id: DiskUsageService.java 173667 2016-05-19 18:17:41Z clinio $
 */
package csbase.server.services.diskusageservice;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Hashtable;
import java.util.List;

import csbase.exception.InvalidRequestException;
import csbase.logic.FileSystemSpaceNotification;
import csbase.logic.User;
import csbase.logic.diskusageservice.DiskOccupation;
import csbase.remote.DiskUsageServiceInterface;
import csbase.server.FileSystem;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.services.algorithmservice.AlgorithmService;
import csbase.server.services.eventlogservice.EventLogService;
import csbase.server.services.mailservice.MailService;
import csbase.server.services.notificationservice.NotificationService;
import csbase.server.services.projectservice.ProjectService;

/**
 * A classe <code>DiskUsageService</code> implementa o servio de uso de espao
 * de disco.
 *
 * @author Tecgraf/PUC-Rio
 */
public class DiskUsageService extends Service implements
  DiskUsageServiceInterface {

  /**
   * Constri a instncia do servio.
   *
   * @throws ServerException se houver exceo no servidor.
   */
  public static void createService() throws ServerException {
    new DiskUsageService();
  }

  /**
   * Busca da referncia ao servio.
   *
   * @return o objeto do servidor <code>DiskUsageService</code>.
   */
  public static DiskUsageService getInstance() {
    return (DiskUsageService) getInstance(SERVICE_NAME);
  }

  /**
   * Flag indicativo de sada da thread de auditagem de usurios
   */
  private boolean exitAuditThread = false;

  /**
   * Thread que faz a auditagem do espao em disco ocupado.
   */
  private Thread auditThread = null;

  /**
   * Lista de diretrios a serem monitorados.
   */
  final private List<MonitoredDirectory> monitoredList =
          new ArrayList<>();

  /**
   * Valor default de alerta ao administrador para ocupao de espao em disco.
   */
  final private int DEFAULT_WARNING_PERC = 75;

  /**
   * Valor default de alerta ao administrador para ocupao de espao em disco.
   */
  final private int DEFAULT_ALERT_PERC = 90;

  /**
   * Percentual (de espao em disco) para aviso ao administrador. Esse valor 
   * lido das properties do servio e tem seu valor default defindo por
   * {@link #DEFAULT_WARNING_PERC}.
   */
  private int warningPerc = DEFAULT_WARNING_PERC;

  /**
   * Percentual (de espao em disco) para alerta ao administrador. Esse valor 
   * lido das properties do servio e tem seu valor default defindo por
   * {@link #DEFAULT_ALERT_PERC}
   */
  private int alertPerc = DEFAULT_ALERT_PERC;

  /**
   * Indicativo de envio de email para admin no caso de alerta.
   */
  private boolean mailOnAlert;

  /**
   * Busca estrura com base em um id.
   *
   * @param id identificador procurado
   * @return estrutura de monitorao.
   */
  private MonitoredDirectory findMonitorFromId(final String id) {
    for (MonitoredDirectory monitored : monitoredList) {
      if (monitored.getId().equals(id)) {
        return monitored;
      }
    }
    return null;
  }

  /**
   * Adiciona um item a monitorao.
   *
   * @param id identificador.
   * @param path path do diretrio
   * @throws ServerException em caso de m configurao.
   */
  private void addToMonitor(final String id, final String path)
    throws ServerException {
    final MonitoredDirectory md = findMonitorFromId(id);
    if (md != null) {
      final String err = "Id de diretrio duplicado: " + id;
      throw new ServerException(err);
    }

    final MonitoredDirectory newMonitor = new MonitoredDirectory(id, path);
    if (!newMonitor.isConsistent()) {
      final String prefix = "rea inconsistente no start-up: ";
      final String err = prefix + newMonitor;
      throw new ServerException(err);
    }

    if (newMonitor.isAvailable()) {
      final String prefix = "rea acessvel no start-up: ";
      Server.logInfoMessage(prefix + newMonitor);
    }
    else {
      final String prefix = "rea inacessvel no start-up: ";
      Server.logSevereMessage(prefix + newMonitor);
    }
    monitoredList.add(newMonitor);
  }

  /**
   * Faz a auditagem do espao em disco da rea do servidor e de projetos.
   */
  private void auditData() {
    final String eventLogPrefix = "disk";
    for (final MonitoredDirectory monitor : monitoredList) {
      final String id = monitor.getId();
      final DiskOccupation occupation = getDiskOccupation(id);
      if (occupation == null || !occupation.isValid()) {
        return;
      }
      final EventLogService eventService = EventLogService.getInstance();
      final String usedFS = "[" + id + "]";
      final String[] usedQueue = new String[] { eventLogPrefix, id
        + "-file-system" };
      final double usedSrvPerc = occupation.getUsedSpacePerc();
      final double usedSrvMb = occupation.getUsedSpaceMb();
      final double totalSrvMb = occupation.getTotalSpaceMb();
      final String[] usedInfo = new String[] { usedFS, formatToAudit(
        usedSrvPerc), formatToAudit(usedSrvMb), formatToAudit(totalSrvMb) };
      eventService.addServerInformation(usedQueue, usedInfo);
      notifyAdminIfNeeded(id, occupation);
    }
  }

  /**
   * Criao da thread que faz a gravao de auditagem.
   *
   * @see #initService()
   */
  private void createAuditThread() {
    /*
     * Busca das propriedades que identificam os tempos para gravao
     * (auditagem) dos dados em disco
     */
    final int ONE_HOUR = 1;
    long auditIntervalHours = getAuditIntervalHours();
    if (auditIntervalHours < ONE_HOUR) {
      auditIntervalHours = ONE_HOUR;
    }
    Server.logInfoMessage("Intervalo de auditagens: " + auditIntervalHours
      + " horas.");
    final long sleepTime = auditIntervalHours * 60 * 60 * 1000;
    // final long sleepTime = 5 * 1000;
    auditThread = new Thread(new Runnable() {
      @Override
      public void run() {
        while (!exitAuditThread) {
          try {
            Server.logFineMessage("Thread iniciada: createAuditThread.");
            auditData();
            Server.logFineMessage("Thread terminada: createAuditThread.");
            Thread.sleep(sleepTime);
          }
          catch (final InterruptedException ie) {
            Server.logWarningMessage("Gravao de auditagem interrompida!");
          }
        }
        Server.logWarningMessage(
          "Finalizando thread de gravao de auditagem...");
      }
    });

    warningPerc = getIntProperty("warningPercentage");
    alertPerc = getIntProperty("alertPercentage");

    warningPerc = Math.abs(warningPerc);
    alertPerc = Math.abs(alertPerc);

    final int minWarnPerc = 10;
    final int minAlertPerc = 20;
    if (warningPerc < minWarnPerc) {
      warningPerc = minWarnPerc;
    }
    if (alertPerc < minAlertPerc) {
      alertPerc = minAlertPerc;
    }

    final int maxPerc = 100;
    if (warningPerc > maxPerc) {
      warningPerc = maxPerc;
    }
    if (alertPerc > maxPerc) {
      alertPerc = maxPerc;
    }

    if (alertPerc <= warningPerc) {
      alertPerc = warningPerc + 1;
    }

    Server.logInfoMessage("Percentual para aviso: " + warningPerc + "%");
    Server.logInfoMessage("Percentual para alerta: " + alertPerc + "%");

    exitAuditThread = false;

    final String threadPrefixName = getClass().getSimpleName();
    auditThread.setName(threadPrefixName + "::auditThread");
    auditThread.start();
  }

  /**
   * Formatador de nmero para arquivo de auditagem.
   *
   * @param value valor
   * @return string
   */
  private String formatToAudit(final double value) {
    final String str = String.format("%.3f", value);
    return str;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public DiskOccupation getAlgorithmOccupation() {
    final DiskOccupation oc = getDiskOccupation(DIR_ALGORITHM_ID);
    return oc;
  }

  /**
   * Atualiza os dados de um FileSystem.
   *
   * @param id o path do fileSystem a atualizar
   * @return a ocupao
   */
  private DiskOccupation getDiskOccupation(final String id) {
    final MonitoredDirectory md = findMonitorFromId(id);
    if (md == null) {
      Server.logSevereMessage("No localizado id :" + id + ".");
      final DiskOccupation dummy = new DiskOccupation();
      return dummy;
    }

    if (!md.isAvailable()) {
      final String err = "rea monitorada no disponvel: " + md;
      Server.logSevereMessage(err);
      final DiskOccupation dummy = new DiskOccupation();
      return dummy;
    }

    if (!md.isConsistent()) {
      final String err = "rea monitorada inconsistente: " + md;
      Server.logSevereMessage(err);
      final DiskOccupation dummy = new DiskOccupation();
      return dummy;
    }

    final File directory = md.getDirectory();
    final DiskOccupation occupation = getDiskOccupationFromPath(directory);
    return occupation;
  }

  /**
   * Consulta a ocupao com base em um diretrio.
   *
   * @param directory diretrio
   * @return ocupao
   */
  private DiskOccupation getDiskOccupationFromPath(final File directory) {
    if (!directory.exists()) {
      final String path = directory.getAbsolutePath();
      final String prefixErr = "No foi possvel localizar diretrio: ";
      final String sufixErr = ". Diretrio ou arquivo no existe.";
      Server.logWarningMessage(prefixErr + path + sufixErr);
      return new DiskOccupation();
    }

    final String path = directory.getAbsolutePath();
    final long total = FileSystem.getTotalSpace(path);
    final long free = FileSystem.getFreeSpace(path);

    // final long total = FileSystem.getTotalSpace(path);
    // final long used = (long)(getDiskUsageMb(file) * 1024. * 1024.);
    // final long free = total - used;

    final DiskOccupation diskOccupation = new DiskOccupation(free, total,
      alertPerc, warningPerc);
    return diskOccupation;
  }

  /**
   * Avaliao do espao (Mb) ocupado por uma estrutura do tipo File do Java que
   * retorna o tamanho do arquivo ou busca recursivamente outros 'Files' nos
   * diretrios.
   *
   * @param dir o diretrio de pesquisa.
   * @return espao em megabytes.
   */
  private double getDiskUsageFromFileMb(final File dir) {
    double diskUsageMb = 0;
    final File[] files = dir.listFiles();
    if (files == null) {
      return 0;
    }
    for (final File file : files) {
      final double itemUsageMb;
      if (file.isDirectory()) {
        itemUsageMb = getDiskUsageFromFileMb(file);
      }
      else {
        itemUsageMb = (file.length()) / 1024.0 / 1024.0;
      }
      diskUsageMb = diskUsageMb + itemUsageMb;
    }
    return diskUsageMb;
  }

  /**
   * Retorno do tamanho ocupado.
   *
   * @param path path.
   * @return espao em megabytes.
   */
  private double getDiskUsageFromPathMb(final String path) {
    final File dir = new File(path);
    final double diskUsage = getDiskUsageFromFileMb(dir);
    return diskUsage;
  }

  /**
   * Busca o nome do diretrio a ser verificado com base em um owner e nome de
   * projeto.
   *
   * @param ownerId owner do projeto
   * @param projName nome do projeto
   * @return o nome do diretrio a ser verificado.
   */
  private String getPathFromOwnerAndProjectName(final Object ownerId,
    final String projName) {
    if (ownerId == null) {
      throw new InvalidRequestException("ownerId null detected!");
    }
    if (projName == null) {
      throw new InvalidRequestException("project name null detected!");
    }
    final String dirName = getUserProjectDirectoryPath(ownerId,
      projName);

    if (dirName == null) {
      final String id = ownerId + File.separator + projName;
      Server.logSevereMessage("Usurio/Projeto [" + id
        + "] no tem diretrio!");
      throw new InvalidRequestException("project not detected!" + id);
    }
    return dirName;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public DiskOccupation getProjectOccupation() {
    final DiskOccupation oc = getDiskOccupation(DIR_PROJECT_ID);
    return oc;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public DiskOccupation getServerOccupation() {
    final DiskOccupation oc = getDiskOccupation(DIR_SERVER_ID);
    return oc;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public List<String> getAllAreasIds() {
    final List<String> list = new ArrayList<>();
    for (MonitoredDirectory m : monitoredList) {
      final String id = m.getId();
      list.add(id);
    }
    Collections.unmodifiableList(list);
    return list;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public DiskOccupation getAreaOccupation(final String id) {
    final MonitoredDirectory monitored = findMonitorFromId(id);
    if (monitored == null) {
      return null;
    }
    final DiskOccupation oc = getDiskOccupation(id);
    return oc;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DiskOccupation getSingleProjectOccupation(final Object ownerId,
    final String projName) {
    final String dirName = getPathFromOwnerAndProjectName(ownerId, projName);
    final File directory = new File(dirName);
    final DiskOccupation occupation = getDiskOccupationFromPath(directory);
    return occupation;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public double getUsedSpaceForProjectMb(final Object ownerId,
    final String projName) {
    final String dirName = getPathFromOwnerAndProjectName(ownerId, projName);
    return getDiskUsageFromPathMb(dirName);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public double getUsedSpaceForUserMb(final Object userId) {
    if (userId == null) {
      final String err = "null user detected!";
      throw new InvalidRequestException(err);
    }

    final String dirName = getUserDirectoryPath(userId);
    if (dirName == null) {
      final String msg = "Usurio " + userId + "no tem diretrio!";
      Server.logSevereMessage(msg);
      final String err = "User area not detected! " + userId;
      throw new InvalidRequestException(err);
    }
    final double usedMegaBytes = getDiskUsageFromPathMb(dirName);
    return usedMegaBytes;
  }

  /**
   * Consulta ao usurio.
   *
   * @param userId o identificador do usurio
   * @return o usurio
   */
  private User getUser(final Object userId) {
    try {
      final User user = User.getUser(userId);
      if (user == null) {
        final String msg = "Detectado usurio nulo para: " + userId;
        Server.logSevereMessage(msg);
      }
      return user;
    }
    catch (final Exception ue) {
      final String msg = "Falha na deteco de usurio!";
      Server.logSevereMessage(msg);
    }
    return null;
  }

  /**
   * Consuta ao diretrio do usurio para avaliao.
   *
   * @param userId identificador do usurio.
   *
   * @return String com o diretrio.
   */
  private String getUserDirectoryPath(final Object userId) {
    final User user = getUser(userId);
    if (user == null) {
      return null;
    }
    final ProjectService projectService = ProjectService.getInstance();
    final String repositoryPath = projectService.getProjectRepositoryPath();
    final String path = repositoryPath + File.separator + user.getId();
    return path;
  }

  /**
   * Consuta ao diretrio do usurio para avaliao.
   *
   * @param ownerId identificador do owner.
   * @param projName identificador do projeto
   *
   * @return String com o diretrio.
   */
  private String getUserProjectDirectoryPath(final Object ownerId, final String projName) {
    final String uPath = getUserDirectoryPath(ownerId);
    final String sep = File.separator;
    return uPath + sep + projName;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public Hashtable<Object, Double> getUsersList() {
    return mountUserHash();
  }

  /**
   * {@inheritDoc}
   *
   * Retorna sempre verdadeiro.
   */
  @Override
  protected boolean has2Update(final Object arg, final Object event) {
    return true;
  }

  /**
   * Inicializao do servio de gerncia de SGAs.
   *
   * @throws ServerException se ocorrer um erro na inicializao
   */
  @Override
  public void initService() throws ServerException {
    // Monitorao default: rea do servidor.
    final String serverPath = ".";
    addToMonitor(DIR_SERVER_ID, serverPath);

    // Monitorao default: rea de projetos.
    final ProjectService prjService = ProjectService.getInstance();
    final String projectPath = prjService.getProjectRepositoryPath();
    addToMonitor(DIR_PROJECT_ID, projectPath);

    // Monitorao default: rea de algoritmos.
    final AlgorithmService algService = AlgorithmService.getInstance();
    final String algorithPath = algService.getAlgorithmRepositoryPath();
    addToMonitor(DIR_ALGORITHM_ID, algorithPath);

    final String idPropName = "monitor.id";
    final List<String> ids = getStringListProperty(idPropName);
    final String pathPropName = "monitor.path";
    final List<String> paths = getStringListProperty(pathPropName);
    if (ids.size() != paths.size()) {
      final String prefix = "Tamanhos distintos nas propriedades: ";
      final String err = prefix + "[" + idPropName + "]/[" + pathPropName + "]";
      throw new ServerException(err);
    }

    final int size = ids.size();
    for (int i = 0; i < size; i++) {
      final String id = ids.get(i);
      final String path = paths.get(i);
      addToMonitor(id, path);
    }

    mailOnAlert = isMailOnAlertEnabled();
    if (mailOnAlert) {
      Server.logInfoMessage("Envio de email habilitado (para casos de alerta)");
    }

    final boolean auditData = this.isAuditEnabled();
    if (auditData) {
      createAuditThread();
      Server.logInfoMessage("Auditagem de ocupao de disco habilitada.");
    }
    else {
      auditThread = null;
      Server.logInfoMessage("Auditagem de ocupao de disco desabilitada.");
    }
  }

  /**
   * Montagem de uma hash de ocupao por usurio.
   *
   * @return uma hash table
   */
  private Hashtable<Object, Double> mountUserHash() {
    final Hashtable<Object, Double> hash = new Hashtable<>();
    try {
      final List<User> allUsers = User.getAllUsers();
      for (final User user : allUsers) {
        double mega;
        try {
          mega = getUsedSpaceForUserMb(user.getId());
        }
        catch (final Exception e) {
          mega = 0;
        }
        hash.put(user.getId(), mega);
      }
      return hash;
    }
    catch (final Exception e) {
      return null;
    }
  }

  /**
   * Notifica o administrador o estado de um file system (se necessrio)
   *
   * @param id identificador do file system.
   * @param occupation a ocupao do file system.
   */
  private void notifyAdminIfNeeded(final String id,
    final DiskOccupation occupation) {
    final NotificationService ntSrv = NotificationService.getInstance();
    final double percentage = occupation.getUsedSpacePerc();

    final String senderName = getSenderName();

    List<Object> adminIds = new ArrayList<>();
    try {
      adminIds.addAll(User.getAdminIds());
    }
    catch (Exception e) {
      Server.logSevereMessage(
        "Erro ao obter a lista de usurios adminstradores.", e);
    }
    Object[] ids = adminIds.toArray(new Object[0]);

    if (percentage >= alertPerc) {
      final FileSystemSpaceNotification data = new FileSystemSpaceNotification(
        senderName, id, percentage, alertPerc, false);

      ntSrv.notifyTo(ids, data);
      if (mailOnAlert) {
        final MailService mailSrv = MailService.getInstance();
        //TODO Notificar todo os usurios com permisso de admin
        mailSrv.mailUserFromService(this, ids, data.toString());
      }
    }
    else {
      if (percentage >= warningPerc) {
        final FileSystemSpaceNotification data =
          new FileSystemSpaceNotification(senderName, id, percentage,
            warningPerc, true);

        ntSrv.notifyTo(ids, data);
      }
    }
  }

  /**
   * Trmino do servio.
   */
  @Override
  public void shutdownService() {
    exitAuditThread = true;
    if (auditThread != null) {
      auditThread.interrupt();
    }
    auditThread = null;
  }

  /**
   * Construtor protegido.
   *
   * @throws ServerException se ocorrer um erro na inicializao.
   */
  protected DiskUsageService() throws ServerException {
    super(SERVICE_NAME);
  }

  /**
   * Retorna
   *
   * @return warningPerc
   */
  public int getWarningPerc() {
    return warningPerc;
  }

  /**
   * Retorna
   *
   * @return alertPerc
   */
  public int getAlertPerc() {
    return alertPerc;
  }

  /**
   * Verifica se a auditagem est habilitada
   *
   * @return {@code true} se a propriedade do servio est habilitada para
   *         auditagem ou {@code false} caso contrrio.
   */
  public boolean isAuditEnabled() {
    return getBooleanProperty("enableAudit");
  }

  /**
   * Verifica se o envio de emails est habilitado em caso de alerta de ocupao
   * do disco.
   *
   * @return {@code true} se a propriedade de envio de emails de alerta est
   *         habilitada ou {@code false} caso contrrio.
   */
  public boolean isMailOnAlertEnabled() {
    return getBooleanProperty("mailOnAlert");
  }

  /**
   * Obtm o valor de tempo configurado para o intervalo de auditagem de
   * ocupao do disco.
   *
   * @return intervalo (em horas) de auditagem de ocupao do disco.
   */
  public int getAuditIntervalHours() {
    return getIntProperty("auditIntervalHours");
  }
}
