/*
 * $Id: WIOService.java 173643 2016-05-18 18:12:35Z clinio $
 */
package csbase.server.services.wioservice;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;

import org.omg.PortableServer.POA;
import org.omg.PortableServer.Servant;

import csbase.logic.User;
import csbase.remote.WIOServiceInterface;
import csbase.server.Server;
import csbase.server.ServerEntryPointImpl;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.services.projectservice.ProjectService;
import csbase.server.services.wioservice.idl.AlgorithmInfo;
import csbase.server.services.wioservice.idl.AlgorithmParameter;
import csbase.server.services.wioservice.idl.WIOServiceException;

/**
 * A classe <code>WIOService</code> implementa o servio de acesso aos arquivos
 * de projeto do WebSintesi.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class WIOService extends Service implements WIOServiceInterface {
  /**
   * Nmero mximo de arquivos projetos.
   */
  private int maxNumProjects = -9;

  /**
   * Nmero mximo de arquivos abertos por projeto.
   */
  private int maxNumFiles = -9;

  /**
   * Tamanho mximo para read de arquivo remoto
   */
  private int maxReadKb = -9;

  /**
   * Tamanho mximo para write de arquivo remoto
   */
  private int maxWriteKb = -9;

  /**
   * Servant para a interface idl WIOFileSystem
   */
  private WIOServerFileSystem wioFileSystem = null;

  /**
   * Servidor CORBA responsvel pela interao do servio.
   */
  private ORBDealer orbDealer = null;

  /**
   * Objeto de auditagem.
   */
  WIOAudit audit = null;

  /**
   * Armazenamento da propriedade textual da porta do WIO.
   */
  private String serverPortText;

  /**
   * Constri a instncia do servio de gerncia de WIO
   * 
   * @throws ServerException se houver falha no servidor.
   */
  public static void createService() throws ServerException {
    new WIOService();
  }

  /**
   * Formatao de Mensagem de exceo para ser serializada por CORBA.
   * 
   * @param e a exceo Java
   * 
   * @return a string formatada.
   */
  public static String getExceptionString(final Exception e) {
    String err = e.getMessage();
    err = (err == null ? "<null>" : err);
    err = err.replaceAll("[^\\p{ASCII}]", "?");
    return err;
  }

  /**
   * Busca do servio de WIO.
   * 
   * @return o servio de WIO.
   */
  public static final WIOService getInstance() {
    return (WIOService) getInstance(SERVICE_NAME);
  }

  /**
   * Ativao de objeto CORBA.
   * 
   * @param id string de identificao.
   * @param o o servant CORBA.
   * 
   * @return o POA.
   * 
   * @throws WIOServiceException se houver exceo remota a ser lanada.
   */
  protected final POA activateCorbaObject(final String id, final Servant o)
    throws WIOServiceException {
    if (orbDealer == null) {
      final String err =
        "Sem ORB dealer na ativacao CORBA [" + o.toString() + "]";
      final WIOServiceException exception = new WIOServiceException(err);
      Server.logSevereMessage(err);
      throw exception;
    }
    return orbDealer.activateObject(id, o);
  }

  /**
   * Criao de thread de monitorao de projetos abertos.
   */
  private void createAuditThread() {
    /* Busca da propriedade que determina o intervalo de atualizao */
    final int FIVE_SECONDS = 5;
    final int ONE_HOUR = 60 * 60;
    int timeSec = getIntProperty("auditIntervalSeconds");
    if (timeSec < FIVE_SECONDS) {
      timeSec = FIVE_SECONDS;
    }
    if (timeSec > ONE_HOUR) {
      timeSec = ONE_HOUR;
    }
    Server.logInfoMessage("Intervalo de auditagem: " + timeSec + " s.");
    audit = new WIOAudit(this, timeSec * 1000);
  }

  /**
   * Desativao de objeto CORBA.
   * 
   * @param o o servant CORBA.
   * 
   * @throws WIOServiceException se houver exceo remota a ser lanada.
   */
  protected final void deactivateCorbaObject(final Servant o)
    throws WIOServiceException {
    if (orbDealer == null) {
      final String err = "Sem ORB na desativacao: [" + o.toString() + "]";
      final WIOServiceException exception = new WIOServiceException(err);
      Server.logSevereMessage(err);
      throw exception;
    }
    orbDealer.deactivateObject(o);
  }

  /**
   * Consulta ao tamanho mximo permitido ao servidor responder em uma leitura
   * de arquivo remoto.
   * 
   * @return o tamanho da leitura em bytes.
   */
  public final int getMaxReadKb() {
    return maxReadKb;
  }

  /**
   * Consulta ao tamanho mximo permitido ao servidor responder em uma escrita
   * de arquivo remoto.
   * 
   * @return o tamanho da leitura em bytes.
   */
  public final int getMaxWriteKb() {
    return maxWriteKb;
  }

  /**
   * Obtm o valor de uma propriedade como um nmero inteiro positivo.
   * 
   * @param propName nome da propriedade.
   * @return o valor ajustado.
   */
  private int getPositiveInt(final String propName) {
    final int prp = getIntProperty(propName);
    return Math.abs(prp);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected final boolean has2Update(final Object arg, final Object event) {
    return true;
  }

  /**
   * Inicializao do ORB do servio. Cria o servidor CORBA responsvel pela
   * interao com os clientes e ativa o objeto responsvel pela implementao
   * da interface
   * 
   * @throws ServerException se ocorrer um erro na criao do servidor CORBA ou
   *         na ativao do servant.
   */
  private void initORB() throws ServerException {
    try {
      Server.logInfoMessage("Acessando ambiente ORB...");
      final Properties props = getExternalPropertyFile("ORB.file");
      /*
       * As propriedades que definem o ip e porta so definidas nas propriedades
       * do servio.
       */
      if (!isPropertyNull("ORB.port")) {
        final int port = getIntProperty("ORB.port");
        final String portStr = Integer.toString(port);
        props.setProperty("OAPort", portStr);
      }
      if (!isPropertyNull("ORB.hostAddr")) {
        final String addr = getStringProperty("ORB.hostAddr");
        props.setProperty("OAIAddr", addr);
      }
      orbDealer = new ORBDealer(props);

      serverPortText = props.getProperty("OAPort");
      if (serverPortText == null) {
        final String err = "WIO/ORB no est com ajuste OAPort!";
        throw new ServerException(err);
      }
      serverPortText = serverPortText.trim();
      final String fmt = "Acesso ao ambiente WIO/ORB feito (porta '%s').";
      final String msg = String.format(fmt, serverPortText);
      Server.logInfoMessage(msg);
    }
    catch (final Exception e) {
      throw new ServerException("Impossvel criar servidor CORBA", e);
    }
    try {
      // Ativa o objeto CORBA que implementa a interface WIOFileSystem
      Server.logFineMessage("Instanciando FS remoto...");
      wioFileSystem = new WIOServerFileSystem(this);
      Server.logFineMessage("Ativando CORBA FS remoto...");
      orbDealer.activateObject("WIOFileSystem", wioFileSystem);
      Server.logFineMessage("FS remoto ativado!");
    }
    catch (final Exception e) {
      throw new ServerException("Impossvel criar handler para o FS remoto", e);
    }
  }

  /**
   * Inicializao do servio.
   * 
   * @throws ServerException se ocorrer um erro na inicializao
   */
  @Override
  public final void initService() throws ServerException {
    initORB();
    readConfiguration();
    createAuditThread();
  }

  /**
   * <p>
   * Executa o pr-login de um usurio no servidor CSBase e retorna uma URL para
   * acesso ao sistema com esse usurio.
   * </p>
   * <p>
   * Quando executar o sistema com essa URL de acesso, o projeto informado
   * (pertencente ao usurio) ser aberto e o algoritmo informado ser
   * executado.
   * </p>
   * 
   * @param login O login do usurio.
   * @param password A senha do usurio.
   * @param projectName O nome de um projeto a ser aberto ao se usar a URL de
   *        acesso.
   * @param algorithmInfo Dados sobre um algoritmo que ser executado ao se usar
   *        a URL de acesso.
   * 
   * @return Uma URL para acesso ao sistema.
   */
  public String preLogin2Web(final String login, final String password,
    final String projectName, final AlgorithmInfo algorithmInfo) {

    Server.logInfoMessage("Execuo WIO - usurio [" + 
        login + "] - projeto [" + projectName + "] - algoritmo: " + 
        algorithmInfo.algorithmName);

    final ServerEntryPointImpl entryPoint =
      (ServerEntryPointImpl) Server.getInstance().getEntryPoint();
    if (entryPoint == null) {
      return null;
    }

    final Object userId;
    try {
      User user = User.getUserByLogin(login);
      userId = user.getId();
    }
    catch (Exception e) {
      return null;
    }

    final Map<String, Serializable> attributes =
      new HashMap<String, Serializable>();

    // Ateno: projectId.toString() usado temporariamente para
    // mater compatibilidade.
    final ProjectService projectService = ProjectService.getInstance();
    final Object projectId = projectService.getProjectId(userId, projectName);
    attributes.put("project_name", projectId.toString());

    // Ajuste do algortimo e parmetros.
    attributes.put("algorithm_name", algorithmInfo.algorithmName);
    final Map<String, String> algorithmParameters =
      new HashMap<String, String>();
    for (final AlgorithmParameter parameter : algorithmInfo.parameters) {
      algorithmParameters.put(parameter.name, parameter.value);
    }
    attributes.put("algorithm_parameters", (Serializable) algorithmParameters);
    return entryPoint.preLogin2Web(login, password, new Locale("pt", "BR"),
      attributes);
  }

  /**
   * Leitura da configurao do servio.
   */
  private void readConfiguration() {
    maxWriteKb = getPositiveInt("maxWriteSizeKb");
    maxReadKb = getPositiveInt("maxReadSizeKb");
    maxNumFiles = getPositiveInt("maxOpenedFiles");
    maxNumProjects = getPositiveInt("maxOpenedProjects");

    Server.logInfoMessage("Tam. mximo de leitura: " + maxReadKb + " Kb");
    Server.logInfoMessage("Tam. mximo de escrita: " + maxWriteKb + " Kb");
    Server.logInfoMessage("# Max. de arquivos por projeto: " + maxNumFiles);
    Server.logInfoMessage("# Max de projetos: " + maxNumProjects);
  }

  /**
   * Trmino do servio que destri o servant que implementa a interface de
   * comunicao com os WIO, e desativa o servidor CORBA correspondente.
   * 
   * @throws ServerException se houver exceo no servidor.
   */
  @Override
  public final void shutdownService() throws ServerException {
    try {
      if (orbDealer != null) {
        orbDealer.shutdown();
      }
      wioFileSystem = null;
      orbDealer = null;
      audit.finish();
    }
    catch (final Exception e) {
      throw new ServerException("Impossvel desativar servidor CORBA", e);
    }
  }

  /**
   * Construtor padro de repasse do nome do servio.
   * 
   * @throws ServerException se houver falha no servidor.
   */
  protected WIOService() throws ServerException {
    super(SERVICE_NAME);
  }

  /**
   * Consulta  configurao: mximo de arquivos abertos
   * 
   * @return valor
   */
  int getMaxOpenedFiles() {
    return maxNumFiles;
  }

  /**
   * Consulta  configurao: mximo de projetos abertos
   * 
   * @return valor
   */
  int getMaxOpenedProjects() {
    return maxNumProjects;
  }

  /**
   * Consulta ao estado atual: nmero de arquivos abertos.
   * 
   * @return valor
   */
  int getNumOpenedFiles() {
    return wioFileSystem.getNumOpenedFiles();
  }

  /**
   * Consulta ao estado atual: nmero de projetos abertos.
   * 
   * @return valor
   */
  int getNumOpenedProjects() {
    return wioFileSystem.getNumOpenedProjects();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getServerPort() {
    return serverPortText;
  }
}
