package csbase.server.services.repositoryservice;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.channels.FileChannel;
import java.util.LinkedList;
import java.util.List;

import tecgraf.javautils.core.io.FileUtils;

/**
 * Arquivo localizado fisicamente na mquina do servidor.
 *
 * NOTA: Essa classe foi construda apenas para migrar os outros servios aos
 * poucos.
 *
 * @see LocalRepository
 *
 * @author Tecgraf/PUC-Rio
 */
class LocalTempFile implements IRepositoryFile {

  /** Arquivo. */
  private File file;

  /** Arquivo para acesso aleatrio. */
  private RandomAccessFile raf;

  /**
   * Construtor.
   *
   * @param file arquivo.
   */
  LocalTempFile(File file) {
    if (file == null) {
      throw new IllegalArgumentException("file no pode ser nulo.");
    }
    this.file = file;
  }

  /** {@inheritDoc} */
  @Override
  public String getName() {
    return file.getName();
  }

  /** {@inheritDoc} */
  @Override
  public List<IRepositoryFile> getChildren() {
    if (isRegularFile()) {
      return null;
    }
    List<IRepositoryFile> result = new LinkedList<IRepositoryFile>();
    for (File child : file.listFiles()) {
      result.add(new LocalTempFile(child));
    }
    return result;
  }

  /** {@inheritDoc} */
  @Override
  public IRepositoryFile getChild(String... path) {
    String filePath = FileUtils.joinPath(File.separatorChar, path);
    filePath = FileUtils.joinPath(File.separatorChar, getPath(), filePath);

    File file = new File(filePath);
    if (!file.exists()) {
      return null;
    }
    return new LocalTempFile(file);
  }

  /** {@inheritDoc} */
  @Override
  public boolean exists() {
    return file.exists();
  }

  /** {@inheritDoc} */
  @Override
  public long getModificationDate() {
    return file.lastModified();
  }

  /** {@inheritDoc} */
  @Override
  public IRepositoryFile getParent() {
    File parentFile = file.getParentFile();
    if (parentFile == null) {
      return null;
    }
    return new LocalTempFile(parentFile);
  }

  /** {@inheritDoc} */
  @Override
  public String getPath() {
    return file.getPath();
  }

  /** {@inheritDoc} */
  @Override
  public long size() {
    return file.length();
  }

  /** {@inheritDoc} */
  @Override
  public boolean isDirectory() {
    return file.isDirectory();
  }

  /** {@inheritDoc} */
  @Override
  public boolean isRegularFile() {
    return file.isFile();
  }

  /** {@inheritDoc} */
  @Override
  public boolean isHidden() {
    return file.isHidden();
  }

  /** {@inheritDoc} */
  @Override
  public boolean isSymbolicLink() {
    return false;
  }

  /** {@inheritDoc} */
  @Override
  public boolean isReadable() {
    return file.canRead();
  }

  /** {@inheritDoc} */
  @Override
  public boolean isWritable() {
    return file.canWrite();
  }

  /** {@inheritDoc} */
  @Override
  public boolean isExecutable() {
    return file.canExecute();
  }

  /** {@inheritDoc} */
  @Override
  public boolean lock() {
    throw new UnsupportedOperationException();
  }

  /** {@inheritDoc} */
  @Override
  public boolean release() {
    throw new UnsupportedOperationException();
  }

  /** {@inheritDoc} */
  @Override
  public boolean isLocked() {
    throw new UnsupportedOperationException();
  }

  /** {@inheritDoc} */
  @Override
  public InputStream getInputStream() throws IOException {
    return new FileInputStream(file);
  }

  /** {@inheritDoc} */
  @Override
  public OutputStream getOutputStream() throws IOException {
    if (isLocked()) {
      throw new IOException(String.format("Arquivo bloqueado: %s", this));
    }
    return new FileOutputStream(file);
  }

  /** {@inheritDoc} */
  @Override
  public FileChannel getFileChannel(String mode) throws IOException {
    if (raf == null) {
      raf = new RandomAccessFile(file, mode);
    }
    return raf.getChannel();
  }

  /** {@inheritDoc} */
  @Override
  public void close() throws IOException {
    if (raf != null) {
      raf.close();
    }
    raf = null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getFullPath() {
    return file.getAbsolutePath();
  }
}
