/*
 * FileSystem.java
 *
 * $Id: FileSystem.java 178921 2017-01-25 17:23:23Z fpina $
 */
package csbase.server;

import java.io.File;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.EnumSet;
import java.nio.file.FileStore;
import java.util.StringTokenizer;

import tecgraf.javautils.core.externalcommand.ExternalCommand;

/**
 * Representa o sistema de arquivos do SSI.
 */
public class FileSystem {
  /**
   * Diretrio raiz dos arquivos de propriedades.
   */
  public static final String PROPERTIES_DIR = "properties";

  /**
   * Verifica o espao disponvel para um determinado filesystem. O mtodo
   * File.getUsableSpace no funciona para filesystems maiores de 5Tb (bug
   * cadastrado na Sun). Para sistemas nessa situao, a propriedade
   * FreeDiskSpace.group deve ser definida no arquivo Server.properties de forma
   * que o espao em disco seja obtido atravs de uma chamada direta ao sistema.
   *
   * @param path O caminho do filesystem.
   * @return O nmero de bytes disponveis (nmero negativo em caso de erro).
   */
  public static long getFreeSpace(String path) {
    try {
      FileStore store = Files.getFileStore(Paths.get(path));
      return store.getUsableSpace();
    }
    catch (Exception e) {
      final String fmt = "Erro obtendo espao livre de disco em: %s";
      final String err = String.format(fmt, path);
      Server.logSevereMessage(err, e);
      return -1;
    }
  }

  /**
   * Verifica o espao total para um determinado filesystem. O mtodo
   * File.getTotalSpace no funciona para filesystems maiores de 5Tb (bug
   * cadastrado na Sun). Para sistemas nessa situao, a propriedade
   * TotalDiskSpace.group deve ser definida no arquivo Server.properties de
   * forma que o espao em disco seja obtido atravs de uma chamada direta ao
   * sistema.
   *
   * @param path O caminho do filesystem.
   * @return O nmero de bytes disponveis (nmero negativo em caso de erro).
   */
  public static long getTotalSpace(String path) {
    try {
      FileStore store = Files.getFileStore(Paths.get(path));
      return store.getTotalSpace();
    }
    catch (Exception e) {
      final String fmt = "Erro obtendo espao total de disco em: %s";
      final String err = String.format(fmt, path);
      Server.logSevereMessage(err, e);
      return -1;
    }
  }

  /**
   * Move um arquivo para o outro. A implementao deste mtodo copia e depois
   * remove o original.
   *
   * @param from O arquivo de origem.
   * @param to O arquivo destino.
   * @return Verdadeiro se foi movido com sucesso ou falso, caso contrrio.
   */
  public static boolean moveRegularFile(File from, File to) {
    Server.logInfoMessage("FileSystem.moveRegularFile:  de: " + from
      .getAbsolutePath() + "  para: " + to.getAbsolutePath());

    if (!from.exists()) {
      Server.logSevereMessage(
        "FileSystem.moveRegularFile: arquivo origem no existe: " + from
          .getAbsolutePath());
      return false;
    }
    if (!copyFile(from, to)) {
      return false;
    }
    if (from.delete()) {
      return true;
    }
    Server.logSevereMessage(
      "FileSystem.moveRegularFile: Erro removendo arquivo origem: " + from
        .getAbsolutePath());
    return false;
  }

  /**
   * Move arquivos ou diretrios.
   *
   * @param from arquivo ou diretrio de origem
   * @param to arquivo ou diretrio de destino
   * @return Verdadeiro se a movimentao foi realizada com sucesso, Falso caso
   *         contrrio
   */
  public static boolean move(File from, File to) {
    if (!from.isDirectory()) {
      return moveRegularFile(from, to);
    }
    if (!to.mkdirs()) {
      Server.logSevereMessage("FileSystem.move: Erro na criando diretrio(s) : "
        + to.getAbsolutePath());
      return false;
    }
    File[] children = from.listFiles();
    boolean success = true;
    for (int i = 0; i < children.length; i++) {
      File newChildFile = new File(to, children[i].getName());
      success = move(children[i], newChildFile);
      if (!success) {
        break;
      }
    }
    if (success) {
      if (from.delete()) {
        return true;
      }
      Server.logSevereMessage(
        "FileSystem.move: Erro removendo diretrio origem: " + from
          .getAbsolutePath());
    }
    return success;
  }

  /**
   * Copia o contedo de um arquivo para outro.
   *
   * @param from O arquivo de origem.
   * @param to O arquivo destino.
   *
   * @return Verdadeiro se a cpia foi feita com sucesso ou falso, caso
   *         contrrio.
   */
  public static boolean copyFile(File from, File to) {
    Path source = Paths.get(from.getAbsolutePath());
    Path target = Paths.get(to.getAbsolutePath());
    Server.logInfoMessage(String.format(
      "FileSystem.copyFile: de '%s' para '%s'", source, target));
    String cmd = String.format(
      "FileSystem:copyFile: Erro copiando arquivo (%s %s)", source, target);
    try {
      Files.copy(source, target, StandardCopyOption.REPLACE_EXISTING);
    }
    catch (IOException e) {
      Server.logSevereMessage(cmd, e);
      return false;
    }
    return true;
  }

  /**
   * Copia o contedo de um diretrio para outro.
   * 
   * @param from O diretrio de origem.
   * @param to O diretrio destino.
   * 
   * @return Verdadeiro se a cpia foi feita com sucesso ou falso, caso
   *         contrrio.
   */
  public static boolean copyDir(File from, File to) {
    Path source = Paths.get(from.getAbsolutePath());
    Path target = Paths.get(to.getAbsolutePath());
    Server.logInfoMessage(String.format("FileSystem.copyDir: de '%s' para '%s'",
      source, target));
    String cmd = String.format(
      "FileSystem:copyDir: Erro copiando diretrio (%s %s)", source, target);
    try {
      Files.walkFileTree(source, EnumSet.of(FileVisitOption.FOLLOW_LINKS),
        Integer.MAX_VALUE, new CopyDirectory(source, target));
    }
    catch (IOException e) {
      Server.logSevereMessage(cmd, e);
      return false;
    }
    return true;
  }

  /**
   * Remove um arquivo ou diretrio. No caso de diretrio, remove recursivamente
   * o seu contedo.
   * 
   * @param file arquivo/diretrio a remover
   * @param removeRoot flag que indica se um diretrio deve ser removido -
   *        <code>true</code> ou apenas o seu contedo.
   * 
   * @throws Exception caso no tenha sido possvel remover algum
   *         arquivo/diretrio. A mensagem da exceo indica qual o elemento no
   *         removido. Se este elemento est contido num diretrio a ser
   *         removido, a operao  abortada
   */
  public static void removeFile(File file, boolean removeRoot)
    throws Exception {
    if (!file.exists()) {
      return;
    }
    if (file.isDirectory()) {
      File[] children = file.listFiles();
      if (!(children == null) && (children.length > 0)) {
        for (int i = 0; i < children.length; i++) {
          removeFile(children[i], true);
        }
      }
    }
    if (removeRoot) {
      if (!file.delete()) {
        throw new Exception("Falha na remoo de " + file.getPath());
      }
    }
  }

  /**
   * <p>
   * Habilita permisso de execuo para o arquivo especificado.
   * </p>
   * Este mtodo  especfico para cada sistema operacional, e portanto usa a
   * constante <code>ENABLE_EXECUTION_PERMISSION_COMMAND</code> para descobrir
   * qual o comando a ser executado, a qual por sua vez faz referncia a uma
   * propriedade em arquivo.<br>
   * Caso esse mtodo no faa sentido para o sistema operacional operacional
   * corrente, deixe essa propriedade em branco.<br>
   * OBS: O mtodo  bloqueante, isto , a thread de execuo ir aguardar at
   * que o processo termine para retornar.
   *
   * @param filePath caminho para o arquivo no NFS.
   *
   * @return <tt>true</tt> se a operao foi bem sucedida.
   *
   * @throws ServerException caso no seja possvel executar o comando.
   */
  public static final boolean enableExecutionPermission(String filePath)
    throws ServerException {

    if (filePath == null) {
      throw new IllegalArgumentException("filePath == null");
    }
    if (filePath.equals("")) {
      throw new IllegalArgumentException("filePath est vazia");
    }

    try {
      return new File(filePath).setExecutable(true, false);
    }
    catch (Exception e) {
      // Espera no deve ser interrompida...
      throw new ServerException("Erro ao mudar execuo do arquivo '" + filePath
        + "'", e);
    }
  }

  /**
   * Cria um array de strings para ser usado pelo comando
   * {@link Runtime#exec(String[])}.
   * <p>
   * Esta deve ser a forma prefervel de se criar linhas de comando ao invs da
   * criao da linha de comando como uma string, pois se fose feito desta forma
   * (uma nica string) o mtodo {@link Runtime#exec(String)} usaria
   * {@link StringTokenizer} para identificar os tokens, e espaos nos
   * parmetros gerariam tokens indesejados. Envolver os parmetros com aspas
   * no d o resultado esperado (no UNIX p.ex. as aspas so repassadas para o
   * comando, o que gera erro).
   *
   * @param cmd - comando. Pode conter espaos; neste caso, o comando ser
   *        quebrado em tokens usando {@link StringTokenizer}. O primeiro token
   *        ser usado como comando, e os demais como parmetros, que sero
   *        inseridos antes dos especificados pelo parmetro <code>args</code>.
   * @param args - parmetros
   * @return array de strings ({cmd, args[0], args[1], ...})
   */
  private static String[] createCommandArray(String cmd, String... args) {
    final StringTokenizer cmdTokenizer = new StringTokenizer(cmd);
    final int numTokens = cmdTokenizer.countTokens();
    final String[] cmdArray = new String[args.length + numTokens];
    int i = 0;
    while (i < numTokens) {
      cmdArray[i++] = cmdTokenizer.nextToken();
    }
    System.arraycopy(args, 0, cmdArray, i, args.length);
    return cmdArray;
  }

  /**
   * Faz a cpia de um diretrio.
   *
   * @author Tecgraf/PUC-Rio
   */
  public static class CopyDirectory extends SimpleFileVisitor<Path> {
    /** Diretrio de origem */
    private Path source;
    /** Diretrio de origem */
    private Path target;

    /**
     * Copia de diretrio.
     * 
     * @param source Diretrio origem
     * @param target Diretrio destino
     */
    public CopyDirectory(Path source, Path target) {
      this.source = source;
      this.target = target;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public FileVisitResult visitFile(Path file, BasicFileAttributes attributes)
      throws IOException {
      Server.logFineMessage("Copying " + source.relativize(file));
      Files.copy(file, target.resolve(source.relativize(file)));
      return FileVisitResult.CONTINUE;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public FileVisitResult preVisitDirectory(Path directory,
      BasicFileAttributes attributes) throws IOException {
      Path targetDirectory = target.resolve(source.relativize(directory));
      try {
        Server.logFineMessage("Copying " + source.relativize(directory));
        Files.copy(directory, targetDirectory);
      }
      catch (FileAlreadyExistsException e) {
        if (!Files.isDirectory(targetDirectory)) {
          throw e;
        }
      }
      return FileVisitResult.CONTINUE;
    }
  }
}
