package csbase.server.services.filetransferservice;

import csbase.logic.filetransferservice.FileTransferRequest;
import csbase.server.Server;

/**
 * Thread de monitorao de requisies pendentes.
 * 
 * @author Tecgraf/PUC-Rio
 */
final public class FileTransferServiceMonitorThread extends Thread {

  /**
   * Servio.
   */
  final private FileTransferService service;

  /**
   * {@inheritDoc}
   */
  @Override
  final public void run() {
    final int maxParallel = service.getIntProperty("max.parallel.transfers");
    Server.logInfoMessage("Configurao de paralelismo: " + maxParallel);
    final int checkTimeSec = 5 * 1000;
    while (!isInterrupted()) {
      try {
        Thread.sleep(checkTimeSec);
        final FileTransferRequestQueue queue = service.getRequestQueue();
        int nRuns = queue.getNumRunningRequests();
        while (nRuns < maxParallel) {
          final FileTransferRequest pck = queue.pickWaitingRequest();
          // No havendo nada na fila, podemos sair do loop interno.
          if (pck == null) {
            break;
          }
          // Lanando novo request para execuo.
          service.startRequest(pck);

          // Dormindo um pouco para no sobrecarregar o servio.
          final int ONE_SEC = 1 * 1000;
          Thread.sleep(ONE_SEC);
          nRuns = queue.getNumRunningRequests();
        }
      }
      catch (final InterruptedException e) {
        final String msg = "Detectada interrupo da thread de monitorao";
        Server.logInfoMessage(msg);
      }
      catch (final Exception e) {
        final String msg = "Exceo no servio (thread de monitorao)";
        Server.logSevereMessage(msg, e);
      }
    }
    Server.logInfoMessage("Fim da thread de monitorao");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public void interrupt() {
    Server.logInfoMessage("Interrompendo thread de monitorao");
    super.interrupt();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public synchronized void start() {
    Server.logInfoMessage("Lanando thread de monitorao");
    super.start();
  }

  /**
   * Construtor
   * 
   * @param service o servio.
   */
  FileTransferServiceMonitorThread(final FileTransferService service) {
    this.service = service;
    this.setName(service.getName() + "::MonitorThread");
  }
}
