package csbase.server.services.opendreamsservice.opendreams.v2_0;

import java.rmi.RemoteException;
import java.util.Arrays;
import java.util.List;

import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.Option;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.openbus.opendreams.v2_0.OpenDreamsJobTemplate;
import csbase.exception.ParseException;
import csbase.exception.PermissionException;
import csbase.exception.algorithms.AlgorithmNotFoundException;
import csbase.exception.algorithms.ParameterNotFoundException;
import csbase.logic.Priority;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.validation.LocalizedMessage;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.ValidationMode;
import csbase.server.Service;
import csbase.server.services.administrationservice.AdministrationService;
import csbase.server.services.algorithmservice.AlgorithmService;
import csbase.server.services.projectservice.ProjectService;

/**
 * Template que descreve um <code>OpenDreamsJobTemplate</code> para execuo de
 * algoritmos no opendreams.  criado a partir de um job template.
 *
 * @author Tecgraf PUC-Rio
 *
 */
public class JobTemplateValidator {

	/** O nome do algoritmo em args */
	@Option(name = "-name")
	private String algoName;

	/** A verso do algoritmo em args */
	@Option(name = "-version")
	private String algoVersion;

	/** O configurador do algoritmo descrito pelo job template */
	private AlgorithmConfigurator configurator;

	/** O identificador da verso do algoritmo */
	private AlgorithmVersionId versionId;

	/** O identificador projeto sobre o qual esse job template atua */
	private Object projectId;

	/** Identificador do usurio. */
	private String userId;

	/** Comando para execuo de algoritmos simples */
	private static String EXEC_ALGO = "execAlgo";

	/** Comando para execuo de fluxo de algoritmos */
	private static String EXEC_FLOW = "execFlow";

	/** Comandos vlidos */
	private static String[] VALID_COMMANDS = new String[]{EXEC_ALGO, EXEC_FLOW};

	/** Categorias de jobs vlidas */
	private static String[] VALID_JOB_CATEGORIES = new String[]{"CSBase", "System"};

	/**
	 * Constri um validador de um <code>OpenDreamsJobTemplate</code>
	 *
	 * @param userId
	 *            o identificador do usuario.
	 * @param jt
	 *            o job template a ser validado
	 * @throws InvalidJobTemplateException
	 *             se houver uma falha na validao do job template
	 */
	public JobTemplateValidator(String userId, OpenDreamsJobTemplate jt) throws InvalidJobTemplateException {
		this.userId = userId;

		validateProject(jt);
		validateArgs(jt);
		validateRemoteCommand(jt);
		validateJobParameters(jt);
		validateJobPriority(jt);
		validateJobEnvironment(jt);
		validateWorkingDirectory(jt);
		validateJobCategory(jt);
		validateEmail(jt);
		validateBlockEmail(jt);
		validateJobName(jt);
		validateInputPath(jt);
		validateOutputPath(jt);
		validateErrorPath(jt);
		validateJoinFiles(jt);
		validateTransferFiles(jt);
	}

	/**
	 * @return project id.
	 */
	public Object getProjectId() {
		return projectId;
	}

	/**
	 * @param jt
	 * @throws InvalidJobTemplateException
	 */
	private void validateProject(OpenDreamsJobTemplate jt) throws InvalidJobTemplateException {
		if (jt.projectName.isEmpty()) {
			throw new InvalidJobTemplateException("O nome do projeto no foi especificado.");
		}
		String projectOwner = jt.projectOwner.isEmpty() ? userId : jt.projectOwner;
		checkProject(jt.projectName, projectOwner);
		checkProjectPermission(jt.projectName, userId);
		ProjectService projectService = ProjectService.getInstance();
		projectId = projectService.getProjectId(projectOwner, jt.projectName);

	}

	/**
	 * Verifica se o usurio possui um projeto cujo nome  passado como
	 * parmetro.
	 *
	 * @param projectName
	 *            nome do projeto
	 * @param userId
	 *            identificador do usurio
	 * @throws InvalidJobTemplateException
	 *             caso no exista um projeto com o identificador passado como
	 *             parmetro
	 */
	private void checkProject(String projectName, String userId) throws InvalidJobTemplateException {
		if (AdministrationService.getInstance().getUser(userId) == null) {
			throw new InvalidJobTemplateException("No existe um usurio com identificador " + userId);
		}
		ProjectService projectService = ProjectService.getInstance();
		Object projectId = projectService.getProjectId(userId, projectName);
		if (!projectService.existsProject(projectId)) {
			throw new InvalidJobTemplateException(
					"O usurio " + userId + " no possui um projeto com nome " + projectName);
		}
	}

	/**
	 * Verifica se o usurio da credencial possui acesso um projeto de um outro
	 * usurio.
	 *
	 * @param projectName
	 *            nome do projeto
	 * @param userId
	 *            identificador do usurio dono do projeto
	 * @throws InvalidJobTemplateException
	 *             caso no exista um usurio com o identificador
	 */
	public static void checkProjectPermission(String projectName, String userId) throws InvalidJobTemplateException {
		ProjectService projectService = ProjectService.getInstance();
		Object projectId = projectService.getProjectId(userId, projectName);
		try {
			projectService.checkWritePermission(projectId);
		} catch (PermissionException e) {
			throw new InvalidJobTemplateException("O usurio " + Service.getUser().getId()
					+ " no possui acesso de escrita no projeto " + projectName + " do usurio " + userId);
		}
	}

	/**
	 * Faz a validao do atributo <b>remoteCommand</b> do job template. Os
	 * seguintes valores so vlidos:
	 * <ul>
	 * <li><i>execAlgo</i>: executa um algoritmo.
	 * <li><i>execFlow</i>: executa um fluxo de algoritmos.
	 * </ul>
	 *
	 * @param jt
	 *            o job template que possui os argumentos a serem validados
	 * @throws InvalidJobTemplateException
	 *             se houver uma falha na validao do job template
	 */
	private void validateRemoteCommand(OpenDreamsJobTemplate jt) throws InvalidJobTemplateException {
		if (jt.remoteCommand.isEmpty()) {
			switch (configurator.getConfiguratorType()) {
				case SIMPLE :
					jt.remoteCommand = EXEC_ALGO;
					break;
				case FLOW :
					jt.remoteCommand = EXEC_FLOW;
					break;
			}
			return;
		}
		for (String command : VALID_COMMANDS) {
			if (jt.remoteCommand.equals(command)) {
				return;
			}
		}
		throw new InvalidJobTemplateException("Falha na validao de remoteCommand:" + jt.remoteCommand);
	}

	/**
	 * Faz a validao do atributo <b>args</b> do job template.
	 *
	 * @param jt
	 *            o job template que possui os argumentos a serem validados
	 * @throws InvalidJobTemplateException
	 *             se houver uma falha na validao do job template
	 */
	private void validateArgs(OpenDreamsJobTemplate jt) throws InvalidJobTemplateException {
		CmdLineParser parser = new CmdLineParser(this);
		try {
			parser.parseArgument(jt.args);
			this.versionId = AlgorithmVersionId.create(this.algoVersion);
			this.configurator = AlgorithmService.getInstance().createAlgorithmConfigurator(getAlgoName(),
					getAlgoVersionId());
		} catch (CmdLineException e) {
			StringBuffer args = new StringBuffer();
			for (String arg : jt.args) {
				args.append(arg);
				args.append(" ");
			}
			throw new InvalidJobTemplateException("Falha na validao de args:" + args, e);
		} catch (AlgorithmNotFoundException e) {
			throw new InvalidJobTemplateException(
					"Algoritmo " + getAlgoName() + " verso " + getAlgoVersionId() + " no encontrados", e);
		}
	}

	/**
	 * Valida os parmetros para execuo do algoritmo.
	 *
	 * @param jt
	 *            o job template que possui os argumentos a serem validados
	 * @throws InvalidJobTemplateException
	 *             se houver uma falha na validao do job template
	 */
	private void validateJobParameters(OpenDreamsJobTemplate jt) throws InvalidJobTemplateException {
		StringBuffer pars = new StringBuffer();
		for (String[] parameter : jt.jobParameters) {
			try {
				String name = parameter[0];
				String value = parameter[1];
				pars.append(name + " " + value + " ");
				configurator.setParameterValue(name, value);
			} catch (ParseException e) {
				throw new InvalidJobTemplateException("O parmetro " + pars + " possui o valor com formato invlido",
						e);
			} catch (ParameterNotFoundException e) {
				throw new InvalidJobTemplateException(
						"O parmetro " + pars + " no existe na configurao do algoritmo", e);
			}
		}
		try {
			ValidationContext context = new ValidationContext(ValidationMode.FULL, projectId, userId);
			Validation result = configurator.validate(context);
			if (!result.isWellSucceeded()) {
				List<LocalizedMessage> message = result.getMessage();
				throw new InvalidJobTemplateException("Validao dos parmetros do algoritmo falhou: " + message);
			}
		} catch (RemoteException e) {
			throw new InvalidJobTemplateException("No foi possvel validar os parmetros do algoritmo", e);
		}
	}

	/**
	 * Valida o atributo <code>jobPriority</code> do job template.
	 *
	 * @param jt
	 *            o job template que possui os argumentos a serem validados
	 * @throws InvalidJobTemplateException
	 *             se houver uma falha na validao do job template
	 */
	private void validateJobPriority(OpenDreamsJobTemplate jt) throws InvalidJobTemplateException {
		if (jt.priority < 0 || jt.priority >= Priority.values().length) {
			throw new InvalidJobTemplateException("O atributo jobPriority deve ter um dos seguintes valores: 0 (ROOT), "
					+ "1 (HIGH), 2 (MEDIUM) ou 3 (LOW)");
		}
	}

	/**
	 * Valida o atributo <code>jobEnvironment</code> do job template. O valor
	 * precisa estar entre zero (ROOT) e o 3 (LOW) que so os nmeros possveis
	 * definidos em {@link Priority}.
	 *
	 * @param jt
	 *            o job template que possui os argumentos a serem validados
	 * @throws InvalidJobTemplateException
	 *             se houver uma falha na validao do job template
	 */
	private void validateJobEnvironment(OpenDreamsJobTemplate jt) throws InvalidJobTemplateException {
		if (jt.priority < 0 || jt.priority >= Priority.values().length) {
			throw new InvalidJobTemplateException("O atributo jobPriority deve ser maior ou igual a"
					+ " 0 (prioridade ROOT) e menor ou igual a 3 (prioridade LOW)");
		}
	}

	/**
	 * Valida o atributo <code>workingDirectory</code> do job template. Esse
	 * atributo no  vlido no OpenDreams e, portanto, no deve possui nenhum
	 * valor.
	 *
	 * @param jt
	 *            o job template que possui os argumentos a serem validados
	 * @throws InvalidJobTemplateException
	 *             se houver uma falha na validao do job template
	 */
	private void validateWorkingDirectory(OpenDreamsJobTemplate jt) throws InvalidJobTemplateException {
		if (jt.workingDirectory != null && !jt.workingDirectory.isEmpty()) {
			throw new InvalidJobTemplateException(
					"O atributo workingDirectory no  " + "implementado pelo OpenDreams");
		}
	}

	/**
	 * Valida o atributo <code>jobCategory</code> do job template. No
	 * OpenDreams, os seguintes valores sero vlidos: CSBase ou System. A
	 * categoria CSBase identifica a execuo de algoritmos do repositrio. A
	 * categoria System serviria para atender a execuo de comandos do sistema,
	 * de acordo com o uso comum do DRMAA. Por enquanto, o OpenDreams somente
	 * reconhece a categoria CSBase.
	 *
	 * @param jt
	 *            o job template que possui os argumentos a serem validados
	 * @throws InvalidJobTemplateException
	 *             se houver uma falha na validao do job template
	 */
	private void validateJobCategory(OpenDreamsJobTemplate jt) throws InvalidJobTemplateException {
		if (jt.jobCategory != null && !Arrays.asList(VALID_JOB_CATEGORIES).contains(jt.jobCategory)) {
			throw new InvalidJobTemplateException("O atributo jobCategory deve ser CSBase ou System");
		}
		if (jt.jobCategory != null && jt.jobCategory.equals("System")) {
			throw new InvalidJobTemplateException("A categoria System ainda no est implementada no OpenDreams");
		}
	}

	/**
	 * Valida o atributo <code>email</code> do job template. Esse atributo
	 * possui a lista de emails para notificar o trmino de execuo dos jobs
	 *
	 * @param jt
	 *            o job template que possui os argumentos a serem validados
	 * @throws InvalidJobTemplateException
	 *             se houver uma falha na validao do job template
	 */
	private void validateEmail(OpenDreamsJobTemplate jt) throws InvalidJobTemplateException {
	}

	/**
	 * Valida o atributo <code>blockEmail</code> do job template. Esse atributo
	 * indica o bloqueio ou no no envio de emails sobre o estado (trmino) dos
	 * jobs.
	 *
	 * @param jt
	 *            o job template que possui os argumentos a serem validados
	 * @throws InvalidJobTemplateException
	 *             se houver uma falha na validao do job template
	 */
	private void validateBlockEmail(OpenDreamsJobTemplate jt) throws InvalidJobTemplateException {
	}

	/**
	 * Valida o atributo <code>jobName</code> do job template. Esse atributo no
	 *  vlido no OpenDreams e, portanto, no deve possui nenhum valor.
	 *
	 * @param jt
	 *            o job template que possui os argumentos a serem validados
	 * @throws InvalidJobTemplateException
	 *             se houver uma falha na validao do job template
	 */
	private void validateJobName(OpenDreamsJobTemplate jt) throws InvalidJobTemplateException {
		if (jt.jobName != null && !jt.jobName.isEmpty()) {
			throw new InvalidJobTemplateException("O atributo jobName no  " + "implementado pelo OpenDreams");
		}
	}

	/**
	 * Valida o atributo <code>inputPath</code> do job template. Esse atributo
	 * no  vlido no OpenDreams e, portanto, no deve possui nenhum valor.
	 *
	 * @param jt
	 *            o job template que possui os argumentos a serem validados
	 * @throws InvalidJobTemplateException
	 *             se houver uma falha na validao do job template
	 */
	private void validateInputPath(OpenDreamsJobTemplate jt) throws InvalidJobTemplateException {
		if (jt.inputPath != null && !jt.inputPath.isEmpty()) {
			throw new InvalidJobTemplateException("O atributo inputPath no  " + "implementado pelo OpenDreams");
		}
	}

	/**
	 * Valida o atributo <code>outputPath</code> do job template.
	 *
	 * @param jt
	 *            o job template que possui os argumentos a serem validados
	 * @throws InvalidJobTemplateException
	 *             se houver uma falha na validao do job template
	 */
	private void validateOutputPath(OpenDreamsJobTemplate jt) throws InvalidJobTemplateException {
		String dirPath = FileUtils.getFilePath(jt.outputPath);
		String[] paths = FileUtils.splitPath(dirPath);
		if (paths.length > 0) {
			if (!ProjectService.getInstance().existsFile(projectId, paths)) {
				throw new InvalidJobTemplateException("O diretrio " + dirPath + " no existe no projeto");
			}
		}
	}

	/**
	 * Valida o atributo <code>errorPath</code> do job template.
	 *
	 * @param jt
	 *            o job template que possui os argumentos a serem validados
	 * @throws InvalidJobTemplateException
	 *             se houver uma falha na validao do job template
	 */
	private void validateErrorPath(OpenDreamsJobTemplate jt) throws InvalidJobTemplateException {
		if (jt.errorPath != null && !jt.errorPath.isEmpty()) {
			throw new InvalidJobTemplateException("O atributo errorPath no  " + "implementado pelo OpenDreams");
		}
	}

	/**
	 * Valida o atributo <code>joinFiles</code> do job template.
	 *
	 * @param jt
	 *            o job template que possui os argumentos a serem validados
	 * @throws InvalidJobTemplateException
	 *             se houver uma falha na validao do job template
	 */
	private void validateJoinFiles(OpenDreamsJobTemplate jt) throws InvalidJobTemplateException {
	}

	/**
	 * Valida o atributo <code>transferFiles</code> do job template.
	 *
	 * @param jt
	 *            o job template que possui os argumentos a serem validados
	 * @throws InvalidJobTemplateException
	 *             se houver uma falha na validao do job template
	 */
	private void validateTransferFiles(OpenDreamsJobTemplate jt) throws InvalidJobTemplateException {
	}

	/**
	 * Obtm o nome do algoritmo, proveniente do argumento -name em args.
	 *
	 * @return o nome do algoritmo
	 */
	public String getAlgoName() {
		return this.algoName;
	}

	/**
	 * Obtm a verso do algoritmo, proveniente do argumento -version em args.
	 *
	 * @return a verso do algoritmo
	 */
	public AlgorithmVersionId getAlgoVersionId() {
		return this.versionId;
	}

	/**
	 * Obtm o configurador do algoritmo correspondente ao job template.
	 *
	 * @return o configurador do algoritmo criado a partir do job template.
	 */
	public AlgorithmConfigurator getAlgorithmConfigurator() {
		return this.configurator;
	}
}
