package csbase.server.services.restservice.websocket;

import csbase.server.services.restservice.RestService;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Map;

/**
 * Class representing a general WebSocket message.
 */
public class Message extends JSONObject {

	// Internationalization keys
	private static final String LOGIN_MESSAGE_KEY = "RestService.websocket.login";
	private static final String LOGOUT_MESSAGE_KEY = "RestService.websocket.logout";

	// Chat message types
	public static final String TYPE_CHAT_MESSAGE = "text";
	public static final String TYPE_USERLIST_UPDATE = "ulupd";
  public static final String TYPE_PROJECT_CHANGE = "prchg";
	public static final String TYPE_CHAT_HISTORY = "chist";

  // Notification message types
  public static final String TYPE_NOTIFICATION_HISTORY = "nhist";
	public static final String TYPE_COMMAND_TERMINATE = "cmdtmt";

	// Attrs
	private static final String TYPE = "type";
	private static final String USER = "user";
	private static final String CONTENT = "content";
	private static final String TIMESTAMP = "timestamp";

	public Message() {
		super();
		this.setTimestamp(System.currentTimeMillis());
	}

	public Message(String jsonString) throws JSONException {
		super(jsonString);

		// Instantiate User object
		if (this.has(USER))
			this.setUser(new User(this.getJSONObject(USER)));

		// Handle special characters in chat messages
		if (getType().equals(TYPE_CHAT_MESSAGE))
			setContent(escape(this.getString(CONTENT)));

		// Set creation timestamp
		this.setTimestamp(System.currentTimeMillis());
	}

	public Boolean isChat() {
		return getType().equals(TYPE_CHAT_MESSAGE);
	}

	public Boolean isProjectChange() {
		return getType().equals(TYPE_PROJECT_CHANGE);
	}

	public String getType() {
		return this.getString(TYPE);
	}

	public void setType(String type) {
		this.put(TYPE, type);
	}

	public User getUser() {
		return (User) this.get(USER);
	}

	public void setUser(User user) {
		this.put(USER, user);
	}

	public Object getContent() {
		return this.opt(CONTENT);
	}

	public void setContent(Object content) {
		this.put(CONTENT, content);
	}

	public long getTimestamp() {
		return this.getLong(TIMESTAMP);
	}

	public void setTimestamp(long timestamp) {
		this.put(TIMESTAMP, timestamp);
	}

	public static String escape(String orig) {
		StringBuilder buffer = new StringBuilder(orig.length());

		for (int i = 0; i < orig.length(); i++) {
			char c = orig.charAt(i);
			switch (c) {
				case '\n' :
					buffer.append("<br />");
					break;
				case '\r' :
					// ignore
					break;
				case '<' :
					buffer.append("&lt;");
					break;
				case '>' :
					buffer.append("&gt;");
					break;
				case '&' :
					buffer.append("&amp;");
					break;
				default :
					buffer.append(c);
			}
		}

		return buffer.toString();
	}

	public static class SystemMessage extends Message {

		private static final User SYSTEM_USER = new User().setLogin("system");

		public SystemMessage(String chatMessage) {
			super();
			this.setType(TYPE_CHAT_MESSAGE);
			this.setUser(SYSTEM_USER);
			this.setContent(chatMessage);
		}

		public static Message login(String username) {
			return new SystemMessage(
					RestService.getInstance().getFormattedString(LOGIN_MESSAGE_KEY, new Object[]{username}));
		}

		public static Message logout(String username) {
			return new SystemMessage(
					RestService.getInstance().getFormattedString(LOGOUT_MESSAGE_KEY, new Object[]{username}));
		}

	}

}
