package csbase.server.services.restservice.websocket.notificationcenter;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.EvictingQueue;
import csbase.logic.CommandInfo;
import csbase.logic.CommandNotification;
import csbase.server.Server;
import csbase.server.services.commandpersistenceservice.CommandPersistenceService;
import csbase.server.services.messageservice.MessageService;
import csbase.server.services.restservice.websocket.*;
import csbase.server.services.restservice.websocket.utils.PersistentMap;
import csbase.server.services.restservice.websocket.utils.WebSocketUtils;
import csbase.util.messages.IMessageListener;
import csbase.util.messages.filters.BodyTypeFilter;
import org.glassfish.grizzly.websockets.*;

import org.json.JSONArray;
import org.json.JSONObject;

public class CSBaseNotificationCenter extends CSBaseWebSocketApplication {

	// Notification persistence file name
	private static final String NOTIFICATIONS_FILE = "wsnotifications.dat";

	// Notification history max size
	private static final int NOTIFICATION_HISTORY_MAX_SIZE = 20;

	// Notifications persistence
	private PersistentMap<String, EvictingQueue<String>> notifications = new PersistentMap<>(
			WebSocketUtils.generatePath(NOTIFICATIONS_FILE));

	public CSBaseNotificationCenter() {
		super();

		// CommandNotification listener
		setOnJobTerminateListener();
	}

	@Override
	public void onConnect(WebSocket socket) {
		super.onConnect(socket);

		if (socket instanceof CSBaseWebSocket) {
			CSBaseWebSocket ws = ((CSBaseWebSocket) socket);

			// Initialize notifications history
			notifications.putIfAbsent(ws.getUser().getLogin(), EvictingQueue.create(NOTIFICATION_HISTORY_MAX_SIZE));

			// Upon connection, send the notification history to the current
			// user
			ws.send(createNotificationHistoryMessage(ws.getUser().getLogin()).toString());
		}

	}

	private void setOnJobTerminateListener() {

		// Initialize Jackson ObjectMapper (POJO to JSON conversion)
		ObjectMapper mapper = new ObjectMapper();

		// POJO to JSON with non null values only
		mapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);

		// Set CommandNotification listener
		MessageService.getInstance().setServerMessageListener((IMessageListener) messages -> {
			for (csbase.util.messages.Message message : messages) {
				try {
					Server.logInfoMessage("CommandNotification received in CSBaseNotificationCenter\n\t"
							+ message.getBody().getClass().getSimpleName() + ": " + message.getBody());
					if (message.getBody() instanceof CommandNotification) {

						// Cast body to CommandNotification
						CommandNotification commandNotification = (CommandNotification) message.getBody();

						// POJO to JSON conversion
						JSONObject content = new JSONObject(mapper.writeValueAsString(commandNotification));

						// Hack to get project name of incoming notification
						String project = content.getString("projectId").split("/")[1];
						content.put("project", project);

						// Get CommandInfo object
						CommandInfo commandInfo = CommandPersistenceService.getInstance().getCommandInfo(
								commandNotification.getProjectId(), commandNotification.getCommandId().toString());
						String user = commandInfo.getUserId().toString();
						content.put("user", user);

						// Encapsulate content into a Message object
						Message commandTerminate = createCommandTerminateMessage(content);

						// Send to user that "owns" the notification
						this.connections.forEach((targetUser, targetUserWebSockets) -> {
							if (targetUser.equals(user)) {
								notifications.compute(targetUser, (savedUser, savedUserNotifications) -> {
									savedUserNotifications.add(content.toString());
									return savedUserNotifications;
								});
								Server.logInfoMessage("Sending notification for user " + user + " "
										+ targetUserWebSockets.size() + " connections");
								broadcaster.broadcast(targetUserWebSockets, commandTerminate.toString());
							}
						});

					}
				} catch (Exception e) {
					e.printStackTrace();
					Server.logSevereMessage(
							"Error processing Message " + message + "\n\tMessage body: " + message.getBody(), e);
				}
			}
		} , new BodyTypeFilter(CommandNotification.class));
	}

	private Message createCommandTerminateMessage(JSONObject content) throws JsonProcessingException {
		Message commandTerminate = new Message();
		commandTerminate.setType(Message.TYPE_COMMAND_TERMINATE);
		commandTerminate.setContent(content);
		return commandTerminate;
	}

	private Message createNotificationHistoryMessage(String user) {
		JSONArray jsonArray = new JSONArray();
		notifications.get(user).forEach(jsonArray::put);
		Message history = new Message();
		history.setType(Message.TYPE_NOTIFICATION_HISTORY);
		history.setContent(jsonArray);
		return history;
	}

}
