/*
 * $Id: WIOServerProject.java 149543 2014-02-11 13:02:45Z oikawa $
 */
package csbase.server.services.wioservice;

import org.omg.PortableServer.POA;

import tecgraf.javautils.core.io.FileUtils;
import csbase.logic.User;
import csbase.server.Server;
import csbase.server.services.projectservice.ProjectService;
import csbase.server.services.wioservice.idl.FileInfo;
import csbase.server.services.wioservice.idl.WIOFile;
import csbase.server.services.wioservice.idl.WIOFileHelper;
import csbase.server.services.wioservice.idl.WIOProjectPOA;
import csbase.server.services.wioservice.idl.WIOServiceException;

/**
 * A classe <code>WIOServerProject</code> implementa o <i>servant</i> associado
 *  interface idl <code>WIOProject</code>.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class WIOServerProject extends WIOProjectPOA {

  /**
   * Nome do rojeto no servidor
   */
  private String projectName = null;

  /**
   * Num de arquivos abertos no projeto.
   */
  private int numFiles = 0;

  /**
   * File system associado.
   */
  private WIOServerFileSystem fileSystem = null;

  /**
   * O login do usurio autenticado.
   */
  private final String userLogin;

  /**
   * O login do usurio dono do projeto.
   */
  final private String ownerLogin;

  /**
   * Mtodo de ativao do objeto CORBA WIOProject.
   * 
   * @return o POA associado ao objeto ativado.
   * 
   * @throws Exception em caso de erro.
   */
  protected POA activateCorbaProject() throws Exception {
    final WIOService wioService = getWIOService();
    return wioService.activateCorbaObject(null, this);
  }

  /**
   * Acrescenta uma descrio ao arquivo.
   * 
   * @param path o path para o arquivo.
   * @param description a string com a descrio.
   * 
   * @throws Exception em caso de erro.
   */
  protected void appendDescription(final String path, final String description)
    throws Exception {
    final String[] srvPath = mountServerPath(path);
    checkExists(srvPath);
    final String text = (description == null ? "<null>" : description);
    final ProjectService projectService =
      getProjectServiceWithThreadUserAdjusted();
    final Object pid = getMyProjectId();
    projectService.appendFileDescription(pid, srvPath, text);
    ProjectService.setUserId(null);
  }

  /**
   * Verificao de no existncia de path no servidor.
   * 
   * @param srvPath o path
   * @throws Exception em caso de erro.
   */
  private void checkExists(final String[] srvPath) throws Exception {
    if (srvPath == null) {
      final String err = "Path array nulo em projeto " + this.toString();
      throw new Exception(err);
    }
    final ProjectService projectService =
      getProjectServiceWithThreadUserAdjusted();
    final Object pid = getMyProjectId();
    final boolean exists = projectService.existsFile(pid, srvPath);
    ProjectService.setUserId(null);
    if (!exists) {
      final String path = FileUtils.joinPath(srvPath);
      final String err =
        "[" + path + "] inexistente no projeto " + this.toString();
      throw new Exception(err);
    }
  }

  /**
   * Verificao de no existncia de path no servidor.
   * 
   * @param srvPath o path
   * 
   * @throws Exception em caso de erro.
   */
  private void checkNotExists(final String[] srvPath) throws Exception {
    if (srvPath == null) {
      final String err = "Path array nulo em projeto " + this.toString();
      throw new Exception(err);
    }
    final ProjectService projectService =
      getProjectServiceWithThreadUserAdjusted();
    final Object pid = getMyProjectId();
    final boolean exists = projectService.existsFile(pid, srvPath);
    ProjectService.setUserId(null);
    if (exists) {
      final String path = FileUtils.joinPath(srvPath);
      final String err =
        "[" + path + "] j existente no projeto " + this.toString();
      throw new Exception(err);
    }
  }

  /**
   * Checagem do <code>serverProject</code> associado.
   * 
   * @throws Exception em caso de erro.
   */
  private void checkServerProject() throws Exception {
    final ProjectService projectService =
      getProjectServiceWithThreadUserAdjusted();
    final Object uid = User.getUserByLogin(userLogin).getId();
    final Object pid = projectService.getProjectId(uid, projectName);
    final boolean projectInexistant = !projectService.existsProject(pid);
    ProjectService.setUserId(null);
    if (projectInexistant) {
      final String fmt = "Projeto nulo detectado: %s (usuario %s)";
      final String err = String.format(fmt, projectName, userLogin);
      throw new Exception(err);
    }
  }

  /**
   * Cpia de arquivo.
   * 
   * @param fromPath o caminho lgico origem do arquivo.
   * @param toPath o caminho lgico destino do arquivo.
   * 
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public void copyFile(final String fromPath, final String toPath)
    throws WIOServiceException {
    try {
      final ProjectService projectService =
        getProjectServiceWithThreadUserAdjusted();
      final String[] fromSrvPath = mountServerPath(fromPath);
      final String[] toSrvPath = mountServerPath(toPath);
      checkExists(fromSrvPath);
      final Object pid = getMyProjectId();
      projectService.copyFile(pid, fromSrvPath, toSrvPath);
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha de copyFile", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
    finally {
      ProjectService.setUserId(null);
    }
  }

  /**
   * Criao de um diretrio no projeto.
   * 
   * @param path o caminho lgico no projeto.
   * 
   * @return um objeto <code>WIOFile</code> associado ao novo diretrio.
   * 
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public WIOFile createDirectory(final String path) throws WIOServiceException {
    try {
      final ProjectService projectService =
        getProjectServiceWithThreadUserAdjusted();
      final String[] srvPath = mountServerPath(path);
      checkNotExists(srvPath);
      final Object pid = getMyProjectId();
      projectService.createDirectory(pid, srvPath);
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha de createDirectory", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
    finally {
      ProjectService.setUserId(null);
    }

    final WIOFile file = this.getFile(path);
    if (file == null) {
      final String err = "Diretorio [" + path + "] nao apareceu no projeto ";
      throw new WIOServiceException(err);
    }
    if (!file.isDirectory()) {
      final String err =
        "[" + path + "] nao apareceu no projeto como diretorio";
      throw new WIOServiceException(err);
    }
    return file;
  }

  /**
   * Criao de um arquivo no servidor.
   * 
   * @param path o caminho lgico do arquivo.
   * @param type o tipo CSBASE associado.
   * @param userId Identificador do usurio que cria o arquivo.
   * 
   * @return um objeto <code>WIOFile</code> associado.
   * 
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public WIOFile createFile(final String path, final String type,
    final String userId) throws WIOServiceException {
    try {
      final ProjectService projectService =
        getProjectServiceWithThreadUserAdjusted();
      final String[] srvPath = mountServerPath(path);
      final Object pid = getMyProjectId();
      if (!projectService.existsFile(pid, srvPath)) {
        final String[] dirPath = new String[srvPath.length - 1];
        for (int i = 0; i < srvPath.length - 1; i++) {
          dirPath[i] = srvPath[i];
        }
        final String fileName = srvPath[srvPath.length - 1];
        projectService.createFile(pid, dirPath, fileName, type);
      }
      return this.getFile(path);
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha de createFile", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
    finally {
      ProjectService.setUserId(null);
    }
  }

  /**
   * Mtodo de desativao do objeto CORBA WIOProject.
   * 
   * @throws Exception em caso de erro.
   */
  protected void deactivateCorbaProject() throws Exception {
    final WIOService wioService = getWIOService();
    wioService.deactivateCorbaObject(this);
  }

  /**
   * Decrementa o nmero de arquivos abertos no projeto via mtodo de
   * <code>destroy</code> de WIOServerFile.
   * 
   * @throws Exception se houver underflow de limite permitido.
   */
  protected void decNumFiles() throws Exception {
    if (numFiles <= 0) {
      final String err = "Underflow em numero de arquivos abertos no projeto.";
      Server.logSevereMessage(err);
      throw new Exception(err);
    }
    numFiles--;
    fileSystem.decNumFiles();
  }

  /**
   * Deleo de arquivo no file system do projeto.
   * 
   * @param path o caminho lgico do arquivo.
   * 
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public void deleteFile(final String path) throws WIOServiceException {
    try {
      final String[] srvPath = mountServerPath(path);
      checkExists(srvPath);
      final ProjectService projectService =
        getProjectServiceWithThreadUserAdjusted();
      final Object pid = getMyProjectId();
      projectService.removeFile(pid, srvPath);
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha de deleteFile", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
    finally {
      ProjectService.setUserId(null);
    }
  }

  /**
   * Destruio do objeto
   * 
   * @throws WIOServiceException em caso de erro
   */
  @Override
  public void destroy() throws WIOServiceException {
    try {
      deactivateCorbaProject();
      fileSystem.decNumProjects();
      fileSystem.incNumProjects();
      final String fmt = "Projeto '%s' do usurio %s fechado pelo usurio %s.";
      final String msg = String.format(fmt, projectName, ownerLogin, userLogin);
      Server.logInfoMessage(msg);
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha de destroy", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }

  }

  /**
   * Consulta de existncia de um arquivo.
   * 
   * @param path o caminho lgico do arquivo.
   * 
   * @return um flag indicativo booleano de existncia.
   * 
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public boolean fileExists(final String path) throws WIOServiceException {
    try {
      final ProjectService projectService =
        getProjectServiceWithThreadUserAdjusted();
      final String[] srvPath = mountServerPath(path);
      final Object pid = getMyProjectId();
      return projectService.existsFile(pid, srvPath);
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha de fileExists", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
    finally {
      ProjectService.setUserId(null);
    }
  }

  /**
   * Consulta a descrio do arquivo.
   * 
   * @param path o path para o arquivo.
   * 
   * @return a string com a descrio.
   * 
   * @throws Exception em caso de erro.
   */
  protected String getDescription(final String path) throws Exception {
    final String[] srvPath = mountServerPath(path);
    checkExists(srvPath);
    final ProjectService projectService =
      getProjectServiceWithThreadUserAdjusted();
    final Object pid = getMyProjectId();
    String desc = projectService.getFileDescription(pid, srvPath);
    ProjectService.setUserId(null);
    desc = (desc == null || desc.trim().equals("") ? "" : desc);
    return desc;
  }

  /**
   * Busca de um arquivo no projeto.
   * 
   * @param path o caminho lgico do arquivo.
   * 
   * @return um objeto <code>WIOFILE</code> associado.
   * 
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public WIOFile getFile(final String path) throws WIOServiceException {
    try {
      checkServerProject();
      final WIOServerFile file = new WIOServerFile(this, path);
      final POA poa = file.activateCorbaFile();
      final WIOFile corbaFile =
        WIOFileHelper.narrow(poa.servant_to_reference(file));
      if (corbaFile == null) {
        final String err = "Arquivo nulo detectado (path: " + path + ")";
        throw new Exception(err);
      }
      return corbaFile;
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha de getFile", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * No implementado.
   * 
   * @param path o caminho lgico ao arquivo..
   * @param recur flag indicativo de busca recursiva.
   * 
   * @return um array de informaes de arquivos.
   * 
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public FileInfo[] getFilesInfo(final String path, final boolean recur)
    throws WIOServiceException {
    try {
      // ProjectService.getFilesInfo no est implementado.
      final String err = "Falha nos paths em getFilesInfo() " + this.toString();
      throw new Exception(err);
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha de getFilesInfo", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }

  }

  /**
   * Retorno do identificador do projeto (no escopo do usurio associado).
   * 
   * @return uma string de identificao.
   * 
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public String getId() throws WIOServiceException {
    try {
      checkServerProject();
      return projectName;
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha de getId", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * Obtm o usurio autenticado.
   * 
   * @return O usurio autenticado.
   */
  String getLoggedUserLogin() {
    return this.userLogin;
  }

  /**
   * Retorna o nmero mximo de arquivos abertos no projeto.
   * 
   * @return o nmero mximo.
   * 
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public int getMaxOpenedFiles() throws WIOServiceException {
    try {
      final WIOService wioService = getWIOService();
      final int MAX_NUM_FILES = wioService.getMaxOpenedFiles();
      return MAX_NUM_FILES;
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha de getMaxOpenedFiles", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }

  }

  /**
   * Obtem o identificador do projeto assocuiado ao arquivo.
   * 
   * @return o id
   */
  private Object getMyProjectId() {
    final User oldUser = ProjectService.getUser();

    final ProjectService projectService =
      getProjectServiceWithThreadUserAdjusted();

    ProjectService.setUserId(User.getAdminId());

    final Object userId = getMyUserId();
    final Object pid = projectService.getProjectId(userId, projectName);
    if (oldUser != null) {
      ProjectService.setUserId(oldUser.getId());
    }
    return pid;
  }

  /**
   * Obtem id do usurio loggado.
   * 
   * @return id
   */
  private Object getMyUserId() {
    User loggedUser = null;
    try {
      loggedUser = User.getUserByLogin(userLogin);
    }
    catch (final Exception e) {
      final String fmt = "Falha na obtencao de usuario: %s";
      final String err = String.format(fmt, userLogin);
      Server.logSevereMessage(err);
      loggedUser = null;
    }

    if (loggedUser == null) {
      final String fmt = "Acesso a getProjectService com usuario invalido: %s";
      final String err = String.format(fmt, userLogin);
      Server.logSevereMessage(err);
      return null;
    }

    final Object luid = loggedUser.getId();
    return luid;
  }

  /**
   * Retorna o nmero de arquivos abertos no projeto.
   * 
   * @return o nmero.
   * 
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public int getNumOpenedFiles() throws WIOServiceException {
    try {
      if (numFiles < 0) {
        final String err = "Valor negativo detectado.";
        throw new Exception(err);
      }
      return numFiles;
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha de getNumOpenedFiles", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }

  }

  /**
   * Informa o identificador do projeto.
   * 
   * @return O identificador do projeto.
   */
  String getProjectName() {
    return projectName;
  }

  /**
   * Busca do servio de projetos.
   * 
   * @return o servio de projetos.
   */
  private ProjectService getProjectServiceWithThreadUserAdjusted() {
    final ProjectService prjService = ProjectService.getInstance();
    User loggedUser = null;
    try {
      loggedUser = User.getUserByLogin(userLogin);
    }
    catch (final Exception e) {
      final String fmt = "Falha na obtencao de usuario: %s";
      final String err = String.format(fmt, userLogin);
      Server.logSevereMessage(err);
      loggedUser = null;
    }

    if (loggedUser == null) {
      final String fmt = "Acesso a getProjectService com usuario invalido: %s";
      final String err = String.format(fmt, userLogin);
      Server.logSevereMessage(err);
      return prjService;
    }

    final Object uid = loggedUser.getId();
    ProjectService.setUserId(uid);
    return prjService;
  }

  /**
   * Busca do diretrio raiz do projeto.
   * 
   * @return um objeto <code>WIOFile</code> associado  raiz do projeto.
   * 
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public WIOFile getRootFile() throws WIOServiceException {
    WIOFile corbaFile = null;
    try {
      checkServerProject();
      final WIOServerFile root = new WIOServerFile(this, ".");
      final POA poa = root.activateCorbaFile();
      corbaFile = WIOFileHelper.narrow(poa.servant_to_reference(root));
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha de getRootFile", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }

    if (corbaFile == null) {
      final String err = "Falha de diretorio nulo raiz";
      throw new WIOServiceException(err);
    }
    return corbaFile;
  }

  /**
   * Consulta ao login (id) do usurio dono do projeto.
   * 
   * @return uma string de identificao.
   * 
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public String getUser() throws WIOServiceException {
    try {
      checkServerProject();
      return this.ownerLogin;
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha de getUser", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
    finally {
      ProjectService.setUserId(null);
    }
  }

  /**
   * Mtodo de busca do servio de projetos
   * 
   * @return o servio de projetos no servidor.
   */
  protected WIOService getWIOService() {
    return WIOService.getInstance();
  }

  /**
   * Incrementa o nmero de arquivos abertos no projeto via construtor de
   * WIOServerFile.
   * 
   * @throws Exception se houver estouro de limite permitido.
   */
  protected void incNumFiles() throws Exception {
    final WIOService wioService = getWIOService();
    final int MAX_NUM_FILES = wioService.getMaxOpenedFiles();
    if (numFiles >= MAX_NUM_FILES) {
      final String err = "Overflow em numero de arquivos abertos no projeto.";
      Server.logSevereMessage(err);
      throw new Exception(err);
    }
    numFiles++;
    fileSystem.incNumFiles();
  }

  /**
   * Montagem de um path para uso no servio de projetos.
   * 
   * @param path o caminho lgico no projeto.
   * @return um array de strings.
   * @throws Exception em caso de erro.
   */
  private String[] mountServerPath(final String path) throws Exception {
    if (path == null) {
      final String err =
        "Path nulo para mountServerPath() : " + this.toString();
      throw new Exception(err);
    }
    final String[] result = FileUtils.splitPath(path);
    if (result == null) {
      final String err =
        "Falha de split em mountServerPath() : " + this.toString();
      throw new Exception(err);
    }
    return result;
  }

  /**
   * Transferncia de arquivo no file system do projeto.
   * 
   * @param fromPath o caminho lgico origem do arquivo.
   * @param toPath o caminho lgico destino do arquivo.
   * 
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public void moveFile(final String fromPath, final String toPath)
    throws WIOServiceException {
    try {
      final ProjectService projectService =
        getProjectServiceWithThreadUserAdjusted();
      final String[] fromSrvPath = mountServerPath(fromPath);
      final String[] toSrvPath = mountServerPath(toPath);
      checkExists(fromSrvPath);
      final Object pid = getMyProjectId();
      projectService.moveFile(pid, fromSrvPath, toSrvPath);
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha de moveFile", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
    finally {
      ProjectService.setUserId(null);
    }
  }

  /**
   * Renomeao de arquivo.
   * 
   * @param path o caminho lgico do arquivo.
   * @param newName o novo nome do arquivo.
   * 
   * @throws WIOServiceException em caso de erro.
   */
  @Override
  public void renameFile(final String path, final String newName)
    throws WIOServiceException {
    try {
      final String[] srvPath = mountServerPath(path);
      checkExists(srvPath);
      final ProjectService projectService =
        getProjectServiceWithThreadUserAdjusted();
      final Object pid = getMyProjectId();
      projectService.renameFile(pid, srvPath, newName);
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha de renameFile", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
    finally {
      ProjectService.setUserId(null);
    }
  }

  /**
   * Ajusta a descrio do arquivo.
   * 
   * @param path o path para o arquivo.
   * @param description a string com a descrio.
   * 
   * @throws Exception em caso de erro.
   */
  protected void setDescription(final String path, final String description)
    throws Exception {
    final String[] srvPath = mountServerPath(path);
    checkExists(srvPath);
    final String text = (description == null ? "<null>" : description);
    final ProjectService projectService =
      getProjectServiceWithThreadUserAdjusted();
    final Object pid = getMyProjectId();
    projectService.setFileDescription(pid, srvPath, text);
    ProjectService.setUserId(null);
  }

  /**
   * Ajuste do flag de cosntruo no arquivo servidor.
   * 
   * @param path caminho lgico do arquivo.
   * @param flag indicativo booleano de construo.
   * 
   * @throws Exception em caso de erro.
   */
  protected void setUnderConstruction(final String path, final boolean flag)
    throws Exception {
    final String[] srvPath = mountServerPath(path);
    checkExists(srvPath);
    final ProjectService projectService =
      getProjectServiceWithThreadUserAdjusted();
    final Object pid = getMyProjectId();
    projectService.setUnderConstruction(pid, srvPath, flag);
    ProjectService.setUserId(null);
  }

  /**
   * Mtodo de dump como texto.
   * 
   * @return uma string na forma usurio/projeto.
   */
  @Override
  public String toString() {
    final Object uid = getMyUserId();
    final Object pid = getMyProjectId();

    String userText = "usurio nulo";
    if (uid != null) {
      userText = uid.toString();
    }

    String prjText = "projeto nulo";
    if (pid != null) {
      prjText = pid.toString();
    }

    return "[WIO Project]: " + userText + "@" + prjText;
  }

  /**
   * Construtor de projeto remoto.
   * 
   * @param fs .
   * @param userLogin identificador o usuario logado
   * @param ownerLogin identificador do usurio dono do projeto
   * @param projectName identificador do projeto.
   * 
   * @throws Exception se houver falha no servio de WIO.
   */
  protected WIOServerProject(final WIOServerFileSystem fs,
    final String userLogin, final String ownerLogin, final String projectName)
    throws Exception {
    if (userLogin == null || ownerLogin == null || projectName == null) {
      final String err = "Usuarios/projeto desconhecidos em WIOServerProject";
      throw new Exception(err);
    }
    if (fs == null) {
      final String err = "File-system nao identificado em WIOServerProject";
      throw new Exception(err);
    }
    this.projectName = projectName;
    this.fileSystem = fs;
    this.userLogin = userLogin;
    this.ownerLogin = ownerLogin;

    fileSystem.incNumProjects();
    final String fmt = "Projeto '%s' do usuario %s aberto pelo usurio %s.";
    final String msg = String.format(fmt, projectName, ownerLogin, userLogin);
    Server.logInfoMessage(msg);
  }
}
