package csbase.sga;

import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;

import csbase.server.plugin.service.sgaservice.SGADaemonCommand;
import csbase.sga.executor.JobData;
import csbase.sga.executor.JobExecutor;
import csbase.sga.executor.JobInfo;
import csbase.sga.executor.JobObserver;
import sgaidl.ActionNotSupportedException;
import sgaidl.InvalidActionException;
import sgaidl.InvalidTransitionException;
import sgaidl.JobControlAction;
import sgaidl.Pair;
import sgaidl.ProcessState;
import sgaidl.RunningCommandInfo;

/**
 * Commando que encapsula as informaes especficas do ambiente de execuo do
 * SGA.
 *
 * @author Tecgraf/PUC-Rio
 */
public class SGALocalCommand extends SGADaemonCommand {
  /** UID */
  private static final long serialVersionUID = -5874016392438461849L;

  private static Logger LOGGER;

  /** Executor de jobs */
  private JobExecutor executor;

  /** Identificador do comando */
  private String commandId;

  /** Dados do job */
  private JobData jobData;

  private ListenableFuture<JobData> jobDataFuture;

  private ListenableFuture<Boolean> recoveryStatusFuture;

  ListeningExecutorService executionThread =
    MoreExecutors.listeningDecorator(Executors.newSingleThreadExecutor());

  protected static void setLogger(String name) {
    LOGGER = Logger.getLogger(name);
  }

  /**
   * Construtor.
   *
   * @param executor executor de jobs
   */
  protected SGALocalCommand(JobExecutor executor, String commandId) {
    this.executor = executor;
    this.commandId = commandId;
  }

  /**
   * Construtor.
   *
   * @param executor executor de jobs
   * @param jobData dados do job
   */
  protected SGALocalCommand(JobExecutor executor, String commandId,
    JobData jobData) {
    this.executor = executor;
    this.commandId = commandId;
    this.jobData = jobData;
  }

  /**
   * Obtm os dados do job.
   *
   * @return os dados do job
   */
  protected JobData getJobData() {
    if (jobData != null) {
      return jobData;
    }

    if (jobDataFuture.isDone()) {
      try {
        return jobDataFuture.get();
      }
      catch (InterruptedException | ExecutionException e) {
        // TODO Auto-generated catch block
        e.printStackTrace();
        return null;
      }
    }
    else {
      return null;
    }
  }

  /**
   * Obtm o executor de jobs.
   *
   * @return o executor de jobs
   */
  protected JobExecutor getExecutor() {
    return executor;
  }

  protected void execute(final String commandString,
    final Map<String, String> execParam, final JobObserver observer) {
    jobDataFuture = executionThread.submit(new Callable<JobData>() {
      @Override
      public JobData call() throws Exception {
        try {
          LOGGER.log(
            Level.INFO, "Solicitando execuo do comando {0} ao executor",
            new Object[] { commandId });

          return executor.executeJob(commandString, execParam, observer);
        }
        catch (CancellationException e) {
          throw e;
        }
        catch (Exception e) {
          LOGGER.log(Level.SEVERE, "Erro ao executar comando " + commandId, e);
          return null;
        }
      }
    });

    Futures.addCallback(jobDataFuture, new FutureCallback<JobData>() {
      @Override
      public void onSuccess(JobData data) {
        if (data == null) {
          //Terminar com erro
          LOGGER.log(
            Level.WARNING, "Comando {0} no iniciado", new Object[] {
                commandId });
          observer.onJobLost();
        }
        else {
          observer.onJobStarted(data);
        }
      }

      @Override
      public void onFailure(Throwable t) {
        if (CancellationException.class.isInstance(t)) {
          LOGGER.log(
            Level.FINE, "Execuo do comando {0} foi cancelada", new Object[] {
                commandId });
        }
        else {
          LOGGER.log(Level.SEVERE, "Erro ao executar comando: " + commandId, t);
          observer.onJobLost();
        }
      }
    });
  }

  protected void recovery(final JobData data, final JobObserver observer) {
    this.jobData = data;

    recoveryStatusFuture = executionThread.submit(new Callable<Boolean>() {
      @Override
      public Boolean call() throws Exception {
        LOGGER.log(
          Level.INFO, "Solicitando recuperao do comando {0} ao executor",
          new Object[] { commandId });

        return executor.recoveryJob(jobData, observer);
      }
    });

    Futures.addCallback(recoveryStatusFuture, new FutureCallback<Boolean>() {
      @Override
      public void onSuccess(Boolean wasRecovered) {
        if (!wasRecovered) {
          LOGGER.log(
            Level.INFO, "Comando {0} no recuperado", new Object[] {
                commandId });
          observer.onJobLost();
        }
      }

      @Override
      public void onFailure(Throwable t) {
        LOGGER.log(Level.SEVERE, "Erro ao recuperar comando: " + commandId, t);
        observer.onJobLost();
      }
    });
  }

  /**
   * Altera o estado de um comando ou de um job filho do comando.
   *
   * @param action ao a ser executada sobre o comando
   * @param child identificador do job filho do comando ou nulo se a ao deve
   *        ser realizada no prprio comando
   *
   * @throws InvalidActionException ao invlida
   * @throws ActionNotSupportedException ao no suportada
   * @throws InvalidTransitionException ??? TODO Unir a
   *         InvalidTransitionException com a InvalidActionException
   */
  @Override
  public void control(JobControlAction action, String child)
    throws InvalidActionException, ActionNotSupportedException,
    InvalidTransitionException {
    LOGGER.log(
      Level.INFO, "Exercendo ao {0} no comando {1}", new Object[] {
          action.toString(), commandId });

    JobData data = getJobData();
    if (data != null) {
      this.executor.controlJob(data, child, action);
    }
    else {
      if (!jobDataFuture.cancel(true)) {
        LOGGER.log(
          Level.WARNING, "Erro ao cancelar comando {0}", new Object[] {
              commandId });
      }
    }
  }

  /**
   * Fornece as informaes de monitorao de um comando.
   *
   * @return as informaes de monitorao de todos os processos.
   */
  @Override
  public RunningCommandInfo getRunningCommandInfo() {
    LOGGER.log(
      Level.INFO, "Obtendo as informaes do comando {0}", new Object[] {
          commandId });

    JobData data = getJobData();
    if (data == null) {
      return convertJobInfoToRunningCommandInfo(new JobInfo());
    }
    JobInfo info = this.executor.getJobInfo(data);

    if (info == null) {
      //TODO Notificar o comando como perdido
      LOGGER.log(
        Level.FINE, "No foi possvel obter as informaes do comando {0}",
        new Object[] { commandId });
      return null;
    }

    return convertJobInfoToRunningCommandInfo(info);
  }

  private RunningCommandInfo makeBlankInfo() {
    return new RunningCommandInfo(new Pair[0][], new Pair[0]);
  }

  private RunningCommandInfo makeWatingInfo() {
    Pair[] processInfo =
    { new Pair(sgaidl.COMMAND_STATE.value, ProcessState.WAITING.toString()) };

    return new RunningCommandInfo(new Pair[][] { processInfo }, new Pair[0]);
  }

  private RunningCommandInfo convertJobInfoToRunningCommandInfo(
    JobInfo jobInfo) {
    List<Pair[]> processData = new LinkedList<Pair[]>();

    List<Pair> mainProcessDic = new LinkedList<Pair>();
    for (String key : jobInfo.jobParam.keySet()) {
      mainProcessDic.add(new Pair(key, jobInfo.jobParam.get(key)));
    }
    processData.add(mainProcessDic.toArray(new Pair[0]));

    for (JobInfo pInfo : jobInfo.children) {
      List<Pair> pDic = new LinkedList<Pair>();
      for (String key : pInfo.jobParam.keySet()) {
        pDic.add(new Pair(key, pInfo.jobParam.get(key)));
      }
      processData.add(pDic.toArray(new Pair[0]));
    }

    return new RunningCommandInfo(processData.toArray(new Pair[0][]),
      new Pair[0]);
  }
}