package csbase.sga;

import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;

import csbase.server.plugin.service.sgaservice.SGADaemonCommand;
import csbase.sga.executor.JobData;
import csbase.sga.executor.JobExecutor;
import csbase.sga.executor.JobInfo;
import csbase.sga.executor.JobObserver;
import sgaidl.ActionNotSupportedException;
import sgaidl.COMMAND_STATE;
import sgaidl.InvalidActionException;
import sgaidl.InvalidTransitionException;
import sgaidl.JobControlAction;
import sgaidl.Pair;
import sgaidl.ProcessState;
import sgaidl.RunningCommandInfo;

/**
 * Commando que encapsula as informaes especficas do ambiente de execuo do
 * SGA.
 *
 * @author Tecgraf/PUC-Rio
 */
public class SGALocalCommand extends SGADaemonCommand {
  /** UID */
  private static final long serialVersionUID = -5874016392438461849L;

  private static Executor notifier = Executors.newCachedThreadPool();

  private Logger logger;

  /** Jobs executor */
  private JobExecutor executor;

  /** Command's id */
  private String commandId;

  /** Job's data */
  private JobData jobData;

  private ListenableFuture<JobData> jobDataFuture;

  private ListenableFuture<Boolean> recoveryStatusFuture;

  private JobObserver observer;

  protected void setLogger(Logger logger) {
    this.logger = logger;
  }

  /**
   * Constructor.
   *
   * @param executor the JobExecutor
   * @param commandId the command identifier
   */
  protected SGALocalCommand(JobExecutor executor, String commandId) {
    this.executor = executor;
    this.commandId = commandId;
  }

  /**
   * Constructor.
   *
   * @param executor the JobExecutor
   * @param commandId the command identifier
   * @param jobData the JobData
   */
  protected SGALocalCommand(JobExecutor executor, String commandId,
    JobData jobData) {
    this.executor = executor;
    this.commandId = commandId;
    this.jobData = jobData;
  }

  /**
   * Gets the job's JobData.
   *
   * @return the JobData
   */
  protected JobData getJobData() {
    if (jobData != null) {
      return jobData;
    }

    if (jobDataFuture.isDone()) {
      try {
        return jobDataFuture.get();
      }
      catch (InterruptedException | ExecutionException e) {
        // TODO Auto-generated catch block
        e.printStackTrace();
        return null;
      }
    }
    else {
      return null;
    }
  }

  /**
   * Gets the JobExecutor.
   *
   * @return the JobExecutor
   */
  protected JobExecutor getExecutor() {
    return executor;
  }

  protected void execute(final String commandString,
    final Map<String, String> execParam, final JobObserver observer) {
    this.observer = observer;
    final ListeningExecutorService commandExecutor =
            MoreExecutors.listeningDecorator(Executors.newSingleThreadExecutor());

    jobDataFuture = commandExecutor.submit(new Callable<JobData>() {
      @Override
      public JobData call() throws Exception {
        try {
          logger.log(
            Level.INFO, "Submitting command {0} to executor",
            new Object[] { commandId });

          return executor.executeJob(commandString, execParam, observer);
        }
        catch (CancellationException e) {
          throw e;
        }
        catch (Exception e) {
          logger.log(Level.SEVERE, "Error while executing command " + commandId, e);
          return null;
        }
      }
    });

    Futures.addCallback(jobDataFuture, new FutureCallback<JobData>() {
      @Override
      public void onSuccess(JobData data) {
        if (data == null) {
          //Terminar com erro
          logger.log(
            Level.WARNING, "Command {0} not initialized", new Object[] {
                commandId });
          observer.onJobLost();
        }
        else {
          observer.onJobStarted(data);
        }

        commandExecutor.shutdown();
      }

      @Override
      public void onFailure(Throwable t) {
        if (CancellationException.class.isInstance(t)) {
          logger.log(
            Level.FINE, "Canceling the execution of command {0}", new Object[] {
                commandId });
        }
        else {
          logger.log(Level.SEVERE, "Error while executing command: " + commandId, t);
        }

        observer.onJobLost();

        commandExecutor.shutdown();
      }
    });
  }

  protected void recovery(final JobData data, final JobObserver observer) {
    this.jobData = data;
    this.observer = observer;
    final ListeningExecutorService commandExecutor =
            MoreExecutors.listeningDecorator(Executors.newSingleThreadExecutor());

    recoveryStatusFuture = commandExecutor.submit(new Callable<Boolean>() {
      @Override
      public Boolean call() throws Exception {
        logger.log(
          Level.INFO, "Recovering command {0} from executor",
          new Object[] { commandId });

        return executor.recoveryJob(jobData, observer);
      }
    });

    Futures.addCallback(recoveryStatusFuture, new FutureCallback<Boolean>() {
      @Override
      public void onSuccess(Boolean wasRecovered) {
        if (!wasRecovered) {
          logger.log(
            Level.INFO, "Command {0} not recovered", new Object[] {
                commandId });
          observer.onJobLost();
        }

        commandExecutor.shutdown();
      }

      @Override
      public void onFailure(Throwable t) {
        logger.log(Level.SEVERE, "Error while recovering command: " + commandId, t);
        observer.onJobLost();

        commandExecutor.shutdown();
      }
    });
  }

  /**
   * Control a job or one of its children.
   *
   * @param action action to send to the job or to the job's child
   * @param child child's id or null if the job is receiving the action
   * @throws InvalidActionException If the action is invalid
   * @throws ActionNotSupportedException If the action is not supported
   * @throws InvalidTransitionException If sending the action the job perform an invalid state transition
   */
  @Override
  public void control(JobControlAction action, String child)
    throws InvalidActionException, ActionNotSupportedException,
    InvalidTransitionException {
    logger.log(
      Level.INFO, "Sending action {0} to command {1}", new Object[] {
          action.toString(), commandId });

    JobData data = getJobData();
    if (data != null) {
      this.executor.controlJob(data, child, action);
    }
    else {
      if (!jobDataFuture.cancel(true)) {
        logger.log(
          Level.WARNING, "Error while canceling command {0}", new Object[] {
              commandId });
      }
    }
  }

  /**
   * Gets the running command information.
   *
   * @return the running command information
   */
  @Override
  public RunningCommandInfo getRunningCommandInfo() {
    logger.log(
      Level.INFO, "Getting info from command {0}", new Object[] {
          commandId });

    JobData data = getJobData();
    if (data == null) {
      return convertJobInfoToRunningCommandInfo(new JobInfo());
    }

    final JobInfo info = this.executor.getJobInfo(data);

    if (info == null) {
      //TODO Notificar o comando como perdido
      logger.log(
        Level.FINE, "Error while getting info from command {0}",
        new Object[] { commandId });
      return null;
    }

    // Command is finished
    if(info.jobParam.get(COMMAND_STATE.value).equals(ProcessState.FINISHED.toString())){
      notifier.execute(new Runnable() {
        @Override
        public void run() {
          long startTime = System.currentTimeMillis();
          observer.onJobCompleted(info);
          logger.fine("Time took to notify command completion: " + (System.currentTimeMillis() - startTime + " ms"));
        }
      });
    }

    return convertJobInfoToRunningCommandInfo(info);
  }

  private RunningCommandInfo makeBlankInfo() {
    return new RunningCommandInfo(new Pair[0][], new Pair[0]);
  }

  private RunningCommandInfo makeWatingInfo() {
    Pair[] processInfo =
    { new Pair(sgaidl.COMMAND_STATE.value, ProcessState.WAITING.toString()) };

    return new RunningCommandInfo(new Pair[][] { processInfo }, new Pair[0]);
  }

  private RunningCommandInfo convertJobInfoToRunningCommandInfo(
    JobInfo jobInfo) {
    List<Pair[]> processData = new LinkedList<Pair[]>();

    List<Pair> mainProcessDic = new LinkedList<Pair>();
    for (String key : jobInfo.jobParam.keySet()) {
      mainProcessDic.add(new Pair(key, jobInfo.jobParam.get(key)));
    }
    processData.add(mainProcessDic.toArray(new Pair[mainProcessDic.size()]));

    for (JobInfo pInfo : jobInfo.children) {
      List<Pair> pDic = new LinkedList<Pair>();
      for (String key : pInfo.jobParam.keySet()) {
        pDic.add(new Pair(key, pInfo.jobParam.get(key)));
      }
      processData.add(pDic.toArray(new Pair[pDic.size()]));
    }

    return new RunningCommandInfo(processData.toArray(new Pair[processData.size()][]),
      new Pair[0]);
  }
}