/*
 * $Id: CSBaseTestSetupInOtherVM.java 57297 2006-10-06 15:07:35Z roberta $
 */
package csbase.test.unit.service.othervm;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.HashMap;
import java.util.Map;

import org.junit.After;
import org.junit.Before;

import csbase.logic.url.URLParameters;
import csbase.test.unit.TestConfig;
import csbase.test.unit.mock.MockCSBaseClient;
import csbase.util.FileSystemUtils;
import junit.framework.Test;
import tecgraf.javautils.configurationmanager.ConfigurationManager;

/**
 * A classe <code>CSBaseTestSetupOVM</code> que  responsvel por:
 * 
 * <ul>
 * <li> Iniciar uma instncia de testes do servidor em uma mquina virtual,
 * antes da execuo dos mtodos de teste dos servios;
 * <li> Iniciar o cliente e conect-lo ao servidor;
 * <li> Encerrar o processo do servidor ao final dos testes.
 * </ul>
 * 
 * Esta classe consiste em um "Wrapper" para todos os casos de testes definidos
 * para os servios. Esta classe foi necessria pois antes da execuao dos
 * testes de um servio uma instncia do servidor deve estar no ar. Desta forma,
 * para que o TestCase de um servio seja executado o seguinte mtodo deve ser
 * associado ao test suite.
 * 
 * Por exemplo:
 * 
 * <code>
 * public static Test suite() { 
 *   TestSuite suite = new TestSuite();
 *   suite.addTestSuite(AdministrationServiceTest.class);
 *   return new CSBaseTestSetup(suite); 
 * }
 * 
 * public static void main(String[] args) {
 *   junit.textui.TestRunner.run(ServerTestSuite.suite()); 
 * }
 * </code>
 */
public abstract class CSBaseTestSetupOVM {

  /** Processo em que o servidor executa. */
  private Process serverProcess = null;

  /** Flag para trmino do processo do servidor. */
  private boolean shutDownServer;

  /**
   * Construtor.
   * 
   * @param test um test case ou um test suite
   */
  public CSBaseTestSetupOVM(Test test) {
    this.shutDownServer = false;
  }

  /**
   * 
   * Este mtodo ser chamado uma nica vez antes da execuo de um conjunto de
   * mtodos de testes contidos em um test case ou test suite.
   * 
   * @throws Exception se ocorrer falha no setup dos testes
   */
  @Before
  protected void setUp() throws Exception {
    if (!TestConfig.init()) {
      System.out.println("Arquivo de propridades no encontrado." 
        + " Usaremos as configuraes padro.");
    }
    try {
      System.out.println("Vai inicializar o servidor CSBase ...");
      Thread thread = new StartServer();
      thread.start();
      int sleep = TestConfig.getIntProperty(TestConfig.SETUP_SLEEP_PROP,
        TestConfig.SETUP_SLEEP_DEFAULT);
      System.out.println("Esperando " + (sleep / 6000)
        + "s para logar no servidor...");
      Thread.sleep(sleep);
      login();
    } catch (Exception e) {
      System.out.println("Falha no setup dos testes.");
      e.printStackTrace();
    }
  }

  /**
   * Mtodo auxiliar que descobre os jars ou zips disponveis em um diretorio
   * 
   * @param dirPath um string que contem o nome completo de um diretorio.
   * @return um string contendo a lista o caminho completo para os jars
   *         separados por ponto e virgula.
   */
  protected String getJarFiles(String dirPath) {
    File dir = new File(dirPath);
    StringBuffer result = new StringBuffer();

    File children[] = dir.listFiles();

    if (children == null) {
      return "";
    }
    if (children != null) {
      for (int i = 0; i < children.length; i++) {
        File child = children[i];

        // Elimina o diretrio do subversion
        if (child.getName().equals(FileSystemUtils.VERSION_CONTROL_DIR)) {
          continue;
        }
        if (child.isDirectory()) {
          result.append(getJarFiles(child.getPath()));
        }
        else {
          if (child.getName().indexOf(".jar") > 0
            || child.getName().indexOf(".zip") > 0) {
            result.append(child.getPath() + File.pathSeparatorChar);
          }
        }
      }
    }

    return result.toString();
  }

  /**
   * Mtodo que cria e retorna o comando que executar um servidor do CSBase
   * definido especificamente para testes dos servios do CSBase. Esse mtodo
   * precisa ser redefinido por cada sistema que estende o CSBase, a fim de
   * criar o comando de execuo do servidor que  especfico de cada um desses
   * sistemas.
   * 
   * @return o comando de execuo do servidor
   */
  public abstract String defineCommandLine();

  /**
   * Este mtodo destri o processo que est executando o servidor.
   * 
   * @throws Exception erro ao destruir o processo Java do servidor.
   */
  @After
  protected void tearDown() throws Exception {
    if (serverProcess != null) {
      shutDownServer = true;
      serverProcess.destroy();
      System.out.println("    Processo do servidor encerrado...");
    }
  }

  /**
   * Thread responsvel por iniciar o servidor em outra mquina virtual.
   */
  class StartServer extends Thread {
    /**
     * {@inheritDoc}
     */
    @Override
    public void run() {
      try {
        String runServerCommand = defineCommandLine();

        serverProcess = Runtime.getRuntime().exec(runServerCommand);

        System.out.println("Imprimindo Saida do servidor ...");
        BufferedReader in = new BufferedReader(new InputStreamReader(
          serverProcess.getInputStream()));
        String lineIn = null;
        while ((!shutDownServer) && ((lineIn = in.readLine()) != null)) {
          System.out.println(lineIn);
        }
      } catch (IOException ioe) {
        if (!shutDownServer) {
          System.err.println("No foi possvel ler a sada do servidor.");
          ioe.printStackTrace();
        }
      } catch (Throwable e) {
        System.err.println("O servidor nao pode ser instanciado.");
        e.printStackTrace();
      }
    }
  }

  /**
   * Inicia o <code>MockCSBaseClient</code> e faz o login do cliente no
   * servidor.
   * 
   * @throws Exception se ocorrer uma exceo durante o login
   */
  public void login() throws Exception {
    String systemName = "Cliente de Testes";
    String server = TestConfig.getProperty(TestConfig.SERVER_PROP,
      TestConfig.SERVER_DEFAULT);
    Map<String, String> params = new HashMap<String, String>();
    params.put(URLParameters.SERVER_URL_PARAMETER, server);
    ConfigurationManager.createInstance();
    MockCSBaseClient client = new MockCSBaseClient(systemName, params);
    if (!client.execute()) {
      System.out.println("No foi possvel iniciar o cliente.");
    }
  }
}
