/*
 * $Id$
 */
package csbase.tools;

import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Toolkit;
import java.awt.image.BufferedImage;
import java.awt.image.ImageObserver;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;

import javax.swing.JFileChooser;

import Acme.JPM.Encoders.GifEncoder;

/**
 * Utilitrio para gerao de imagens de arquivos derivados (associado com tipos
 * de arquivos).
 * 
 * @author Tecgraf
 */
public class BuildVariants {

  /**
   * Variantes.
   */
  private static Variant[] variants = new Variant[] {
      new Variant("_cut", 255, 255, 255, 050),
      new Variant("_uc", 205, 205, 205, 245) };

  /**
   * Fator da equao
   */
  private static float A = 0.6272F;

  /**
   * Fator da equao
   */
  private static float B = -0.3403F;

  /**
   * Fator da equao
   */
  private static float C = 0.1132F;

  /**
   * Fator da equao
   */
  private static float D = 0.6F;

  /**
   * Main
   * 
   * @param args argumentos
   */
  public static void main(String[] args) {
    if (args.length == 0) {
      while (true) {
        String path = chooseFile();
        if (path == null) {
          break;
        }
        buildVariants(path);
      }
    }
    else {
      for (int i = 0; i < args.length; i++) {
        buildVariants(args[i]);
      }
    }
    System.exit(0);
  }

  /**
   * Escolha de arquivo.
   * 
   * @return path
   */
  private static String chooseFile() {
    JFileChooser dialog = new JFileChooser(".");
    dialog.setFileFilter(new javax.swing.filechooser.FileFilter() {
      @Override
      public boolean accept(File f) {
        String name = f.getName().toLowerCase();
        return name.endsWith(".gif") || name.endsWith(".jpg")
          || name.endsWith(".jpeg") || name.endsWith(".png") || f.isDirectory();
      }

      @Override
      public String getDescription() {
        return "Image files";
      }
    });

    int resp = dialog.showOpenDialog(null);
    if (resp == JFileChooser.APPROVE_OPTION) {
      return dialog.getSelectedFile().getAbsolutePath();
    }
    return null;
  }

  /**
   * Monta variantes.
   * 
   * @param path path
   */
  private static void buildVariants(String path) {
    ImageData imageData = loadImage(path);
    for (int i = 0; i < variants.length; i++) {
      System.out.println("Gerando " + variants[i].suffix + "...");
      buildVariant(imageData, variants[i]);
    }
  }

  /**
   * Faz carga da imagem.
   * 
   * @param path path
   * @return imagem
   */
  private static ImageData loadImage(String path) {
    System.out.print("Carregando " + path);
    ImageData imageData = new ImageData();
    Observer observer = new Observer(imageData);
    imageData.path = path;
    imageData.img = Toolkit.getDefaultToolkit().getImage(path);
    imageData.width = imageData.img.getWidth(observer);
    imageData.height = imageData.img.getHeight(observer);
    while (!observer.isReady()) {
      try {
        Thread.sleep(1000);
        System.out.print(".");
      }
      catch (InterruptedException e) {
        e.printStackTrace();
      }
    }
    System.out.println();
    return imageData;
  }

  /**
   * Faz um avariao.
   * 
   * @param imageData imagem
   * @param variant variante
   */
  private static void buildVariant(ImageData imageData, Variant variant) {
    Image image = applyFilter(imageData, variant);
    String newPath = getVariantPath(imageData, variant);
    try {
      FileOutputStream fos = new FileOutputStream(newPath);
      BufferedOutputStream bos = new BufferedOutputStream(fos);
      GifEncoder encoder = new GifEncoder(image, bos);
      encoder.encode();
      bos.flush();
      bos.close();
    }
    catch (IOException e) {
      e.printStackTrace();
    }
  }

  /**
   * Aplica filtro.
   * 
   * @param imageData imagem
   * @param v variante
   * @return imagem
   */
  private static Image applyFilter(ImageData imageData, Variant v) {
    BufferedImage img =
      getBufferedImage(imageData, BufferedImage.TYPE_INT_ARGB);
    BufferedImage gry =
      getBufferedImage(imageData, BufferedImage.TYPE_BYTE_GRAY);
    for (int y = 0; y < imageData.height; y++) {
      for (int x = 0; x < imageData.width; x++) {
        int argb = img.getRGB(x, y);
        float a = ((argb & 0xFF000000) >>> 24) / 255F;
        if (a == 0) {
          continue;
        }
        float f = (gry.getRGB(x, y) & 0xFF) / 255F;
        f = (A * f * f * f) + (B * f * f) + (C * f) + D;
        argb =
          (op(1, a) << 24) | (op(f, v.r) << 16) | (op(f, v.g) << 8)
            | op(f, v.b);
        img.setRGB(x, y, argb);
      }
    }
    return img;
  }

  /**
   * Faz operap bsica.
   * 
   * @param f fator
   * @param c cor (componente)
   * @return inteiro equivalente
   */
  private static int op(float f, float c) {
    return (int) (Math.min(1, f * c) * 255);
  }

  /**
   * Monta path de variante
   * 
   * @param imageData imagem
   * @param variant variante
   * @return path
   */
  private static String getVariantPath(ImageData imageData, Variant variant) {
    int p = imageData.path.lastIndexOf('.');
    if (p == -1) {
      return imageData.path + variant.suffix;
    }
    return imageData.path.substring(0, p) + variant.suffix
      + imageData.path.substring(p);
  }

  /**
   * Monta o {@code BufferedImage}
   * 
   * @param imageData imagem
   * @param type tipos
   * @return imagem (buffered)
   */
  private static BufferedImage getBufferedImage(ImageData imageData, int type) {
    BufferedImage img =
      new BufferedImage(imageData.width, imageData.height, type);
    Graphics2D g2d = img.createGraphics();
    g2d.drawImage(imageData.img, 0, 0, null);
    return img;
  }

  /**
   * Variante
   * 
   * @author Tecgraf
   */
  private static class Variant {
    /**
     * Sufixo
     */
    String suffix;

    /**
     * Componente de RGBA
     */
    float r;

    /**
     * Componente de RGBA
     */
    float g;

    /**
     * Componente de RGBA
     */
    float b;

    /**
     * Componente de RGBA
     */
    float a;

    /**
     * Construtor
     * 
     * @param suffix sufixo
     * @param r componente RGBA
     * @param g componente RGBA
     * @param b componente RGBA
     * @param a componente RGBA
     */
    Variant(String suffix, int r, int g, int b, int a) {
      this.suffix = suffix;
      this.r = r / 255F;
      this.g = g / 255F;
      this.b = b / 255F;
      this.a = a / 255F;
    }
  }

  /**
   * Imagem
   * 
   * @author Tecgraf
   */
  private static class ImageData {
    /**
     * Path
     */
    String path;

    /**
     * Imagem
     */
    Image img;

    /**
     * Largura
     */
    int width;

    /**
     * Altura
     */
    int height;
  }

  /**
   * Observador
   * 
   * @author Tecgraf
   */
  private static class Observer implements ImageObserver {
    /**
     * Imagem
     */
    private ImageData imageData;

    /**
     * Indicativo de pronto.
     */
    private boolean ready;

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean imageUpdate(Image i, int f, int x, int y, int w, int h) {
      if ((f & WIDTH) == WIDTH) {
        imageData.width = w;
      }
      if ((f & HEIGHT) == HEIGHT) {
        imageData.height = h;
      }
      if ((f & ALLBITS) == ALLBITS) {
        synchronized (this) {
          ready = true;
        }
        return false;
      }
      return true;
    }

    /**
     * Indica imagem pronta.
     * 
     * @return indicativo
     */
    synchronized boolean isReady() {
      return ready;
    }

    /**
     * Construtor.
     * 
     * @param imageData imagem
     */
    Observer(ImageData imageData) {
      this.imageData = imageData;
      ready = false;
    }
  }
}
