package csbase.util.messages;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Timer;
import java.util.TimerTask;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;

import tecgraf.javautils.core.filter.IFilter;
import csbase.util.data.dispatcher.ExecutorDispatcher;
import csbase.util.data.dispatcher.IDispatcher;
import csbase.util.messages.dao.IMessageStoreDAO;

/**
 * Funciona como intermedirio entre os produtores e os receptores de mensagem.
 *  responsvel por criar, encapsular e persistir os tpicos.
 * 
 * @author Tecgraf
 */
public class MessageBroker {

  /**
   * Tempo mnimo entre sucessivas execues de tarefas de gerenciamento.
   * 
   * @see #managerPeriod
   */
  private static final long MIN_MANAGER_PERIOD = TimeUnit.MINUTES.toMillis(10);

  /**
   * <p>
   * Identificador nico do broker.
   * </p>
   * <p>
   * Utilizado para verificar se os idetificadores de consumidores foram criados
   * por esta instncia.
   * </p>
   */
  private UUID brokerId = UUID.randomUUID();

  /**
   * Objeto utilizado pelos tpicos para entregar as mensagens para seus
   * ouvintes.
   */
  private IDispatcher<IMessageListener, Message> dispatcher;

  /**
   * Objeto utilizado para fazer a persistncia das mensagens dos tpicos.
   */
  private IMessageStoreDAO dao;
  /**
   * Timer responsvel por salvar o mapa de tpicos usando o dao.
   */
  private Timer managerTimer;
  /**
   * Tarefa responsvel pela execuo da persistncia dos tpicos.
   */
  private TimerTask persistTask;
  /**
   * Tempo, em milisegundos, entre sucessivas execues da tarefa de
   * persistncia dos tpicos no dao.
   */
  private long managerPeriod;

  /**
   * Tempo mximo, em milisegundos, que um consumidor tem entre cada chamada ao
   * mtodo {@link #receive(String, Serializable, IFilter)} para ser considerado
   * ativo e no perder mensagens. <br>
   * O valor mnimo para este parmetro 
   * {@value MessageStore#MIN_RECEIVE_TIMEOUT}ms. <br>
   * O valor mnimo para este parmetro 
   * {@value MessageStore#MAX_RECEIVE_TIMEOUT}ms.
   */
  private long receiveTimeout;

  /**
   * Tpicos criados por seus identificadores.
   */
  private HashMap<String, Topic> topicsByName;

  /**
   * Flag utilzada para indicar se o gerente foi iniciado.
   */
  private AtomicBoolean started;

  /**
   * Utilizado para logar informaes no sistema.
   */
  private static final Logger LOGGER = Logger.getLogger(MessageBroker.class
    .getName());

  /**
   * Construtor.
   * 
   * @param dao Objeto utilizado para fazer a persistncia das mensagens dos
   *        tpicos.
   * @param persistPeriod Tempo, em milisegundos, entre sucessivas execues da
   *        tarefa de persistncia dos tpicos no dao.
   * @param receiveTimeout Tempo mximo, em milisegundos, que um consumidor tem
   *        entre cada chamada ao mtodo
   *        {@link #receive(String, Serializable, IFilter)} para ser considerado
   *        ativo e no perder mensagens. <br>
   *        O valor mnimo para este parmetro 
   *        {@value MessageStore#MIN_RECEIVE_TIMEOUT}ms. <br>
   *        O valor mnimo para este parmetro 
   *        {@value MessageStore#MAX_RECEIVE_TIMEOUT}ms.
   * @param maxThreads Nmero mximo de threads utilizado para entregar as
   *        mensagens aos ouvintes.
   */
  public MessageBroker(IMessageStoreDAO dao, long persistPeriod,
    long receiveTimeout, int maxThreads) {

    this(new ThreadPoolExecutor(maxThreads, maxThreads, 60L, TimeUnit.SECONDS,
      new LinkedBlockingQueue<Runnable>()), dao, persistPeriod, receiveTimeout);
  }

  /**
   * Construtor.
   * 
   * @param executor Responsvel por executar a tarefa de entregar as mensagens
   *        aos consumidores assncronos.
   * @param dao Objeto utilizado para fazer a persistncia das mensagens dos
   *        tpicos.
   * @param persistPeriod Tempo, em milisegundos, entre sucessivas execues da
   *        tarefa de persistncia dos tpicos no dao.
   * @param receiveTimeout Tempo mximo, em milisegundos, que um consumidor tem
   *        entre cada chamada ao mtodo
   *        {@link #receive(String, Serializable, IFilter)} para ser considerado
   *        ativo e no perder mensagens. <br>
   *        O valor mnimo para este parmetro 
   *        {@value MessageStore#MIN_RECEIVE_TIMEOUT}ms. <br>
   *        O valor mnimo para este parmetro 
   *        {@value MessageStore#MAX_RECEIVE_TIMEOUT}ms.
   */
  public MessageBroker(ExecutorService executor, IMessageStoreDAO dao,
    long persistPeriod, long receiveTimeout) {

    IDispatcher<IMessageListener, Message> dispatcher =
      new ExecutorDispatcher<IMessageListener, Message>(executor,
        new MessageListenerDispatcher());

    init(dispatcher, dao, persistPeriod, receiveTimeout);
  }

  /**
   * Construtor.
   * 
   * @param dispatcher Responsvel por entregar mensagens aos consumidores
   *        assncronos.
   * @param dao Objeto utilizado para fazer a persistncia das mensagens dos
   *        tpicos.
   * @param persistPeriod Tempo, em milisegundos, entre sucessivas execues da
   *        tarefa de persistncia dos tpicos no dao.
   * @param receiveTimeout Tempo mximo, em milisegundos, que um consumidor tem
   *        entre cada chamada ao mtodo
   *        {@link #receive(String, Serializable, IFilter)} para ser considerado
   *        ativo e no perder mensagens. <br>
   *        O valor mnimo para este parmetro 
   *        {@value MessageStore#MIN_RECEIVE_TIMEOUT}ms. <br>
   *        O valor mnimo para este parmetro 
   *        {@value MessageStore#MAX_RECEIVE_TIMEOUT}ms.
   */
  public MessageBroker(IDispatcher<IMessageListener, Message> dispatcher,
    IMessageStoreDAO dao, long persistPeriod, long receiveTimeout) {

    init(dispatcher, dao, persistPeriod, receiveTimeout);
  }

  /**
   * <p>
   * Inicializa a instncia do MessageBroker.
   * </p>
   * <p>
   * No pode ser implementado como um construtor privado pois j existe um
   * construtor com os mesmos parmetros -
   * {@link #MessageBroker(IDispatcher, IMessageStoreDAO, long, long)} -,
   * variando apenas no tipo de {@link IDispatcher despachante} que recebe.
   * </p>
   * 
   * @param dispatcher Responsvel por entregar mensagens aos consumidores
   *        assncronos.
   * @param dao Objeto utilizado para fazer a persistncia das mensagens dos
   *        tpicos.
   * @param persistPeriod Tempo, em milisegundos, entre sucessivas execues da
   *        tarefa de persistncia dos tpicos no dao.
   * @param receiveTimeout Tempo mximo, em milisegundos, que um consumidor tem
   *        entre cada chamada ao mtodo
   *        {@link #receive(String, Serializable, IFilter)} para ser considerado
   *        ativo e no perder mensagens. <br>
   *        O valor mnimo para este parmetro 
   *        {@value MessageStore#MIN_RECEIVE_TIMEOUT}ms. <br>
   *        O valor mnimo para este parmetro 
   *        {@value MessageStore#MAX_RECEIVE_TIMEOUT}ms.
   */
  private void init(IDispatcher<IMessageListener, Message> dispatcher,
    IMessageStoreDAO dao, long persistPeriod, long receiveTimeout) {
    this.dispatcher = dispatcher;
    this.dao = dao;
    this.managerPeriod = Math.max(persistPeriod, MIN_MANAGER_PERIOD);
    this.receiveTimeout = receiveTimeout;

    // Mapa de tpicos por nome.
    this.topicsByName = new HashMap<String, Topic>();
    // Flag indicando se o timer de gerenciamento foi inicializado.
    this.started = new AtomicBoolean(false);
    // Tarefa de gerenciamento.
    this.persistTask = new TimerTask() {
      @Override
      public void run() {
        ArrayList<MessageStore> stores = new ArrayList<MessageStore>();
        for (Topic aTopic : topicsByName.values()) {
          MessageStore aStore = aTopic.getMessageStore();
          stores.add(aStore);
        }
        // Salva as stores.
        MessageBroker.this.dao.saveAllMessageStores(stores);
      }
    };
  }

  /**
   * Inicializa a gerncia dos tpicos.
   */
  public synchronized void start() {
    if (started.compareAndSet(false, true)) {
      // Recupera os tpicos persistidos.
      topicsByName.clear();
      for (MessageStore store : dao.getAllMessageStores()) {
        // Corrige o expirationDelay da store.
        store.setReceiveTimeout(receiveTimeout);
        // Cria um tpico com ela.
        Topic topic = new Topic(store, dispatcher);
        // Faz cache do tpico.
        topicsByName.put(store.getName(), topic);
      }
      // Cria um timer para executar tarefas recorrentes de gerncia dos tpicos.
      managerTimer = new Timer();
      managerTimer.schedule(persistTask, 0, managerPeriod);
    }
  }

  /**
   * Interrompe a gerencia dos tpicos.
   */
  public synchronized void stop() {
    if (started.compareAndSet(true, false)) {
      // Para o timer de persistncia de tpicos.
      managerTimer.cancel();
      managerTimer = null;
      // Persiste os tpicos com as ltimas alteraes.
      //  necessrio, pois no se sabe quando foi a ltima vez que o timer persistiu.   
      ArrayList<MessageStore> pools = new ArrayList<MessageStore>();
      for (Topic aTopic : topicsByName.values()) {
        pools.add(aTopic.getMessageStore());
      }
      dao.saveAllMessageStores(pools);
    }
  }

  /**
   * Cria um novo identificador de consumidor.
   * 
   * @return um novo identificador de consumidor.
   * 
   * @see #receive(String, Serializable, IFilter)
   * @see #setMessageListener(String, Serializable, IMessageListener, IFilter)
   * @see #removeMessageListener(String, Serializable)
   */
  public Serializable createConsumerId() {
    return new ConsumerId(brokerId);
  }

  /**
   * Recebe novas mensagens, destinadas a seo do usurio conectado ao servio.
   * 
   * @param topicName Nome do tpico de onde as mensagens sero obtidas.
   * @param consumerId Identificador do consumidor.
   * @param filter filtro que determina as mensagens que sero consumidas.
   * 
   * @return Novas mensagens de um determinado tpico.
   */
  public Message[] receive(String topicName, Serializable consumerId,
    IFilter<Message> filter) {
    // Valida o identificador do consumidor.
    validateConsumerId(consumerId);

    Topic topic = getTopic(topicName, false);
    if (topic != null) {
      return topic.receive(consumerId, filter);
    }

    return new Message[0];
  }

  /**
   * Atribui o ouvinte de um determinado consumidor a um determinado tpico.
   * 
   * @param topicName Nome do tpico no qual o ouvinte ser cadastrado.
   * @param consumerId Identificador do consumidor ao qual o ouvinte pertence.
   * @param listener Ouvinte de mensagens.
   * @param filter Filtro que determina as mensagens que sero repassadas ao
   *        ouvinte.
   */
  public void setMessageListener(String topicName, Serializable consumerId,
    IMessageListener listener, IFilter<Message> filter) {
    // Valida o identificador do consumidor.
    validateConsumerId(consumerId);

    Topic topic = getTopic(topicName, true);
    topic.subscribe(consumerId, listener, filter);
  }

  /**
   * Remove o ouvinte de um determinado consumidor de um tpico.
   * 
   * @param topicName Tpico ao qual o ouvinte a ser retirado pertence.
   * @param consumerId Identificador do consumidor ao qual o ouvinte a ser
   *        retirado pertence.
   */
  public void removeMessageListener(String topicName, Serializable consumerId) {
    // Valida o identificador do consumidor.
    validateConsumerId(consumerId);

    Topic topic = getTopic(topicName, false);
    if (null != topic) {
      topic.unsubscribe(consumerId);
    }
  }

  /**
   * Envia uma mensagem para vrios tpicos.
   * 
   * @param message Mensagem a ser enviada.
   * @param timeToLive Tempo, em milisegundos, que essa mensagem deve persistir
   *        at que seja consumida.
   * @param topicsDestination Nome dos tpicos de destino.
   */
  public void send(Message message, long timeToLive,
    String... topicsDestination) {
    message.setId(UUID.randomUUID());
    for (String aTopicName : topicsDestination) {
      try {
        Topic topic = getTopic(aTopicName, true);
        topic.publish(message, timeToLive);
      }
      catch (Exception e) {
        LogRecord record =
          new LogRecord(Level.WARNING,
            "Error while publishing a message on topic: " + aTopicName + " .");
        record.setThrown(e);
        LOGGER.log(record);
      }
    }
  }

  /**
   * <p>
   * Obtm um tpico.
   * </p>
   * <p>
   * Se o tpico no existir e o parmetro reate' {@code true}, ele ser
   * criado.
   * </p>
   * 
   * @param name Nome do tpico.
   * @param create {@code true} indica que o tpico deve ser criado caso no
   *        exista.
   * 
   * @return Retorna o tpico; ou {@code null} caso esta instncia no tenha
   *         sido inicializada atravs do mtodo {@link #start()}, ou caso ele
   *         no exista e no tenha sido criado.
   */
  private Topic getTopic(String name, boolean create) {
    if (!started.get()) {
      return null;
    }

    Topic topic = topicsByName.get(name);
    if (topic == null && create) {
      topic = new Topic(new MessageStore(name, receiveTimeout), dispatcher);
      topicsByName.put(name, topic);
    }
    return topic;
  }

  /**
   * <p>
   * Valida o identificador do consumidor.
   * </p>
   * <p>
   * Verifica se ele foi criado por esta instncia.
   * </p>
   * 
   * @param consumerId identificador a ser validado.
   * 
   * @throws NullPointerException Se o identificador for nulo.
   * @throws IllegalArgumentException Se o identificador no for vlido.
   */
  private void validateConsumerId(Serializable consumerId) {

    if (consumerId == null) {
      throw new NullPointerException("consumerId == null");
    }

    if (ConsumerId.class != consumerId.getClass()) {
      throw new IllegalArgumentException("Wrong consumer ID type.");
    }

    ConsumerId id = (ConsumerId) consumerId;
    if (!id.getBrokerId().equals(brokerId)) {
      throw new IllegalArgumentException("Consumer ID belongs to other broker.");
    }
  }
}
