/**
 * $Id$
 */

package csbase.util.messages;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;

import tecgraf.javautils.core.filter.IFilter;
import tecgraf.javautils.core.lng.LNG;

/**
 * Responsvel pelo armazenamento das mensagens. Tambm mantm a informao de
 * quais mensagens foram recebidas por quais consumidores.
 *
 * O MessageStore permite consumir as mensagens de duas maneiras:
 * <ul>
 * <li>Assncrona - chamando o mtodo {@link #peek(Serializable, IFilter)}
 * seguido do
 * {@link csbase.util.messages.MessageStore.Entry#setAcknowledgedBy(Serializable)
 * Entry#setAcknowledgedBy(Serializable)} para marcar cada mensagem como
 * recebida</li>
 * <li>Sncrona - atravs do mtodo {@link #receive(Serializable, IFilter)} que
 * j marca as mensagens como recebidas antes de devolv-las.</li>
 * </ul>
 * 
 * Para garantir que todas as mensagens tenham chance de serem consumidas,
 * independente de seu tempo de vida, assim que esse termina elas so postas em
 * estado de expirando, ao invs de expirarem imediatamente. Esse estado dura
 * {@link #receiveTimeout}ms. Os consumidores que estavam ativos no momento em
 * que elas entraram neste estado podem consumi-las, se j no as consumiram,
 * antes que elas expirem.
 * 
 * Para evitar que clientes que caram e voltaram com outro identificador de
 * consumidor recebam uma mensagem duplicada, assim que uma mensagem  marcada
 * como recebida, ela entra em estado de expirando e fica salvo quais
 * consumidores estavam ativos no momento em que ela trocou de estado. a partir
 * da, apenas esses poderam consumi-la antes que ela entre em estado de
 * expirada. Assim, se um consumidor recebeu aquela mensagem, perdeu a conexo,
 * voltou e obteve um novo identificador de consumidor, ele no tem mais acesso
 * aquela mensagem por que aquele identificador no estava ativo no primeiro
 * momento em que ela foi marcada como recebida.
 * 
 * Sempre que chamar um dos mtodos para se obter as mensagens,
 * {@link #receive(Serializable, IFilter)} ou
 * {@link #peek(Serializable, IFilter)}, o consumidor ser marcado como ativo e
 * assim permanecer por {@link #receiveTimeout}ms. Sendo assim, para ele se
 * manter ativo, basta que faa chamadas consecutivas a esses mtodos com um
 * intervalo de tempo menor ou igual a {@link #receiveTimeout}.
 *
 * @author Tecgraf/PUC-Rio
 */
public class MessageStore implements Serializable {

  /**
   * Nome da classe (utilizado para internacionalizao a priori).
   */
  final static String className = MessageStore.class.getSimpleName();
	
  /**
   * Verso da classe.
   */
  private static final long serialVersionUID = 1L;

  /**
   * Utilizado para "<i>logar</i>" informaes no sistema.
   */
  private static final Logger LOGGER = Logger.getLogger(MessageStore.class
    .getName());

  /**
   * Tempo mnimo que um consumidor tem entre cada chamada ao mtodo
   * {@link #receive(Serializable, IFilter)} ou
   * {@link #peek(Serializable, IFilter)} para ser considerado ativo.
   * 
   * @see #receiveTimeout
   */
  public static final long MIN_RECEIVE_TIMEOUT = TimeUnit.SECONDS.toMillis(20);

  /**
   * Tempo mximo que um consumidor tem entre cada chamada ao mtodo
   * {@link #receive(Serializable, IFilter)} ou
   * {@link #peek(Serializable, IFilter)} para ser considerado ativo.
   * 
   * @see #receiveTimeout
   */
  public static final long MAX_RECEIVE_TIMEOUT = TimeUnit.MINUTES.toMillis(10);

  /**
   * @serial Nome deste tpico.
   * @since 1.0
   */
  private String name;

  /**
   * @serial Tempo mximo, em milisegundos, que um consumidor tem entre cada
   *         chamada ao mtodo {@link #receive(Serializable, IFilter)} ou
   *         {@link #peek(Serializable, IFilter)} para ser considerado ativo.
   *         Esse tambm  o tempo que os consumidores ativos tem para consumir
   *         uma mensagem expirada. O valor mnimo para este parmetro 
   *         {@link #MIN_RECEIVE_TIMEOUT}.
   * @since 1.0
   */
  private long receiveTimeout;

  /**
   * @serial Entradas de mensagens pelos ids das mensagens.
   * @since 1.0
   */
  private Map<UUID, Entry> entries;
  /**
   * Lock para ler e escrever no mapa de entradas de mensagens.
   */
  private transient ReentrantReadWriteLock entriesLock;

  /**
   * <p>
   * Mapa tendo como chave o identificador dos consumidores que esto recebendo
   * mensagens e como valor a data de seu ltimo acesso.
   * </p>
   * <p>
   * Se a diferena da data do ltimo acesso para a data corrente for maior que
   * {@link #receiveTimeout},  considerado que o consumidor no est mais
   * recebendo mensagens.
   * </p>
   */
  private Map<Serializable, Long> consumers;
  /**
   * Lock para ler e escrever no mapa de consumidores ativos.
   */
  private transient ReentrantReadWriteLock consumersLock;

  /**
   * Construtor.
   * 
   * @param name Nome deste tpico.
   * @param receiveTimeout Tempo mximo, em milisegundos, que um consumidor tem
   *        entre cada chamada ao mtodo {@link #receive(Serializable, IFilter)}
   *        ou {@link #peek(Serializable, IFilter)} para ser considerado ativo.
   *        O valor mnimo para este parmetro  {@link #MIN_RECEIVE_TIMEOUT}.
   */
  public MessageStore(String name, long receiveTimeout) {
    this.name = name;
    this.entries = new HashMap<UUID, Entry>();
    this.entriesLock = new ReentrantReadWriteLock();
    this.consumers = new HashMap<Serializable, Long>();
    this.consumersLock = new ReentrantReadWriteLock();

    setReceiveTimeout(receiveTimeout);
  }

  /**
   * Obtm o nome deste tpico.
   * 
   * @return o nome deste tpico.
   */
  public String getName() {
    return name;
  }

  /**
   * Atribui atraso na coleta de mensagens expiradas.
   * 
   * @param timeout Tempo mximo, em milisegundos, que um consumidor tem entre
   *        cada chamada ao mtodo {@link #receive(Serializable, IFilter)} ou
   *        {@link #peek(Serializable, IFilter)} para ser considerado ativo. O
   *        valor mnimo para este parmetro  {@link #MIN_RECEIVE_TIMEOUT}. O
   *        valor mximo para este parmetro  {@link #MAX_RECEIVE_TIMEOUT}.
   */
  public void setReceiveTimeout(long timeout) {
    timeout = Math.max(MIN_RECEIVE_TIMEOUT, timeout);
    this.receiveTimeout = Math.min(MAX_RECEIVE_TIMEOUT, timeout);
  }

  /**
   * Adiciona uma nova mensagem a esta estrutura.
   * 
   * @param message mensagem a ser adicionada.
   * @param timeToLive tempo de vida da mensagem em milisegundos.
   * 
   * @return Entrada representando a mensagem publicada.
   */
  public Entry publish(Message message, long timeToLive) {
    Entry entry = new Entry(message, timeToLive);
    entriesLock.writeLock().lock();
    try {
      entries.put(message.getId(), entry);
    }
    finally {
      entriesLock.writeLock().unlock();
    }
    return entry;
  }

  /**
   * Obtm mensagens que ainda no expiradas e no recebidas por um determinado
   * consumidor. Essas mensagens sero marcadas como lidas.
   * 
   * @param consumerId Identificador do consumidor.
   * @param filter filtro utilizado para obter as mensagens que se tem
   *        interesse.
   * @return Mensagens que ainda no expiradas e no recebidas por um
   *         determinado consumidor.
   */
  public Message[] receive(Serializable consumerId, IFilter<Message> filter) {
    return get(consumerId, filter, true);
  }

  /**
   * Obtm mensagens que ainda no expiradas e no recebidas por um determinado
   * consumidor. Essas mensagens no sero marcadas como lidas.
   * 
   * @param consumerId Identificador do consumidor.
   * @param filter filtro utilizado para obter as mensagens que se tem
   *        interesse.
   * @return Mensagens que ainda no expiradas e no recebidas por um
   *         determinado consumidor.
   */
  public Message[] peek(Serializable consumerId, IFilter<Message> filter) {
    return get(consumerId, filter, false);
  }

  /**
   * Obtm a entrada que encapsula uma determinada mensagem.
   * 
   * @param messageId Identificador nico da mensagem.
   * 
   * @return a entrada que encapsula uma determinada mensagem.
   */
  public Entry getEntry(UUID messageId) {
    Entry anEntry;
    entriesLock.readLock().lock();
    try {
      anEntry = entries.get(messageId);
    }
    finally {
      entriesLock.readLock().unlock();
    }
    return anEntry;
  }

  /**
   * Apaga as mensagens que j expiraram.
   */
  private void clearExpireds() {
    entriesLock.writeLock().lock();
    try {
      Iterator<Map.Entry<UUID, Entry>> iterator = entries.entrySet().iterator();

      while (iterator.hasNext()) {
        Map.Entry<UUID, Entry> anEntry = iterator.next();
        if (anEntry.getValue().isDiscardable()) {
          iterator.remove();
        }
      }
    }
    finally {
      entriesLock.writeLock().unlock();
    }
  }

  /**
   * Apaga o registro dos consumidores que esto inativos a mais tempo que
   * {@code #receiveTimeout}. ...
   */
  private void cleanTimedOutConsumers() {
    consumersLock.writeLock().lock();
    try {
      Iterator<Map.Entry<Serializable, Long>> iterator = consumers.entrySet()
        .iterator();

      long now = System.currentTimeMillis();
      while (iterator.hasNext()) {
        Map.Entry<Serializable, Long> anEntry = iterator.next();
        long lastReceive = anEntry.getValue();
        if (lastReceive + receiveTimeout < now) {
          iterator.remove();
        }
      }
    }
    finally {
      consumersLock.writeLock().unlock();
    }
  }

  /**
   * Obtm mensagens que ainda no expiradas e no recebidas por um determinado
   * consumidor.
   * 
   * @param consumerId Identificador do consumidor.
   * @param filter filtro utilizado para obter as mensagens que se tem
   *        interesse.
   * @param ack se <tt>true</tt>, a mensagem ser marcada como recebida.
   * @return Mensagens que ainda no expiradas e no recebidas por um
   *         determinado consumidor.
   */
  private Message[] get(Serializable consumerId, IFilter<Message> filter,
    boolean ack) {
    consumersLock.writeLock().lock();
    try {
      consumers.put(consumerId, System.currentTimeMillis());
    }
    finally {
      consumersLock.writeLock().unlock();
    }

    List<Message> messages = new ArrayList<Message>();
    entriesLock.readLock().lock();
    try {
      for (Entry anEntry : entries.values()) {
        if (filter.accept(anEntry.message) && anEntry.isAvailableFor(
          consumerId)) {
          messages.add(anEntry.message);
          if (ack) {
            anEntry.setAcknowledgedBy(consumerId);
          }
        }
      }
      return messages.toArray(new Message[0]);
    }
    finally {
      entriesLock.readLock().unlock();
    }
  }

  /**
   * Serializa os dados dessa instncia em um {@link ObjectOutputStream}.
   * 
   * @param out {@link ObjectOutputStream} onde os dados sero serializados.
   * @throws IOException Indica que houve um erro ao escrever os dados.
   */
  private void writeObject(ObjectOutputStream out) throws IOException {
    // Limpa a MessageStore de mensagens descartveis e consumidores inativos.
    cleanTimedOutConsumers();
    clearExpireds();

    out.writeUTF(name);

    out.writeLong(receiveTimeout);

    entriesLock.readLock().lock();
    try {
      out.writeInt(entries.size());
      for (Entry entry : entries.values()) {
        try {
          out.writeObject(entry.message);
          out.writeLong(entry.expires);
          out.writeObject(entry.receiveds);
        }
        catch (Exception e) {
          LogRecord record =
            new LogRecord(Level.WARNING, LNG.get(className + ".warning.serialize.message"));
          record.setThrown(e);
          LOGGER.log(record);
        }
      }
    }
    finally {
      entriesLock.readLock().unlock();
    }

    consumersLock.readLock().lock();
    try {
      out.writeInt(consumers.size());
      for (Map.Entry<Serializable, Long> entry : consumers.entrySet()) {
        try {
          out.writeObject(entry.getKey());
          out.writeLong(entry.getValue());
        }
        catch (Exception e) {
        	final String className = MessageStore.class.getSimpleName();
          LogRecord record =
            new LogRecord(Level.WARNING, LNG.get(className + ".warning.serialize.consumer"));
          record.setThrown(e);
          LOGGER.log(record);
        }
      }
    }
    finally {
      consumersLock.readLock().unlock();
    }

    out.flush();
  }

  /**
   * Desserializa os dados dessa instncia a partir de um
   * {@link ObjectInputStream}.
   * 
   * @param in {@link ObjectInputStream} a partir do qual os dados sero lidos.
   * @throws IOException Indica que houve um erro ao ler os dados.
   * @throws ClassNotFoundException Indica que houve um erro ao ler os dados.
   */
  @SuppressWarnings("unchecked")
  private void readObject(ObjectInputStream in) throws IOException,
    ClassNotFoundException {
    /*
     * Para evitar voar durante a desserializao, qualquer exceo lanada
     * durante a leitura ser capturada e ser atribudo um valor padro ao que
     * estava sendo lido.
     */

    // Desserializa o nome da MessageStore. 
    this.name = "";
    try {
      this.name = in.readUTF();
    }
    catch (Exception e) {
      LogRecord record =
        new LogRecord(Level.WARNING, LNG.get(className + ".warning.deserialize.name"));
      record.setThrown(e);
      LOGGER.log(record);
    }

    // Desserializando tempo mximo para consumo de mensagens. 
    this.receiveTimeout = MAX_RECEIVE_TIMEOUT;
    try {
      this.receiveTimeout = in.readLong();
    }
    catch (Exception e) {
      LogRecord record =
        new LogRecord(Level.WARNING, LNG.get(className + ".warning.deserialize.timeout"));
      record.setThrown(e);
      LOGGER.log(record);
    }

    this.entriesLock = new ReentrantReadWriteLock();
    entriesLock.writeLock().lock();
    try {
      // Obtendo o nmero de entradas de mensagens a serem desserializadas.
      int entriesSize = 0;
      try {
        entriesSize = in.readInt();
      }
      catch (Exception e) {
        LogRecord record =
          new LogRecord(Level.WARNING, LNG.get(className + ".warning.deserialize.messagenum"));
        record.setThrown(e);
        LOGGER.log(record);
      }
      // Desserializando entradas de mensagens.
      this.entries = new HashMap<UUID, Entry>(entriesSize);
      for (int inx = 0; inx < entriesSize; inx++) {
        try {
          Message message = (Message) in.readObject();
          long expires = in.readLong();
          Map<Serializable, Boolean> receiveds = (Map<Serializable, Boolean>) in
            .readObject();
          Entry entry = new Entry(message, expires, receiveds);
          this.entries.put(entry.message.getId(), entry);
        }
        catch (Exception e) {
          LogRecord record =
            new LogRecord(Level.WARNING, LNG.get(className + ".warning.deserialize.message"));
          record.setThrown(e);
          LOGGER.log(record);
        }
      }
    }
    finally {
      entriesLock.writeLock().unlock();
    }

    this.consumersLock = new ReentrantReadWriteLock();
    consumersLock.writeLock().lock();
    try {
      // Obtendo o nmero de consumidores ativos a serem desserializados.
      int consumersSize = 0;
      try {
        consumersSize = in.readInt();
      }
      catch (Exception e) {
        LogRecord record =
          new LogRecord(Level.WARNING, LNG.get(className + ".warning.deserialize.consumernum"));
        record.setThrown(e);
        LOGGER.log(record);
      }
      // Desserializando consumidores ativos.
      this.consumers = new HashMap<Serializable, Long>(consumersSize);
      for (int inx = 0; inx < consumersSize; inx++) {
        try {
          Serializable key = (Serializable) in.readObject();
          Long value = in.readLong();

          this.consumers.put(key, value);
        }
        catch (Exception e) {
          LogRecord record =
            new LogRecord(Level.WARNING, LNG.get(className + ".warning.deserialize.consumer"));
          record.setThrown(e);
          LOGGER.log(record);
        }
      }
    }
    finally {
      consumersLock.writeLock().unlock();
    }

    // Limpa a MessageStore de mensagens descartveis e consumidores inativos.
    cleanTimedOutConsumers();
    clearExpireds();
  }

  /**
   * <p>
   * Representa uma mensagem dentro do tpico.
   * </p>
   * <p>
   * Encapsula a mensagem junto com seu status, quais consumidores tem acesso a
   * ela e se j a receberam.
   * </p>
   * 
   * @author Tecgraf
   */
  class Entry {

    /**
     * Mensagem no tpico.
     */
    private Message message;

    /**
     * Data de expirao.
     */
    private long expires;
    /**
     * Mapa contendo o identificador dos consumidores que j receberam, ou que
     * ainda podem receber essa mensagem e como chave uma <i>flag</i> indicando
     * o status do recebimento. {@code true} indica que a mensagem j foi
     * recebida.
     */
    private Map<Serializable, Boolean> receiveds;

    /**
     * Lista de usurios que esto em processo de receber a mensagem.
     */
    private Set<Serializable> receiving;

    /**
     * Construtor utilizado pela desserializao da {@link MessageStore}.
     * 
     * @param message Mensagem encapsulada.
     * @param expires Data de expirao da mensagem.
     * @param receiveds Mapa contendo consumidores que receberam, ou faltam
     *        receber a mensagem.
     */
    private Entry(Message message, long expires,
      Map<Serializable, Boolean> receiveds) {
      this.message = message;
      this.expires = expires;
      this.receiveds = receiveds;
      this.receiving = new HashSet<Serializable>();
    }

    /**
     * Construtor.
     * 
     * @param message Mensagem sendo encapsulada.
     * @param timeToLive Tempo, em milisegundos, que essa mensagem deve
     *        persistir em um servio de mensagem, at que seja consumida.
     */
    public Entry(Message message, long timeToLive) {
      if (message == null) {
        throw new IllegalArgumentException("message == null");
      }
      if (timeToLive < 0) {
        throw new IllegalArgumentException(LNG.get(className + ".illegalarg.timetolive.negative"));
      }

      this.message = message;
      this.expires = System.currentTimeMillis() + timeToLive;
      this.receiveds = new HashMap<Serializable, Boolean>();
      this.receiving = new HashSet<Serializable>();
    }

    /**
     * Obtm a mensagem representada por esta entrada.
     * 
     * @return a mensagem representada por esta entrada.
     */
    public Message getMessage() {
      return message;
    }

    /**
     * Marca a mensagem desta entrada como sendo enviada, ou no, para um
     * determinado consumidor.
     * 
     * @param consumerId Identificador do consumidor para o qual se est
     *        tentando enviar a mensagem.
     * @param sending <tt>True</tt> se a mensagem est sendo enviada para o
     *        consumidor.
     * @return <tt>True</tt> se a alterao de estado ocorreu com sucesso.
     */
    public synchronized boolean setBeingSentTo(Serializable consumerId,
      boolean sending) {
      if (sending) {
        return this.receiving.add(consumerId);
      }
      else {
        return this.receiving.remove(consumerId);
      }
    }

    /**
     * <p>
     * Usado para marcar a mensagem como recebida por um dado consumidor.
     * </p>
     * <p>
     * Chama {@code #setBeingSentTo(consumerId, false)} indicando que a mensagem
     * no est mais no estado de sendo enviada.
     * </p>
     * 
     * @param consumerId Identificador do consumidor.
     */
    public synchronized void setAcknowledgedBy(Serializable consumerId) {
      long now = System.currentTimeMillis();
      // Verifico se ningum ainda recebeu a mensagem...
      if (this.receiveds.size() == 0) {
        // Se a mensagem ainda no expirou, ...
        if (expires > now) {
          // ... expira a mensagem.
          expires = now;
        }

        /*
         * Marca todos consumidores ativos em modo de recebendo a mensagem. Isso
         * permite que eles possam receb-la antes do timeout de consumo.
         */
        consumersLock.readLock().lock();
        try {
          for (Map.Entry<Serializable, Long> consumerData : consumers
            .entrySet()) {
            Serializable aConsumerId = consumerData.getKey();
            long lastConsume = consumerData.getValue();
            // Ignora os consumidores inativos.
            if (lastConsume + receiveTimeout < now) {
              continue;
            }

            /*
             * Salva os ativos no mapa de consumidores que esto recebendo a
             * mensagem.
             * 
             * No precisa se preocupar com sobrescrever o ack de nenhum
             * consumidor, pois esse cdigo roda apenas na primeira vez que a
             * mensagem for consumida.
             */
            receiveds.put(aConsumerId, false);
          }
        }
        finally {
          consumersLock.readLock().unlock();
        }
      }
      // Marca a mensagem como recebida pelo consumidor passado como parmetro .
      receiveds.put(consumerId, true);
      receiving.remove(consumerId);
    }

    /**
     * <p>
     * Indica se a entrada  descartvel.
     * </p>
     * <p>
     * Uma entrada descartvel  ...
     * </p>
     * 
     * @return {@code true} se essa entrada for descartvel.
     */
    public boolean isDiscardable() {
      return (expires + receiveTimeout) <= System.currentTimeMillis();
    }

    /**
     * Descobre se a mensagem encapsulada por esta entrada est disponvel para
     * ser recebida por um dado consumidor.
     * 
     * @param consumerId COnsumidor com intens"ao de consumir a entrada.
     * 
     * @return {@code true} se a mensagem encapsulada por esta entrada est
     *         disponvel para ser recebida por um dado consumidor.
     */
    public synchronized boolean isAvailableFor(Serializable consumerId) {
      long now = System.currentTimeMillis();

      if (receiving.contains(consumerId)) {
        return false;
      }

      // Verifica se ainda no expirou...
      if (expires > now) {
        /*
         * ... ningum ainda deu acknowledged, pois no 1o acknowledged ele
         * expira a mensagem. Neste caso, essa mensagem est disponvel a todos
         * os consumidores.
         */
        return true;
      }
      // Se expirou...
      else {
        /*
         * ... e deu o timeout para os consumidores pendentes receberem essa
         * mensagem, ...
         */
        if (expires + receiveTimeout < now) {
          /*
           * ... ningum mais tem acesso a essa mensagem e ela se torna
           * descartvel.
           */
          return false;
        }
        // Caso no tenha dado timeout, ...
        else {
          if (receiveds.size() == 0) {
            /*
             * Ningum ainda recebeu essa mensagem, ento no h perigo de
             * duplicata.
             */
            return true;
          }
          /*
           * ... apenas os consumidores que estavam ativos quando a mensagem
           * expirou ainda tem acesso a ela.
           */
          Boolean ack = receiveds.get(consumerId);
          return ack != null && !ack;
        }
      }
    }
  }
}
