/**
 * $Id: ValidationsLogger.java 105234 2010-05-07 19:08:53Z clinio $
 */
package validations.util;

import java.util.logging.ConsoleHandler;
import java.util.logging.FileHandler;
import java.util.logging.Formatter;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;

/**
 * Especializao de {@link Logger} para uso das classes de validao.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ValidationsLogger extends Logger {
  /**
   * Separador para sees.
   */
  private static final String SECTION_SEPARATOR =
    "==============================================================";
  /**
   * Separador simples.
   */
  private static final String SIMPLE_SEPARATOR =
    "--------------------------------------------------------------";
  /**
   * Instncia do formatador bsico usado pelo logger.
   */
  private static Formatter LOG_FORMATTER = new Formatter() {
    @Override
    public String format(LogRecord record) {
      StringBuilder buffer = new StringBuilder();
      buffer.append(record.getLevel().getLocalizedName());
      buffer.append(": ");
      buffer.append(record.getMessage());
      buffer.append('\n');
      return buffer.toString();
    }
  };
  /**
   * Handler para o console.
   */
  private final ConsoleHandler consoleHandler;
  /**
   * Handler para um arquivo (pode ser <code>null</code>).
   */
  private FileHandler fileHandler;
  /**
   * Path para o arquivo de log (pode ser <code>null</code>).
   */
  private String logFilePath;

  /**
   * Construtor. Define um logger sem internacionalizao.
   */
  public ValidationsLogger() {
    super(ValidationsLogger.class.getName(), null);
    setUseParentHandlers(false);
    consoleHandler = new ConsoleHandler();
    consoleHandler.setFormatter(LOG_FORMATTER);
    addHandler(consoleHandler);
    /*
     * precisamos chamar o mtodo da superclasse porque queremos setar o nvel
     * do logger como um todo, e no apenas do ConsoleHandler
     */
    super.setLevel(Level.FINE);
  }

  /**
   * Define o handler para um arquivo.
   * 
   * @param logFilePath path para o arquivo de log
   * @return <code>true</code> se a operao foi bem-sucedida
   */
  public boolean setFileHandler(String logFilePath) {
    if (fileHandler != null) {
      fileHandler.close();
      removeHandler(fileHandler);
    }
    try {
      fileHandler = new FileHandler(logFilePath, true);
    }
    catch (Exception e) {
      severe("Erro configurando log para o arquivo " + logFilePath);
      severe(e.toString());
      return false;
    }
    fileHandler.setFormatter(LOG_FORMATTER);
    fileHandler.setLevel(Level.FINE);
    addHandler(fileHandler);
    this.logFilePath = logFilePath;
    return true;
  }

  /**
   * Obtm o path para o arquivo de log.
   * 
   * @return path para o arquivo de log
   */
  public String getLogFilePath() {
    return logFilePath;
  }

  /**
   * Registra um separador padro, usando um nvel especfico.
   * 
   * @param level o nvel
   */
  public void separator(Level level) {
    log(level, SIMPLE_SEPARATOR);
  }

  /**
   * Registra uma linha em branco, usando um nvel especfico
   * 
   * @param level o nvel
   */
  public void blank(Level level) {
    log(level, "");
  }

  /**
   * Registra um separador padro de seo, usando um nvel especfico.
   * 
   * @param level o nvel
   */
  public void sectionSeparator(Level level) {
    log(level, SECTION_SEPARATOR);
  }

  /**
   * Define o nvel do handler para o console.
   * 
   * @param level o nvel
   */
  public void setConsoleLevel(Level level) {
    consoleHandler.setLevel(level);
  }

  /**
   * Registra uma mensagem apenas no handler para o console.
   * 
   * @param msg a mensagem
   */
  public void toConsole(String msg) {
    publish(consoleHandler, msg);
  }

  /**
   * Registra uma mensagem apenas no handler para arquivo.
   * 
   * @param msg a mensagem
   */
  public void toFile(String msg) {
    if (fileHandler != null) {
      publish(fileHandler, msg);
    }
  }

  /**
   * Registra uma mensagem apenas em um handler especfico.
   * 
   * @param handler o handler
   * @param msg a mensagem
   */
  private void publish(Handler handler, String msg) {
    handler.publish(new LogRecord(handler.getLevel(), msg));
  }

  /**
   * Encerra todos os handlers (console e arquivo).
   */
  public void close() {
    consoleHandler.close();
    if (fileHandler != null) {
      fileHandler.close();
    }
  }

  /**
   * {@inheritDoc}
   * <p>
   * Obtm o nvel do console (o log para arquivo sempre  feito no nvel FINE).
   */
  @Override
  public Level getLevel() {
    return consoleHandler.getLevel();
  }

  /**
   * {@inheritDoc}
   * <p>
   * Define apenas o nvel do console (o log para arquivo sempre  feito no
   * nvel FINE).
   */
  @Override
  public void setLevel(Level newLevel) throws SecurityException {
    consoleHandler.setLevel(newLevel);
  }

  /**
   * Desabilita o logging (todos os loggers so associados ao nvel
   * {@link Level#OFF}).
   */
  public void disable() {
    consoleHandler.setLevel(Level.OFF);
    fileHandler.setLevel(Level.OFF);
  }

  /**
   * Reabilita o logging.
   * 
   * @param level nvel para o console (o log para arquivo  reabilitado no
   *        nvel {@link Level#FINE})
   */
  public void enable(Level level) {
    fileHandler.setLevel(Level.FINE);
    consoleHandler.setLevel(level);
  }

  /**
   * Registra uma exceo. O nome da exceo  registrado com nvel SEVERE, e a
   * pilha de execuo com nvel FINE (i.e. por default aparece apenas no
   * arquivo).
   * 
   * @param e exceo
   */
  public void exception(Throwable e) {
    severe(e.getClass().getName() + " (mais detalhes no arquivo de log)");
    StringBuilder buffer = new StringBuilder("Pilha de execuo:");
    collectStackTrace(buffer, e);
    fine(buffer.toString());
  }

  /**
   * Processa recursivamente as excees que causaram a exceo em questo.
   * 
   * @param buffer buffer com a pilha de execuo
   * @param e exceo a ser processada
   */
  private void collectStackTrace(StringBuilder buffer, Throwable e) {
    if (e == null) {
      return;
    }
    buffer.append('\n');
    buffer.append(e.getClass().getName() + " - " + e.getMessage());
    StackTraceElement[] stackTrace = e.getStackTrace();
    for (StackTraceElement ste : stackTrace) {
      buffer.append("\n    ");
      buffer.append(ste.toString());
    }
    collectStackTrace(buffer, e.getCause());
  }

  /**
   * Registra uma exceo. O nome da exceo  registrado com nvel SEVERE, e a
   * pilha de execuo com nvel FINE (i.e. por default aparece apenas no
   * arquivo).
   * 
   * @param msg mensagem exibida antes da mensagem da exceo
   * @param e exceo
   * 
   * @see #exception(Throwable)
   */
  public void exception(String msg, Throwable e) {
    severe(msg);
    exception(e);
  }
}
