/*
 * $Id: PersistencyMoveValidation.java 115569 2011-02-16 20:50:28Z clinio $
 */
package validations;

import java.io.File;
import java.util.List;

import validations.util.ValidatorUtils;
import csbase.server.services.sgaservice.SGAService;
import csbase.server.services.sharedobjectservice.SharedObjectService;
import csbase.util.FileSystemUtils;

/**
 * Validao de nova estrutura de diretrios de persistncia
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class PersistencyMoveValidation extends AbstractValidation {

  /**
   * Nome do subdiretrio de backup do usersdata.
   */
  private static final String BCK_USERDATA_SUBDIR_NAME = "usersdata";

  /**
   * Nome do subdiretrio de backup do shared-objects.
   */
  private static final String BCK_SHROBJS_SUBDIR_NAME = "shared-objects";

  /**
   * Nome do subdiretrio de backup do persistent-data
   */
  private static final String BCK_PERS_SUBDIR_NAME = "persistent-data";

  /**
   * Nome da propriedade que armazena o novo diretrio de persistncia
   */
  private static final String NEW_PERS_DIR_PROP = "Persistency.newDirectory";

  /**
   * Nome da propriedade que armazena o novo diretrio de persistncia
   */
  private static final String OLD_PERS_DIR_PROP = "Persistency.oldDirectory";

  /**
   * Nome da propriedade do diretrio original shared-objects
   */
  private static final String OLD_SHR_OBJ_DIR_PROP =
    "SharedObjects.oldDirectory";

  /**
   * Nome da propriedade do diretrio original usersdata
   */
  private static final String OLD_USR_DAT_DIR_PROP = "UsersData.oldDirectory";

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean applyPatch() {
    final File oldUsrDatDir = getFileProperty(OLD_USR_DAT_DIR_PROP);
    final File oldShrObjDir = getFileProperty(OLD_SHR_OBJ_DIR_PROP);
    final File newUsrDatDir = getNewUsersDataDir();
    final File newShrObjDir = getNewSharedObjectsDir();

    final File newPersDir = getFileProperty(NEW_PERS_DIR_PROP);
    if (FileSystemUtils.dirExists(newPersDir)) {
      final String path = newPersDir.getAbsolutePath();
      logger.severe("Ja existe novo diretorio: " + path);
      return false;
    }

    if (!newPersDir.mkdirs()) {
      final String path = newPersDir.getAbsolutePath();
      logger.severe("Falha na criao de: " + path);
      return false;
    }

    if (!move(oldUsrDatDir, newUsrDatDir)) {
      return false;
    }
    if (!move(oldShrObjDir, newShrObjDir)) {
      return false;
    }

    final File oldPersDir = getFileProperty(OLD_PERS_DIR_PROP);
    final File[] children = oldPersDir.listFiles();
    final String sep = File.separator;
    if (children != null && children.length > 0) {
      for (File child : children) {
        final String fileName = child.getName();
        final String newPath = newPersDir.getAbsolutePath() + sep + fileName;
        final File newFile = new File(newPath);
        final String fmt = "Movendo elemento [%s] para novo diretrio";
        final String msg = String.format(fmt, newFile.getName());
        logger.fine(msg);
        if (!child.renameTo(newFile)) {
          final String path = newPersDir.getAbsolutePath();
          logger.severe("Falha no move interno de: " + path);
          return false;
        }
      }
    }

    FileSystemUtils.deleteRecursively(oldPersDir, logger);
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void finish() {
  }

  /**
   * Retorn objeto {@code File} de uma propriedade.
   * 
   * @param propName nome da propriedade.
   * @return objeto ou {@code null}
   */
  private File getFileProperty(final String propName) {
    final String dirPath = getMandatorySystemProperty(propName);
    if (dirPath == null) {
      return null;
    }
    final File dir = new File(dirPath);
    return dir;
  }

  /**
   * Retorna o novo diretrio do shared-objects.
   * 
   * @return o diretrio
   */
  private File getNewSharedObjectsDir() {
    final File newPersDir = getFileProperty(NEW_PERS_DIR_PROP);
    final String subDirName = SharedObjectService.OBJECTS_SUBDIR_NAME;
    final String newUsrDatDirPath = newPersDir + File.separator + subDirName;
    return new File(newUsrDatDirPath);
  }

  /**
   * Retorna o novo diretrio do usersdata.
   * 
   * @return o diretrio
   */
  private File getNewUsersDataDir() {
    final File newPersDir = getFileProperty(NEW_PERS_DIR_PROP);
    final String subDirName = SGAService.SGA_GROUPS_SUBDIR_NAME;
    final String newUsrDatDirPath = newPersDir + File.separator + subDirName;
    return new File(newUsrDatDirPath);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void getSpecificFailureMessage(final Status status,
    final List<String> errors) {
    switch (status) {
      case VALIDATION_FAILED:
        errors.add("*** ESTRUTURA DE PERSISTENCIA NO FORMATO ANTIGO");
        break;

      case PATCH_FAILED:
        errors.add("*** A CONVERSO DA PERSISTNCIA RESULTOU EM ERRO");
        break;

      case INIT_FAILED:
        errors.add("*** FALHA NA INICIALIZAO");
        break;

      default:
        errors.add("ESTADO INVLIDO: " + status.toString());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getStartMessage() {
    return "Convertendo estrutura de persistncia do servidor";
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getSuccessMessage(final Status status) {
    switch (status) {
      case VALIDATION_OK:
        return "*** REESTRUTURAO DE PERSISTNCIA J FOI FEITA";

      case PATCH_OK:
        return "*** REESTRUTURAO DE PERSISTNCIA REALIZADA COM SUCESSO";

      default:
        return "ESTADO INVLIDO: " + status.toString();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean init() {
    final File oldPersDir = getFileProperty(OLD_PERS_DIR_PROP);
    final File newPersDir = getFileProperty(NEW_PERS_DIR_PROP);
    final File oldShrObjDir = getFileProperty(OLD_SHR_OBJ_DIR_PROP);
    final File oldUsrDatDir = getFileProperty(OLD_USR_DAT_DIR_PROP);

    if (oldShrObjDir == null || newPersDir == null || oldUsrDatDir == null
      || oldPersDir == null) {
      return false;
    }

    final String nPath = newPersDir.getAbsolutePath();
    final String oPath = oldPersDir.getAbsolutePath();
    if (nPath.equals(oPath)) {
      final String msg =
        "O novo diretrio de persistncia no pode ser o mesmo do antigo!";
      logger.severe(msg);
      return false;
    }

    return true;
  }

  /**
   * Move
   * 
   * @param srcDir diretrio origem
   * @param dstDir diretrio destino
   * @return indicativo de sucesso.
   */
  private boolean move(final File srcDir, final File dstDir) {
    final String srcPath = srcDir.getAbsolutePath();
    final String destPath = dstDir.getAbsolutePath();
    if (!srcDir.renameTo(dstDir)) {
      logger.severe("Falha no move: " + srcPath + " para " + destPath);
      return false;
    }
    logger.fine("Movendo " + srcPath + " -> " + destPath);
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean backupData() {
    final String bckPath = getBackupDirPath();
    final String sep = File.separator;

    final File oldUsrDatDir = getFileProperty(OLD_USR_DAT_DIR_PROP);
    final String bckUsrDatPath = bckPath + sep + BCK_USERDATA_SUBDIR_NAME;
    final File bckUsrDatDir = new File(bckUsrDatPath);
    FileSystemUtils.deleteRecursively(bckUsrDatDir, logger);
    ValidatorUtils.copyDirectory(oldUsrDatDir, bckUsrDatDir, logger);

    final File oldShrObjDir = getFileProperty(OLD_SHR_OBJ_DIR_PROP);
    final String bckShrObjPath = bckPath + sep + BCK_SHROBJS_SUBDIR_NAME;
    final File bckShrObjDir = new File(bckShrObjPath);
    FileSystemUtils.deleteRecursively(bckShrObjDir, logger);
    ValidatorUtils.copyDirectory(oldShrObjDir, bckShrObjDir, logger);

    final File oldPersDir = getFileProperty(OLD_PERS_DIR_PROP);
    final String bckPersPath = bckPath + sep + BCK_PERS_SUBDIR_NAME;
    final File bckPersDir = new File(bckPersPath);
    FileSystemUtils.deleteRecursively(bckPersDir, logger);
    ValidatorUtils.copyDirectory(oldPersDir, bckPersDir, logger);

    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean restoreBackup() {
    final String bckPath = getBackupDirPath();
    final String sep = File.separator;

    final File newPersDir = getFileProperty(NEW_PERS_DIR_PROP);
    FileSystemUtils.deleteRecursively(newPersDir, logger);

    final File oldUsrDatDir = getFileProperty(OLD_USR_DAT_DIR_PROP);
    final String bckUsrDatPath = bckPath + sep + BCK_USERDATA_SUBDIR_NAME;
    final File bckUsrDatDir = new File(bckUsrDatPath);
    FileSystemUtils.deleteRecursively(oldUsrDatDir, logger);
    ValidatorUtils.copyDirectory(bckUsrDatDir, oldUsrDatDir, logger);

    final File oldShrObjDir = getFileProperty(OLD_SHR_OBJ_DIR_PROP);
    final String bckShrObjPath = bckPath + sep + BCK_SHROBJS_SUBDIR_NAME;
    final File bckShrObjDir = new File(bckShrObjPath);
    FileSystemUtils.deleteRecursively(oldShrObjDir, logger);
    ValidatorUtils.copyDirectory(bckShrObjDir, oldShrObjDir, logger);

    final File oldPersDir = getFileProperty(OLD_PERS_DIR_PROP);
    final String bckPersPath = bckPath + sep + BCK_PERS_SUBDIR_NAME;
    final File bckPersDir = new File(bckPersPath);
    FileSystemUtils.deleteRecursively(oldPersDir, logger);
    ValidatorUtils.copyDirectory(bckPersDir, oldPersDir, logger);

    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean validate() {
    final File newPersDir = getFileProperty(NEW_PERS_DIR_PROP);
    final File newUsrDatDir = getNewUsersDataDir();
    final File newShrObjDir = getNewSharedObjectsDir();
    if (FileSystemUtils.dirExists(newPersDir)
      && FileSystemUtils.dirExists(newUsrDatDir)
      && FileSystemUtils.dirExists(newShrObjDir)) {
      logger.info("Novos diretrios de persistncia j existem. VALIDADO!");
      return true;
    }

    final File oldPersDir = getFileProperty(OLD_PERS_DIR_PROP);
    final File oldShrObjDir = getFileProperty(OLD_SHR_OBJ_DIR_PROP);
    final File oldUsrDatDir = getFileProperty(OLD_USR_DAT_DIR_PROP);
    if (!FileSystemUtils.dirExists(oldPersDir)
      && !FileSystemUtils.dirExists(oldUsrDatDir)
      && !FileSystemUtils.dirExists(oldShrObjDir)) {
      logger
        .info("No existem os diretrios de persistncia no formato antigo. VALIDADO!");
      return true;
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean runsOnlyOnce() {
    return true;
  }
}
