package csdk.v1_0.api.filesystem.local;

import java.awt.Window;

import csdk.v1_0.api.core.IContext;
import csdk.v1_0.api.filesystem.FileSelectionType;
import csdk.v1_0.api.filesystem.IFile;

/**
 * Define os mtodos disponveis para acesso da aplicao ao sistema de arquivos
 * local do cliente. A vantagem de se usar este contexto ao invs de acessar
 * diretamente os arquivos pelas classes de I/O padro  a forma transparente de
 * tratar arquivos locais e de projeto na aplicao atravs da interface
 * {@link IFile}.
 */
public interface ILocalFileSystemContext extends IContext {

  /**
   * Mostra um navegador para seleo de um arquivo local em modo de leitura.
   * 
   * @param fileTypes os tipos de arquivos selecionveis no navegador.
   * @param allowAll determina a incluso da opo de selecionar qualquer tipo
   *        de arquivo, alm dos listados no parmetro fileTypes.
   * @param currentLocalDirectory diretrio inicial do navegador de arquivos.
   * @param selectionType modo de seleo do navegador (somente arquivos ou
   *        somente diretrios).
   * @param window a janela-me do navegador de arquivos.
   * @return o arquivo selecionado ou <code>null</code> caso a operao tenha
   *         sido cancelada.
   * @throws LocalFileSystemException em caso de erro ao acessar o sistema de
   *         arquivos local.
   */
  public IFile browseLocalFileInOpenMode(String[] fileTypes, boolean allowAll,
    IFile currentLocalDirectory, FileSelectionType selectionType, Window window)
    throws LocalFileSystemException;

  /**
   * Mostra um navegador para seleo de mltiplos arquivos locais em modo de
   * leitura.
   * 
   * @param fileTypes os tipos de arquivos selecionveis
   * @param allowAll determina a incluso da opo de selecionar qualquer tipo
   *        de arquivo, alm dos listados no parmetro fileTypes.
   * @param currentLocalDirectory diretrio inicial do navegador de arquivos.
   * @param selectionType tipo de seleo do navegador (somente arquivos ou
   *        somente diretrios).
   * @param window a janela-me do navegador de arquivos.
   * @return o arquivo selecionado ou <code>null</code> caso a operao tenha
   *         sido cancelada.
   * @throws LocalFileSystemException em caso de erro ao acessar o sistema de
   *         arquivos local.
   */
  public IFile[] browseMultipleLocalFilesInOpenMode(String[] fileTypes,
    boolean allowAll, IFile currentLocalDirectory,
    FileSelectionType selectionType, Window window)
    throws LocalFileSystemException;

  /**
   * Mostra um navegador para seleo de um arquivo local em modo de escrita.
   * 
   * @param fileTypes os tipos de arquivo selecionveis
   * @param allowAll determina a incluso da opo de selecionar qualquer tipo
   *        de arquivo, alm dos listados no parmetro fileTypes.
   * @param defaultName o nome padro do arquivo.
   * @param currentLocalDirectory diretrio inicial do navegador de arquivos.
   * @param window a janela-me do navegador de arquivos.
   * @return o arquivo selecionado ou <code>null</code> caso a operao tenha
   *         sido cancelada.
   * @throws LocalFileSystemException em caso de erro ao acessar o sistema de
   *         arquivos local.
   */
  public IFile browseLocalFileInSaveMode(String[] fileTypes, boolean allowAll,
    String defaultName, IFile currentLocalDirectory, Window window)
    throws LocalFileSystemException;

  /**
   * Obtm o arquivo local com o caminho especificado.
   * 
   * @param path o caminho.
   * @return o arquivo ou <code>null</code> caso no exista.
   * @throws LocalFileSystemException em caso de erro ao obter o arquivo.
   */
  public IFile getLocalFile(String[] path) throws LocalFileSystemException;

  /**
   * Cria um arquivo local no diretrio especificado.
   * 
   * @param path o caminho do diretrio pai.
   * @param name o nome do arquivo.
   * @param isDir determina se o arquivo a ser criado  um diretrio.
   * @return o arquivo criado.
   * @throws LocalFileSystemException em caso de erro ao criar o arquivo.
   */
  public IFile createLocalFile(String[] path, String name, boolean isDir)
    throws LocalFileSystemException;

  /**
   * Remove um arquivo ou diretrio local. Para remover um diretrio, este
   * precisa estar vazio.
   * 
   * @param path o caminho para o arquivo.
   * @return <code>true</code> caso a remoo seja feita com sucesso ou
   *         <code>false</code> caso contrrio.
   * @throws LocalFileSystemException em caso de erro ao remover o arquivo.
   */
  public boolean removeLocalFile(String[] path) throws LocalFileSystemException;
}