package csdk.v2.helper;

import java.awt.Window;

import csdk.v2.api.application.ApplicationException;
import csdk.v2.api.command.CommandException;
import csdk.v2.api.command.IAlgorithmTemplate;
import csdk.v2.api.command.ICommandContext;
import csdk.v2.api.command.ICommandObserver;
import csdk.v2.api.command.IFlowTemplate;
import csdk.v2.api.core.ICSDKEnvironment;
import csdk.v2.api.filesystem.FileSelectionType;
import csdk.v2.api.filesystem.IFile;
import csdk.v2.api.filesystem.local.ILocalFileSystemContext;
import csdk.v2.api.filesystem.local.LocalFileSystemException;
import csdk.v2.api.filesystem.project.IProjectContext;
import csdk.v2.api.filesystem.project.IProjectObserver;
import csdk.v2.api.filesystem.project.ProjectException;
import csdk.v2.api.application.IMessage;
import csdk.v2.api.application.manager.IApplicationManagerContext;
import csdk.v2.helper.application.AbstractApplication;

/**
 * Classe abstrata para facilitar o desenvolvimento de aplicaes que acessam a
 * rea de projetos com CSDK.
 */
public abstract class AbstractCSDKApplication extends AbstractApplication
  implements IProjectContext, ILocalFileSystemContext, ICommandContext,
  IApplicationManagerContext {

  /**
   * Contexto de acesso  rea de projetos.
   */
  private IProjectContext projectContext;

  /**
   * Contexto de acesso ao sistema de arquivos local ao cliente.
   */
  private ILocalFileSystemContext localFileSystemContext;

  /**
   * Contexto de acesso  execuo de comandos.
   */
  private ICommandContext commandContext;

  /**
   * Contexto de gerncia de aplicaes.
   */
  private IApplicationManagerContext appManagerContext;

  /**
   * Construtor.
   * 
   * @param csdkInterface interface padro para o ambiente CSDK.
   */
  public AbstractCSDKApplication(ICSDKEnvironment csdkInterface) {
    super(csdkInterface);
    this.projectContext = getContext(IProjectContext.class);
    this.localFileSystemContext = getContext(ILocalFileSystemContext.class);
    this.commandContext = getContext(ICommandContext.class);
    this.appManagerContext = getContext(IApplicationManagerContext.class);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final IFile browseProjectFileInOpenMode(String[] fileTypes,
    boolean allowAll, FileSelectionType selectionType, Window frame)
    throws ProjectException {
    checkProjectContext();
    return this.projectContext.browseProjectFileInOpenMode(fileTypes, allowAll,
      selectionType, frame);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile[] browseMultipleProjectFilesInOpenMode(String[] fileTypes,
    boolean allowAll, FileSelectionType selectionType, Window frame)
    throws ProjectException {
    checkProjectContext();
    return this.projectContext.browseMultipleProjectFilesInOpenMode(fileTypes,
      allowAll, selectionType, frame);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final IFile browseProjectFileInSaveMode(String[] fileTypes,
    boolean allowAll, String defaultName, Window frame) throws ProjectException {
    checkProjectContext();
    return this.projectContext.browseProjectFileInSaveMode(fileTypes, allowAll,
      defaultName, frame);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final IFile getProjectFile(String[] path, Window window)
    throws ProjectException {
    checkProjectContext();
    return this.projectContext.getProjectFile(path, window);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final IFile createProjectFile(String[] path, String name,
    String fileType, Window window) throws ProjectException {
    checkProjectContext();
    return this.projectContext.createProjectFile(path, name, fileType, window);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final String getCurrentProjectId() {
    checkProjectContext();
    return this.projectContext.getCurrentProjectId();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void addProjectObserver(IProjectObserver observer) {
    checkProjectContext();
    this.projectContext.addProjectObserver(observer);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removeProjectObserver(IProjectObserver observer) {
    checkProjectContext();
    this.projectContext.removeProjectObserver(observer);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean removeProjectFile(String[] path, Window window)
    throws ProjectException {
    checkProjectContext();
    return this.projectContext.removeProjectFile(path, window);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final IFile browseLocalFileInOpenMode(String[] fileTypes,
    boolean allowAll, IFile currentDir, FileSelectionType fileSelectionType,
    Window frame) throws LocalFileSystemException {
    checkFileSystemContext();
    return this.localFileSystemContext.browseLocalFileInOpenMode(fileTypes,
      allowAll, currentDir, fileSelectionType, frame);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile[] browseMultipleLocalFilesInOpenMode(String[] fileTypes,
    boolean allowAll, IFile currentDir, FileSelectionType fileSelectionType,
    Window frame) throws LocalFileSystemException {
    checkFileSystemContext();
    return this.localFileSystemContext.browseMultipleLocalFilesInOpenMode(
      fileTypes, allowAll, currentDir, fileSelectionType, frame);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final IFile browseLocalFileInSaveMode(String[] fileTypes,
    boolean allowAll, String defaultName, IFile currentDir, Window frame)
    throws LocalFileSystemException {
    checkFileSystemContext();
    return this.localFileSystemContext.browseLocalFileInSaveMode(fileTypes,
      allowAll, defaultName, currentDir, frame);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile getLocalFile(String[] path) throws LocalFileSystemException {
    checkFileSystemContext();
    return this.localFileSystemContext.getLocalFile(path);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final IFile createLocalFile(String[] path, String name, boolean isDir)
    throws LocalFileSystemException {
    checkFileSystemContext();
    return this.localFileSystemContext.createLocalFile(path, name, isDir);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean removeLocalFile(String[] path) throws LocalFileSystemException {
    checkFileSystemContext();
    return this.localFileSystemContext.removeLocalFile(path);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final String executeAlgorithm(IAlgorithmTemplate info,
    String description, String sgaServerName, Window owner,
    ICommandObserver observer) throws CommandException {
    checkCommandContext();
    return this.commandContext.executeAlgorithm(info, description,
      sgaServerName, owner, observer);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final String executeFlow(IFlowTemplate info, String description,
    String sgaServerName, Window owner, ICommandObserver observer)
    throws CommandException {
    checkCommandContext();
    return this.commandContext.executeFlow(info, description, sgaServerName,
      owner, observer);
  }

  /**
   * Indica se o contexto de projetos est disponvel.
   *
   * @return {@code true} se o contexto estiver disponvel ou
   *         {@code false} caso contrrio.
   */
  public boolean isProjectContextAvailable() {
    return this.projectContext != null && this.projectContext.isActive();
  }

  /**
   * Verifica se existe projeto corrente.
   *
   * @return {@code true} caso exista um projeto aberto ou
   *         {@code false} caso contrrio.
   */
  public boolean isProjectOpen() {
    return isProjectContextAvailable() && getCurrentProjectId() != null;
  }

  /**
   * Indica se o contexto de sistema de arquivo est disponvel.
   *
   * @return {@code true} se o contexto estiver disponvel ou
   *         {@code false} caso contrrio.
   */
  public boolean isFileSystemContextAvailable() {
    return this.localFileSystemContext != null
      && this.localFileSystemContext.isActive();
  }

  /**
   * Indica se o contexto de execuo de comandos est disponvel.
   *
   * @return {@code true} se o contexto estiver disponvel ou
   *         {@code false} caso contrrio.
   */
  public boolean isCommandContextAvailable() {
    return this.commandContext != null && this.commandContext.isActive();
  }

  /**
   * Indica se o contexto de gerncia de aplicaes est disponvel.
   *
   * @return {@code true} se o contexto estiver disponvel ou
   * {@code false} caso contrrio.
   */
  public boolean isApplicationManagerContextAvailable() {
    return this.appManagerContext != null && this.appManagerContext.isActive();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String runApplication(String id) throws ApplicationException {
    checkApplicationManagerContext();
    return this.appManagerContext.runApplication(id);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isApplicationRunning(String id) {
    checkApplicationManagerContext();
    return this.appManagerContext.isApplicationRunning(id);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void killApplication(String instanceId) throws ApplicationException {
    checkApplicationManagerContext();
    this.appManagerContext.killApplication(instanceId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void sendAsyncMessage(String instanceId,
    IMessage message) throws ApplicationException {
    checkApplicationManagerContext();
    this.appManagerContext.sendAsyncMessage(instanceId, message);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object sendSyncMessage(String instanceId,
    IMessage message) throws ApplicationException {
    checkApplicationManagerContext();
    return this.appManagerContext.sendSyncMessage(instanceId, message);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void broadcastAsyncMessage(
    IMessage message) throws ApplicationException {
    checkApplicationManagerContext();
    this.appManagerContext.broadcastAsyncMessage(message);
  }

  /**
   * Verifica se o contexto de projetos est disponvel, lanando uma exceo
   * {@link ContextNotAvailableException} caso negativo.
   */
  private void checkProjectContext() {
    if (!isProjectContextAvailable()) {
      throw new ContextNotAvailableException(IProjectContext.class);
    }
  }

  /**
   * Verifica se o contexto de sistema de arquivos est disponvel, lanando uma
   * exceo {@link ContextNotAvailableException} caso negativo.
   */
  private void checkFileSystemContext() {
    if (!isFileSystemContextAvailable()) {
      throw new ContextNotAvailableException(ILocalFileSystemContext.class);
    }
  }

  /**
   * Verifica se o contexto de execuo de comandos est disponvel, lanando
   * uma exceo {@link ContextNotAvailableException} caso negativo.
   */
  private void checkCommandContext() {
    if (!isCommandContextAvailable()) {
      throw new ContextNotAvailableException(ICommandContext.class);
    }
  }

  /**
   * Verifica se o contexto de gerncia de aplicaes est disponvel,
   * lanando uma exceo {@link ContextNotAvailableException} caso negativo.
   */
  private void checkApplicationManagerContext() {
    if (!isApplicationManagerContextAvailable()) {
      throw new ContextNotAvailableException(IApplicationManagerContext.class);
    }
  }


  /**
   * {@inheritDoc}
   */
  @Override
  protected void disposeContexts() {
    super.disposeContexts();
    this.commandContext = null;
    this.localFileSystemContext = null;
    this.projectContext = null;
  }
}
